<?php
/* * *******************************************************************************
 * The content of this file is subject to the MultiWarehouses4You license.
 * ("License"); You may not use this file except in compliance with the License
 * The Initial Developer of the Original Code is IT-Solutions4You s.r.o.
 * Portions created by IT-Solutions4You s.r.o. are Copyright(C) IT-Solutions4You s.r.o.
 * All Rights Reserved.
 * ****************************************************************************** */

class ITS4YouWarehouses_SeqStatusActionHandler_Helper
{
    const RC_STATUS = 'receiptcardstatus';
    const DN_STATUS = 'deliverynotestatus';
    const IC_STATUS = 'issuecardstatus';
    const WT_STATUS = 'wtstatus';
    /** @var Vtiger_Request */
    protected static $request;
    /** @var Vtiger_Viewer */
    protected static $viewer;
    /** @var string */
    protected static $module;
    /** @var string */
    protected static $sourceModule;
    /** @var int */
    protected static $sourceRecord;

    /**
     * @param Vtiger_Request $request
     * @param Vtiger_Viewer $viewer
     * @throws Exception
     */
    public static function handle(Vtiger_Request $request, Vtiger_Viewer $viewer)
    {
        self::$request = $request;
        self::$sourceModule = $request->get('sourceModule');
        self::$sourceRecord = $request->get('sourceRecord');
        self::$module = 'ITS4YouWHDeliveryNotes';
        self::$viewer = $viewer;

        $allowedModules = array('ITS4YouReceiptcards', 'ITS4YouWHDeliveryNotes', 'ITS4YouIssuecards', 'ITS4YouWarehouseTransfers');
        $mode = $request->get('mode');

        if (!in_array(self::$sourceModule, $allowedModules)) {
            throw new Exception('MultiWarehouses4You :: Unsupported module for class SeqStatusActionHandler.');
        }

        switch ($mode) {
            case 'getStatus':
                self::handleGetStatus();
                break;

            case 'changeStatus':
                self::handleChangeStatus();
                break;

            case 'changeStatusCancel':
                self::handleChangeStatusCancel();
                break;

            case 'changeStatusDeliver':
                self::handleChangeStatusDeliver();
                break;

            default:
                throw new Exception('Warehouses :: Unsupported action mode SeqStatusActionHandler.');
        }
    }

    /**
     * @throws Exception
     */
    protected static function handleGetStatus()
    {
        $statusFieldName = self::getStatusField();
        $recordModel = Vtiger_Record_Model::getInstanceById(self::$sourceRecord, self::$sourceModule);
        $status = $recordModel->get($statusFieldName);

        $response = new Vtiger_Response();
        $response->setResult(array('sourceRecord' => self::$sourceRecord, 'status' => $status));
        $response->emit();
    }

    /**
     * @return string
     * @throws Exception
     */
    protected static function getStatusField()
    {
        switch (self::$sourceModule) {
            case 'ITS4YouWHDeliveryNotes':
                $statusFieldName = self::DN_STATUS;
                break;

            case 'ITS4YouReceiptcards':
                $statusFieldName = self::RC_STATUS;
                break;

            case 'ITS4YouIssuecards':
                $statusFieldName = self::IC_STATUS;
                break;

            case 'ITS4YouWarehouseTransfers':
                $statusFieldName = self::WT_STATUS;
                break;

            default:
                throw new Exception('MultiWarehouses4You :: Unsupported module for method handleGetStatus.');
        }

        return $statusFieldName;
    }

    /**
     *
     * @throws AppException
     */
    protected static function handleChangeStatus()
    {
        $newStatus = self::getNewStatus();

        if (!empty($newStatus)) {
            /* @var $moduleModel Vtiger_Module_Model */
            $moduleModel = Vtiger_Module_Model::getInstance(self::$sourceModule);
            $loadUrl = $moduleModel->getDetailViewUrl(self::$sourceRecord);
            $urlHashTag = self::getNewStatusMessage($newStatus);
            //for info alert - see JS class ITS4YouWHDeliveryNotes_Detail_Js in ITS4YouWHDeliveryNotes Detail.js file
            #displayNotifyMsg
            $loadUrl = (!empty($urlHashTag) ? $loadUrl . '#' . $urlHashTag : $loadUrl);
        } else {
            //this branch is only for DN
            $loadUrl = sprintf('index.php?module=%s&view=CustomInvoiceEdit&sourceModule=%s&sourceRecord=%s', self::$sourceModule, self::$sourceModule, self::$sourceRecord);
        }

        header("Location: $loadUrl");
    }

    /**
     * @return string
     * @throws AppException
     */
    protected static function getNewStatus()
    {
        switch (self::$sourceModule) {
            case 'ITS4YouWHDeliveryNotes':
                $newStatus = self::doChangeStatusDN();
                break;

            case 'ITS4YouReceiptcards':
                $newStatus = self::doChangeStatusRC();
                break;

            case 'ITS4YouIssuecards':
                $newStatus = self::doChangeStatusIC();
                break;

            case 'ITS4YouWarehouseTransfers':
                $newStatus = self::doChangeStatusWT();
                break;

            default:
                $newStatus = '';
        }

        return $newStatus;
    }

    /**
     * @return string
     * @throws AppException
     * @throws Exception
     */
    protected static function doChangeStatusDN()
    {
        $statusMode = self::$request->get('statusMode');
        $recordModel = Vtiger_Record_Model::getInstanceById(self::$sourceRecord);
        //check whether there is already related invoice despite the status
        $relatedInvoiceId = intval($recordModel->get('invoice_id'));
        $newStatus = '';

        if (ITS4YouWarehouses_DNStatusPool_Sequential::ACTION_MODE_INVOICE !== $statusMode || $relatedInvoiceId) {
            $recordModel->set('id', self::$sourceRecord);
            $recordModel->set('mode', 'edit');

            switch ($statusMode) {
                case ITS4YouWarehouses_DNStatusPool_Sequential::ACTION_MODE_CANCEL:
                    $newStatus = 'Canceled';
                    $reason = sprintf("%s \r\n%s \r\n%s ", vtranslate('LBL_CANCEL_REASON_COMMENT', 'ITS4YouWHDeliveryNotes'), self::$request->get('reason'), $recordModel->get('description'));
                    $recordModel->set('description', $reason);
                    $recordModel->save();
                    //when DN is Canceled, then all related RCs (as a result of Products Return) need to be Canceled as well
                    $returnProductsRCs = ITS4YouWarehouses_DNUtil_ActionBlock::getRelatedReceiptcards(self::$sourceRecord, array("Created", "Delivered"));

                    foreach ($returnProductsRCs as $receiptcardId => $receiptcardNo) {
                        $rcRecordModel = Vtiger_Record_Model::getInstanceById($receiptcardId);
                        $rcRecordModel->set('id', $receiptcardId);
                        $rcRecordModel->set('mode', 'edit');
                        $rcRecordModel->set(self::RC_STATUS, 'Canceled');
                        $rcRecordModel->set('description', $reason);
                        $rcRecordModel->save();
                    }
                    break;

                case ITS4YouWarehouses_DNStatusPool_Sequential::ACTION_MODE_DELIVER:
                    $newStatus = !empty($relatedInvoiceId) ? 'Invoiced' : 'Delivered';
                    //keep original REQUEST
                    $realRequest = $_REQUEST;
                    unset($_REQUEST);
                    //In order to avoid incosistencies when user changes quantity via this action, since there are IC at background related
                    $recordModel->save();
                    $_REQUEST = $realRequest;
                    break;

                case ITS4YouWarehouses_DNStatusPool_Sequential::ACTION_MODE_INVOICE:
                    $newStatus = 'Invoiced';
                    break;

                case ITS4YouWarehouses_DNStatusPool_Sequential::ACTION_MODE_READY_TO_DELIVER:
                    $newStatus = ITS4YouWarehouses_Base_ActionBlock::factory(self::$sourceModule, self::$sourceRecord)->checkInitialStatus();
                    break;
            }

            if (!empty($newStatus)) {
                ITS4YouWarehouses_DNUtil_ActionBlock::setNewParentStatus(self::$sourceRecord, $newStatus);
            }
        }

        return $newStatus;
    }

    /**
     * @return string
     * @throws AppException
     * @throws Exception
     */
    protected static function doChangeStatusRC()
    {
        $statusFieldName = self::RC_STATUS;
        $statusMode = self::$request->get('statusMode');
        /** @var ITS4YouReceiptcards_Record_Model $recordModel */
        $recordModel = Vtiger_Record_Model::getInstanceById(self::$sourceRecord, self::$sourceModule);
        $newStatus = '';

        $recordModel->set('id', self::$sourceRecord);
        $recordModel->set('mode', 'edit');

        switch ($statusMode) {
            case ITS4YouWarehouses_RCStatusPool_Sequential::ACTION_MODE_CANCEL:
                $newStatus = 'Canceled';
                $reason = sprintf("%s \r\n%s \r\n%s ", vtranslate('LBL_CANCEL_REASON_COMMENT', 'ITS4YouWHDeliveryNotes'), self::$request->get('reason'), $recordModel->get('description'));
                $recordModel->set('description', $reason);
                $recordModel->set($statusFieldName, $newStatus);
                $recordModel->save();

                //when RC is Canceled, then all related ICs (as a result of Products Return) need to be Canceled as well
                //Cancelation of related WTs (as a result of Products Return in case of RC is part of WT process)
                $actionBlock = ITS4YouWarehouses_Base_ActionBlock::factory(self::$sourceModule, self::$sourceRecord);
                $returnProductsICs = $actionBlock->getReturnProductsIssuecards();

                foreach ($returnProductsICs as $issuecardId => $issuecardNo) {
                    $ICRecordModel = Vtiger_Record_Model::getInstanceById($issuecardId);
                    $ICRecordModel->set('id', $issuecardId);
                    $ICRecordModel->set('mode', 'edit');
                    $ICRecordModel->set(self::IC_STATUS, 'Canceled');
                    $ICRecordModel->set('description', $reason);
                    $ICRecordModel->save();
                }
                break;

            case ITS4YouWarehouses_RCStatusPool_Sequential::ACTION_MODE_DELIVER:
                $newStatus = 'Delivered';
                //keep original REQUEST
                $realRequest = $_REQUEST;
                unset($_REQUEST);
                $toDeliverQuantities = ITS4YouWarehouses_Products_Helper::getItemsQtyFromRequest(self::$request);
                ITS4YouWarehouses_InventoryUtils_Helper::prepareRequestForExistingProducts($recordModel->getEntity(), $toDeliverQuantities);

                $recordModel->set('adoption_date', getNewDisplayDate());
                $recordModel->set($statusFieldName, $newStatus);
                $recordModel->save();

                $_REQUEST = $realRequest;
                $vendorId = $recordModel->get('vendor_id');

                self::updateTransfersStatus(self::$sourceRecord, $vendorId);

                break;
        }

        return $newStatus;
    }

    public static function updateTransfersStatus($record, $vendorId)
    {
        $module = getSalesEntityType($record);
        $vendorModule = getSalesEntityType($vendorId);
        $oldAction = $_REQUEST['action'];
        $_REQUEST['action'] = 'ITS4YouWarehouseTransfersAjax';

        if ('ITS4YouReceiptcards' === $module && 'ITS4YouWarehouseTransfers' === $vendorModule) {
            $vendorRecord = Vtiger_Record_Model::getInstanceById($vendorId, $vendorModule);
            $vendorRecord->set('mode', 'edit');
            $vendorRecord->set(self::WT_STATUS, 'Delivered');
            $vendorRecord->save();
        }

        $_REQUEST['action'] = $oldAction;
    }

    /**
     * @throws Exception
     */
    public static function updateIssueCardsStatus($record, $vendorId)
    {
        $module = getSalesEntityType($record);
        $vendorModule = getSalesEntityType($vendorId);
        $oldAction = $_REQUEST['action'];
        $_REQUEST['action'] = 'ITS4YouIssuecardsAjax';

        if ('ITS4YouReceiptcards' === $module && 'ITS4YouIssuecards' === $vendorModule) {
            if (!self::hasReturnableProducts($vendorModule, $vendorId)) {
                $vendorRecord = Vtiger_Record_Model::getInstanceById($vendorId, $vendorModule);
                $vendorRecord->set('mode', 'edit');
                $vendorRecord->set(self::IC_STATUS, 'Delivered');
                $vendorRecord->save();
            }
        }

        $_REQUEST['action'] = $oldAction;
    }

    /**
     * @param string $module
     * @param int $record
     * @return array
     */
    public static function getIssueProductQuantities($module, $record)
    {
        $products = array();
        $issueRecord = Vtiger_Record_Model::getInstanceById($record, $module);
        $issueProducts = $issueRecord->getProducts();

        foreach ($issueProducts as $key => $product) {
            $products[$product['hdnProductId' . $key]] += $product['qty' . $key];
        }

        return $products;
    }

    /**
     * @param string $module
     * @param int $record
     * @return array
     * @throws Exception
     */
    public static function getDeliveredReceiptProductQuantities($module, $record)
    {
        $adb = PearDatabase::getInstance();
        $products = array();
        $sql = 'SELECT vtiger_crmentityrel.relcrmid FROM vtiger_crmentityrel 
            LEFT JOIN vtiger_crmentity ON vtiger_crmentity.crmid=vtiger_crmentityrel.relcrmid
            LEFT JOIN its4you_receiptcards ON its4you_receiptcards.receiptcardid=vtiger_crmentity.crmid 
            WHERE vtiger_crmentity.deleted=0 AND vtiger_crmentityrel.crmid=? AND vtiger_crmentityrel.module=? AND vtiger_crmentityrel.relmodule=? AND its4you_receiptcards.receiptcardstatus=?';
        $params = array($record, $module, 'ITS4YouReceiptcards', 'Delivered');
        $result = $adb->pquery($sql, $params);

        /** @var ITS4YouReceiptcards_Record_Model $receiptRecord */

        while ($row = $adb->fetchByAssoc($result)) {
            $receiptRecord = Vtiger_Record_Model::getInstanceById($row['relcrmid'], 'ITS4YouReceiptcards');
            $receiptProducts = $receiptRecord->getProducts();

            foreach ($receiptProducts as $key => $product) {
                $products[$product['hdnProductId' . $key]] += $product['qty' . $key];
            }
        }

        return $products;
    }

    /**
     * @param string $module
     * @param int $record
     * @return bool
     * @throws Exception
     */
    public static function hasReturnableProducts($module, $record)
    {
        $products = self::getIssueProductQuantities($module, $record);
        $receiptProducts = self::getDeliveredReceiptProductQuantities($module, $record);

        foreach ($products as $key => $productQty) {
            if (0 < ($productQty - $receiptProducts[$key])) {
                return true;
            }
        }

        return false;
    }

    /**
     * @throws Exception
     */

    /**
     * @return string
     * @throws AppException
     * @throws Exception
     */
    protected static function doChangeStatusIC()
    {
        $statusFieldName = self::IC_STATUS;
        $statusMode = self::$request->get('statusMode');
        $newStatus = '';
        $recordModel = Vtiger_Record_Model::getInstanceById(self::$sourceRecord, self::$sourceModule);
        $recordModel->set('id', self::$sourceRecord);
        $recordModel->set('mode', 'edit');

        switch ($statusMode) {
            case ITS4YouWarehouses_ICStatusPool_Sequential::ACTION_MODE_CANCEL:
                $newStatus = 'Canceled';
                $reason = sprintf("%s \r\n%s \r\n%s ", vtranslate('LBL_CANCEL_REASON_COMMENT', 'ITS4YouWHDeliveryNotes'), self::$request->get('reason'), $recordModel->get('description'));
                $recordModel->set('description', $reason);
                $recordModel->set($statusFieldName, $newStatus);
                $recordModel->save();

                //when IC is Canceled, then all related RCs (as a result of Products Return) need to be Canceled as well
                $actionBlock = ITS4YouWarehouses_Base_ActionBlock::factory(self::$sourceModule, self::$sourceRecord);
                $returnProductsRCs = $actionBlock->getReturnProductsReceiptcards();
                foreach ($returnProductsRCs as $receiptcardId => $receiptcardNo) {
                    $rcRecordModel = Vtiger_Record_Model::getInstanceById($receiptcardId);
                    $rcRecordModel->set('id', $receiptcardId);
                    $rcRecordModel->set('mode', 'edit');
                    $rcRecordModel->set(self::RC_STATUS, 'Canceled');
                    $rcRecordModel->set('description', $reason);
                    $rcRecordModel->save();
                }
                break;
        }
        return $newStatus;
    }

    /**
     * @return string
     * @throws AppException
     */
    protected static function doChangeStatusWT()
    {
        $statusFieldName = self::WT_STATUS;
        $statusMode = self::$request->get('statusMode');
        $recordModel = Vtiger_Record_Model::getInstanceById(self::$sourceRecord, self::$sourceModule);
        $newStatus = '';
        $recordModel->set('id', self::$sourceRecord);
        $recordModel->set('mode', 'edit');

        switch ($statusMode) {
            case ITS4YouWarehouses_WTStatusPool_Sequential::ACTION_MODE_CANCEL:
                $newStatus = 'Canceled';
                $reason = sprintf("%s \r\n%s \r\n%s ", vtranslate('LBL_CANCEL_REASON_COMMENT', 'ITS4YouWHDeliveryNotes'), self::$request->get('reason'), $recordModel->get('description'));
                $recordModel->set('description', $reason);
                $recordModel->set($statusFieldName, $newStatus);
                $recordModel->save();
                break;
            case ITS4YouWarehouses_WTStatusPool_Sequential::ACTION_MODE_DELIVER:
                $newStatus = 'Delivered';
                $recordModel->set($statusFieldName, $newStatus);
                $recordModel->save();
                break;
        }

        return $newStatus;
    }

    protected static function getNewStatusMessage($status)
    {
        switch ($status) {
            case 'Canceled':                // both DN and RC and IC and WT
                $urlHashTag = 'ALERT_STATUS_CHANGE_CANCELED';
                break;

            case 'Delivered':               // both DN and RC
                $urlHashTag = 'ALERT_STATUS_CHANGE_DELIVERED';
                break;

            case 'Invoiced':                //only for DN
                $urlHashTag = 'ALERT_STATUS_CHANGE_INVOICED';
                break;

            case 'Waiting Availability':    //only for DN
                $urlHashTag = 'ALERT_STATUS_CHANGE_AWAITING';
                break;

            case 'Ready to Deliver':        //only for DN
                $urlHashTag = 'ALERT_STATUS_CHANGE_READY_TO_DELIVER';
                break;

            case 'Created':                 //only for DN
                $urlHashTag = 'ALERT_STATUS_CHANGE_CREATED';
                break;

            default:
                $urlHashTag = '';
        }

        return $urlHashTag;
    }

    /**
     * This function is used by both modules DN and RC
     * @throws Exception
     */
    protected static function handleChangeStatusCancel()
    {
        self::$viewer->assign('MODULE', self::$module);
        self::$viewer->assign('SOURCE_MODULE', self::$sourceModule);
        self::$viewer->assign('RECORD', self::$sourceRecord);
        self::$viewer->assign('STATUS_MODE', ITS4YouWarehouses_DNStatusPool_Sequential::ACTION_MODE_CANCEL);   //also ITS4YouWarehouses_RCStatusPool_Sequential::ACTION_MODE_DELIVER

        self::$viewer->view('ActionBlockChangeStatusCancel.tpl', self::$module);
    }

    /**
     * This function is used by both modules DN and RC
     * @throws Exception
     */
    protected static function handleChangeStatusDeliver()
    {
        /* @var $actionBlock ITS4YouWarehouses_IStatusActionBlock_Sequential */
        $actionBlock = ITS4YouWarehouses_Base_ActionBlock::factory(self::$sourceModule, self::$sourceRecord);
        $viewModel = $actionBlock->getChangeStatusDeliverViewModel();

        self::$viewer->assign('MODULE', self::$module);
        self::$viewer->assign('SOURCE_MODULE', self::$sourceModule);
        self::$viewer->assign('RECORD', self::$sourceRecord);
        self::$viewer->assign('STATUS_MODE', ITS4YouWarehouses_DNStatusPool_Sequential::ACTION_MODE_DELIVER);  //also ITS4YouWarehouses_RCStatusPool_Sequential::ACTION_MODE_DELIVER
        self::$viewer->assign('VIEWMODEL', $viewModel);

        self::$viewer->view('ActionBlockChangeStatusDeliver.tpl', self::$module);
    }

    /**
     * @param $moduleName
     * @param $status
     * @return bool
     * @throws Exception
     */
    public static function isStatusDisabledForModule($moduleName, $status)
    {
        return isset(self::getDisabledStatusesForDelete($moduleName)[$status]);
    }

    /**
     * @param string $_moduleName
     * @return array
     * @throws Exception
     */
    public static function getDisabledStatusesForDelete($_moduleName)
    {
        switch ($_moduleName) {
            case 'ITS4YouWHDeliveryNotes':
                return array(
                    'Delivered' => vtranslate('LBL_DELETE_DELIVERED_FORBIDDEN', $_moduleName),
                    'Invoiced' => vtranslate('LBL_DELETE_INVOICED_FORBIDDEN', $_moduleName),
                    'Canceled' => vtranslate('LBL_DELETE_CANCELED_FORBIDDEN', $_moduleName),
                );

            case 'ITS4YouReceiptcards':
                return array(
                    'Delivered' => vtranslate('LBL_DELETE_DELIVERED_FORBIDDEN', $_moduleName),
                    'Canceled' => vtranslate('LBL_DELETE_CANCELED_FORBIDDEN', $_moduleName),
                );

            case 'ITS4YouIssuecards':
            case 'ITS4YouWarehouseTransfers':
                return array(
                    'Transferred from Warehouse' => vtranslate('LBL_TRANSFERRED_DELIVERED_FORBIDDEN', $_moduleName),
                    'Delivered' => vtranslate('LBL_DELETE_DELIVERED_FORBIDDEN', $_moduleName),
                    'Canceled' => vtranslate('LBL_DELETE_CANCELED_FORBIDDEN', $_moduleName),
                );

            default:
                throw new Exception("MultiWarehouses4You :: Unsupported module for method getDisabledStatuses.");
        }
    }

    /**
     * @param string $moduleName
     * @param string $status
     * @throws AppException
     * @throws Exception
     */
    public static function checkStatusPermission($moduleName, $status)
    {
        $disabledStatuses = self::getDisabledStatusesForDelete($moduleName);

        if (isset($disabledStatuses[$status])) {
            throw new AppException($disabledStatuses[$status]);
        }
    }
}
