<?php
/** *******************************************************************************
 * The content of this file is subject to the MultiWarehouses4You license.
 * ("License"); You may not use this file except in compliance with the License
 * The Initial Developer of the Original Code is IT-Solutions4You s.r.o.
 * Portions created by IT-Solutions4You s.r.o. are Copyright(C) IT-Solutions4You s.r.o.
 * All Rights Reserved.
 * ****************************************************************************** */

class ITS4YouWarehouses_InventoryLineItem_Helper
{

    private $itemId;
    private $productId;
    private $quantity;
    private $price;
    private $comment;
    private $parentId;
    private $sequenceNo;
    private $name;
    private $crossInventoryLineItems = array();
    private $subProductIds;
    private $subProductNames;
    private $discountPercent;
    private $discountAmount;

    /**
     * ITS4YouWarehouses_InventoryLineItem_Helper constructor.
     * @param $_itemId
     * @param $_productId
     * @param $_quantity
     * @param int $_price
     * @param string $_comment
     * @param int $_parentId
     * @param int $_sequenceNo
     */
    public function __construct($_itemId, $_productId, $_quantity, $_price = -1, $_comment = "", $_parentId = -1, $_sequenceNo = -1)
    {
        $this->itemId = $_itemId;
        $this->productId = $_productId;
        $this->quantity = $_quantity;
        $this->price = $_price;
        $this->comment = $_comment;
        $this->parentId = $_parentId;
        $this->sequenceNo = $_sequenceNo;

        $this->subProductIds = null;
        $this->subProductNames = null;
        $this->discountPercent = null;
        $this->discountAmount = null;
    }

    /**
     * Return instance of ITS4YouWarehouses_InventoryLineItem_Helper from DB
     * @param int $_itemId
     * @return ITS4YouWarehouses_InventoryLineItem_Helper
     * @global PearDatabase $adb
     */
    public static function getLineItemFromDB($_itemId)
    {
        global $adb;

        $sql = "SELECT vtiger_inventoryproductrel.*,
                    CASE WHEN vtiger_products.productid != '' THEN vtiger_products.productname ELSE vtiger_service.servicename END AS productname
                FROM vtiger_inventoryproductrel 
                LEFT JOIN vtiger_products
                    ON vtiger_products.productid=vtiger_inventoryproductrel.productid
                LEFT JOIN vtiger_service
                    ON vtiger_service.serviceid=vtiger_inventoryproductrel.productid                
                WHERE lineitem_id =?";
        $result = $adb->pquery($sql, array($_itemId));

        /* @var $lineItem ITS4YouWarehouses_InventoryLineItem_Helper */
        $lineItem = null;
        if ($adb->num_rows($result) > 0) {
            $row = $adb->fetchByAssoc($result);
            $lineItem = new ITS4YouWarehouses_InventoryLineItem_Helper($row["lineitem_id"], $row["productid"], $row["quantity"], $row["listprice"], $row["comment"], $row["id"], $row["sequence_no"]);
            $lineItem->setName($row["productname"]);
            $lineItem->setDiscountPercent($row["discount_percent"]);
            $lineItem->setDiscountAmount($row["discount_amount"]);
        }
        return $lineItem;
    }

    /**
     * Returns instance of ITS4YouWarehouses_InventoryLineItem_Helper based on parent entity ID and sequence
     * number of the line item in Inventory table
     * @param int $_parentId
     * @param int $_sequenceNo
     * @return ITS4YouWarehouses_InventoryLineItem_Helper
     */
    public static function getLineItemFromDBBySequence($_parentId, $_sequenceNo)
    {
        global $adb;

        $sql = "SELECT vtiger_inventoryproductrel.*,
                    CASE WHEN vtiger_products.productid != '' THEN vtiger_products.productname ELSE vtiger_service.servicename END AS productname
                FROM vtiger_inventoryproductrel 
                LEFT JOIN vtiger_products
                    ON vtiger_products.productid=vtiger_inventoryproductrel.productid
                LEFT JOIN vtiger_service
                    ON vtiger_service.serviceid=vtiger_inventoryproductrel.productid                
                WHERE id = ? AND sequence_no = ?";
        $result = $adb->pquery($sql, array($_parentId, $_sequenceNo));

        /* @var $lineItem ITS4YouWarehouses_InventoryLineItem_Helper */
        $lineItem = null;
        if ($adb->num_rows($result) > 0) {
            $row = $adb->fetchByAssoc($result);
            $lineItem = new ITS4YouWarehouses_InventoryLineItem_Helper($row["lineitem_id"], $row["productid"], $row["quantity"], $row["listprice"], $row["comment"], $row["id"], $row["sequence_no"]);
            $lineItem->setName($row["productname"]);
            $lineItem->setDiscountPercent($row["discount_percent"]);
            $lineItem->setDiscountAmount($row["discount_amount"]);
        }
        return $lineItem;
    }

    /**
     *
     * @param int $_parentId
     * @param bool $_unique
     * @return array Collection of instances of ITS4YouWarehouses_InventoryLineItem_Helper
     * @global PearDatabase $adb
     */
    public static function getParentLineItems($_parentId, $_unique = false)
    {
        global $adb;

        $servicesEnabledPreference = ITS4YouWHDeliveryNotes_Preferences_Model::getInstance(ITS4YouWHDeliveryNotes_Preferences_Model::PREFNAME_SERVICES_ENABLED);
        $servicesEnabled = ($servicesEnabledPreference->getValue() != "0");

        $lineItems = array();
        $sql = "SELECT                     
                    vtiger_inventoryproductrel.lineitem_id,
                    vtiger_inventoryproductrel.productid,
                    vtiger_inventoryproductrel.quantity,
                    vtiger_inventoryproductrel.listprice,                  
                    vtiger_inventoryproductrel.comment,
                    vtiger_inventoryproductrel.id,
                    vtiger_inventoryproductrel.sequence_no,
                    CASE WHEN vtiger_products.productid != '' THEN vtiger_products.productname ELSE vtiger_service.servicename END AS productname,
                    vtiger_inventoryproductrel.discount_percent,
                    vtiger_inventoryproductrel.discount_amount,
                    CASE WHEN vtiger_products.productid != '' THEN 'Products' ELSE 'Services' END AS `entity_type`
                 FROM vtiger_inventoryproductrel
                 LEFT JOIN vtiger_products
                    ON vtiger_products.productid=vtiger_inventoryproductrel.productid
                 LEFT JOIN vtiger_service
                    ON vtiger_service.serviceid=vtiger_inventoryproductrel.productid
                 WHERE id=?
                 ORDER BY sequence_no";
        $params = array($_parentId);
        $result = $adb->pquery($sql, $params);
        $productIds = array();
        while ($row = $adb->fetchByAssoc($result)) {
            if (!$servicesEnabled && $row["entity_type"] == "Services") {
                continue;
            }
            $lineItem = new ITS4YouWarehouses_InventoryLineItem_Helper($row["lineitem_id"], $row["productid"], $row["quantity"], $row["listprice"], $row["comment"], $row["id"], $row["sequence_no"]);
            $lineItem->setName($row["productname"]);
            $lineItem->setDiscountPercent($row["discount_percent"]);
            $lineItem->setDiscountAmount($row["discount_amount"]);
            if ($_unique === false || !isset($productIds[$row["productid"]])) {
                $lineItems[$lineItem->getItemId()] = $lineItem;
            }
            $productIds[$row["productid"]] = $row["productid"];
        }
        return $lineItems;
    }

    /**
     * @return string
     */
    public function getItemId()
    {
        if ($this->itemId <= 0) {
            return "";
        } else {
            return $this->itemId;
        }
    }

    /**
     * @return mixed
     */
    public function getProductId()
    {
        return $this->productId;
    }

    /**
     * @return mixed
     */
    public function getQuantity()
    {
        return $this->quantity;
    }

    /**
     * @param $_quantity
     */
    public function setQuantity($_quantity)
    {
        $this->quantity = $_quantity;
    }

    /**
     * @return int
     */
    public function getPrice()
    {
        return $this->price;
    }

    /**
     * @param $price
     */
    public function setPrice($price)
    {
        $this->price = $price;
    }

    /**
     * @return string
     */
    public function getComment()
    {
        return $this->comment;
    }

    /**
     * @return int
     */
    public function getParentId()
    {
        return $this->parentId;
    }

    /**
     * @return int
     */
    public function getSequenceNo()
    {
        return $this->sequenceNo;
    }

    /**
     * @return mixed
     */
    public function getName()
    {
        return $this->name;
    }

    /**
     * @param $name
     */
    public function setName($name)
    {
        $this->name = $name;
    }

    /**
     * @param bool $asString
     * @return string|null
     */
    public function getSubProductIds($asString = false)
    {
        $this->initSubproducts();
        if (!$asString) {
            return $this->subProductIds;
        } else {
            return implode(":", $this->subProductIds);
        }
    }

    /**
     *
     */
    private function initSubproducts()
    {
        if ($this->subProductIds === null) {
            $db = PearDatabase::getInstance();
            $sql = "SELECT  vtiger_products.productid, 
                            vtiger_products.productname 
                    FROM    vtiger_products
                    INNER JOIN vtiger_inventorysubproductrel 
                            ON vtiger_inventorysubproductrel.productid = vtiger_products.productid
                    WHERE vtiger_inventorysubproductrel.id = ? AND vtiger_inventorysubproductrel.sequence_no = ?";
            $result = $db->pquery($sql, array($this->parentId, $this->sequenceNo));
            if ($db->num_rows($result) > 0) {
                while ($row = $db->fetchByAssoc($result)) {
                    $this->subProductIds[] = $row["productid"];
                    $this->subProductNames[$row["productid"]] = $row["productname"];
                }
            } elseif($this->productId > 0) {
                $productRecordModel = Vtiger_Record_Model::getInstanceById($this->productId, "Products");   // services does not have subServices
                /* @var $productRecordModel Products_Record_Model */
                if ($productRecordModel) {
                    $subProducts = $productRecordModel->getSubProducts();
                    foreach ($subProducts as $subProductRecordModel) {
                        /* @var $subProductRecordModel Products_Record_Model */
                        $this->subProductIds[] = $subProductRecordModel->getId();
                        $this->subProductNames[$subProductRecordModel->getId()] = $subProductRecordModel->getName();
                    }
                }
            }

            //make sure that we do not go over the same SELECTs if simply there are no subProducts
            if ($this->subProductIds === null) {
                $this->subProductIds = array();
                $this->subProductNames = array();
            }
        }
    }

    /**
     * @param bool $asString
     * @return string|null
     */
    public function getSubProductNames($asString = false)
    {
        $this->initSubproducts();
        if (!$asString) {
            return $this->subProductNames;
        } else {
            if (!empty($this->subProductNames)) {
                $stringOutput = implode("<br> - ", $this->subProductNames);
                return " - " . $stringOutput;
            } else {
                return "";
            }
        }
    }

    /**
     * @return array
     */
    public function getCrossInventoryLineItems()
    {
        return $this->crossInventoryLineItems;
    }

    /**
     * @return string|null
     */
    public function getDiscountType()
    {
        if (!empty($this->discountPercent)) {
            return 'percentage';
        } elseif (!empty($this->discountAmount)) {
            return 'amount';
        } else {
            return null;
        }
    }

    /**
     * @return null
     */
    public function getDiscountPercent()
    {
        return $this->discountPercent;
    }

    /**
     * @param $discountPercent
     */
    public function setDiscountPercent($discountPercent)
    {
        $this->discountPercent = $discountPercent;
    }

    /**
     * @return null
     */
    public function getDiscountAmount()
    {
        return $this->discountAmount;
    }

    /**
     * @param $discountAmount
     */
    public function setDiscountAmount($discountAmount)
    {
        $this->discountAmount = $discountAmount;
    }

    /**
     * @param ITS4YouWarehouses_InventoryLineItem_Helper $lineItem
     */
    public function addCrossInventoryLineItem(ITS4YouWarehouses_InventoryLineItem_Helper $lineItem)
    {
        $this->crossInventoryLineItems[$lineItem->getItemId()] = $lineItem;
    }

    /**
     * @return int
     */
    public function getAdjustedQuantity()
    {
        $adjustedQuantity = $this->quantity;
        /* @var ITS4YouWarehouses_InventoryLineItem_Helper $lineItem */
        foreach ($this->crossInventoryLineItems as $lineItem) {
            $adjustedQuantity -= $lineItem->getAdjustedQuantity();
            if ($adjustedQuantity <= 0) {
                return 0;
            }
        }

        return $adjustedQuantity;
    }

    /**
     * @return string
     */
    public function getSubProductsToRequest()
    {
        $recordId = $this->getProductId();
        /** @var Products_Record_Model $recordModel */

        if(isRecordExists($recordId)) {
            $recordModel = Vtiger_Record_Model::getInstanceById($recordId, 'Products');
        }

        if ($recordModel) {
            $products = $recordModel->getSubProducts(true);
            $toRequest = '';

            foreach ((array)$products as $productId => $productRecord) {
                $toRequest .= sprintf('%s:%s,', $productId, $productRecord->get('quantityInBundle'));
            }

            return rtrim($toRequest, ',');
        }

        return '';
    }
}
