<?php
/* * *******************************************************************************
 * The content of this file is subject to the MultiWarehouses4You license.
 * ("License"); You may not use this file except in compliance with the License
 * The Initial Developer of the Original Code is IT-Solutions4You s.r.o.
 * Portions created by IT-Solutions4You s.r.o. are Copyright(C) IT-Solutions4You s.r.o.
 * All Rights Reserved.
 * ****************************************************************************** */

/**
 * Class ITS4YouWarehouses_SO_ActionBlock
 */
class ITS4YouWarehouses_SO_ActionBlock extends ITS4YouWarehouses_DNRelated_ActionBlock
{

    /**
     * @var
     */
    private $sostatus;

    /**
     * ITS4YouWarehouses_SO_ActionBlock constructor.
     * @param $_salesOrderId
     * @throws Exception
     */
    public function __construct($_salesOrderId)
    {
        parent::__construct($_salesOrderId);
        $this->readParentAtttributes();
        $this->processRelatedPurchaseOrders();
        $this->processRelatedDeliveryNotes();
        $this->processRelatedInvoices();
    }

    /**
     * @throws Exception
     */
    private function readParentAtttributes()
    {
        global $adb;

        $sql = "SELECT sostatus FROM vtiger_salesorder WHERE salesorderid = ?";
        $result = $adb->pquery($sql, array($this->parentId));
        $this->sostatus = $adb->query_result($result, 0, "sostatus");
    }

    /**
     * @throws Exception
     */
    private function processRelatedPurchaseOrders()
    {
        /* @var $poActionBlockItem ITS4YouWarehouses_ItemInbound_ActionBlock */
        $relatedPurchaseOrderIDs = $this->getToBeReceiptedPurchaseOrders();

        foreach ($relatedPurchaseOrderIDs as $purchaseOrderId => $purchaseOrderNo) {
            $poActionBlock = ITS4YouWarehouses_Base_ActionBlock::factory("PurchaseOrder", $purchaseOrderId);
            $poActionBlockItems = $poActionBlock->getItems();

            foreach ($poActionBlockItems as $itemId => $poActionBlockItem) {
                //skip non direct items
                if ($poActionBlockItem->getParentId() != $purchaseOrderId) {
                    continue;
                }
                $parentItem = $this->getItem($itemId);
                if ($parentItem === null) {
                    $newItem = new ITS4YouWarehouses_Item_ActionBlock($poActionBlockItem->getId(), $poActionBlockItem->getName(), $poActionBlockItem->getParentId(), 0, $poActionBlockItem->getAvailableQty(), $poActionBlockItem->getItemInfo(), $poActionBlockItem->getSEType());
                    if ($this->addItem($newItem)) {
                        $parentItem = $newItem;
                    }
                }
                if ($parentItem !== null) {
                    $parentItem->mergeItems($poActionBlockItem);
                }
            }
        }
    }

    /**
     * @return array
     */
    private function getToBeReceiptedPurchaseOrders()
    {
        return $this->getRelatedPurchaseOrderIDs(array('Created', 'Approved', 'Delivered', 'Received Shipment'));
    }

    /**
     * @param $_statuses
     * @return array
     */
    private function getRelatedPurchaseOrderIDs($_statuses)
    {
        global $adb;
        $sql = "SELECT vtiger_crmentityrel.relcrmid, vtiger_purchaseorder.purchaseorder_no
                FROM vtiger_crmentityrel 
                INNER JOIN vtiger_crmentity
                    ON vtiger_crmentity.crmid=vtiger_crmentityrel.relcrmid
                INNER JOIN vtiger_purchaseorder
                    ON vtiger_crmentity.crmid=vtiger_purchaseorder.purchaseorderid
                WHERE vtiger_crmentity.deleted = 0 
                    AND vtiger_crmentityrel.crmid=? 
                    AND vtiger_crmentityrel.module=? 
                    AND vtiger_crmentityrel.relmodule=?
                    AND vtiger_purchaseorder.postatus IN (" . generateQuestionMarks($_statuses) . ")";
        $params = array_merge(array($this->parentId, "SalesOrder", "PurchaseOrder"), $_statuses);
        $result = $adb->pquery($sql, $params);
        $purchaseOrderIDs = array();
        if ($adb->num_rows($result) > 0) {
            while ($row = $adb->fetchByAssoc($result)) {
                $purchaseOrderIDs[$row["relcrmid"]] = $row["purchaseorder_no"];
            }
        }

        return $purchaseOrderIDs;
    }

    /**
     * @throws Exception
     */
    private function processRelatedDeliveryNotes()
    {
        /* @var $dnActionBlock ITS4YouWarehouses_DN_ActionBlock */
        /* @var $dnActionBlockItem ITS4YouWarehouses_Item_ActionBlock */
        /* @var $itemId int */
        /* @var $item ITS4YouWarehouses_Item_ActionBlock */
        $deliveryNoteIDs = $this->getRelatedDeliveryNoteIDs(array('Delivered', 'Invoiced', 'Ready to Deliver'));

        foreach ($deliveryNoteIDs as $deliveryNoteID => $deliveryNoteNo) {
            $dnActionBlock = ITS4YouWarehouses_Base_ActionBlock::factory("ITS4YouWHDeliveryNotes", $deliveryNoteID);
            $dnActionBlockItems = $dnActionBlock->getItems();

            foreach ($dnActionBlockItems as $itemId => $dnActionBlockItem) {
                //skip non direct items
                if ($dnActionBlockItem->getParentId() != $deliveryNoteID) {
                    continue;
                }
                $parentItem = $this->getItem($itemId);
                if ($parentItem === null) {
                    $newItem = new ITS4YouWarehouses_Item_ActionBlock($dnActionBlockItem->getId(), $dnActionBlockItem->getName(), $dnActionBlockItem->getParentId(), 0, $dnActionBlockItem->getAvailableQty(), $dnActionBlockItem->getItemInfo(), $dnActionBlockItem->getSEType());
                    if ($this->addItem($newItem)) {
                        $parentItem = $newItem;
                    }
                }
                if ($parentItem !== null) {
                    $parentItem->mergeItems($dnActionBlockItem);
                }
            }
        }
        //check if any related opened DN exists, if so then add status for possibility to add items to these DNs
        $openedDeliveryNoteIDs = $this->getRelatedDeliveryNoteIDs(array('Created', 'Waiting availability', 'Ready to Deliver'));
        if (count($openedDeliveryNoteIDs) > 0) {
            $parentItems = $this->getItems();
            foreach ($parentItems as $itemId => $item) {
                $availableQty = $item->getAvailableQty();
                if ($item->getRestQty() > 0 && $availableQty > 0) {
                    $item->modifyExplicitStatus(ITS4YouWarehouses_ItemStatus_ActionBlock::STATUS_AVAILABLE_EXIST, $availableQty);
                }
            }
        }
    }

    /**
     * Return IDs of related Delivery Notes
     * @param array $_dnStatuses Array of statuses used as a filter for selection related Delivery Notes
     * @return array IDs of related Delivery Notes
     * @global PearDatabase $adb
     */
    protected function getRelatedDeliveryNoteIDs($_dnStatuses)
    {
        global $adb;

        if (!is_array($_dnStatuses)) {
            $_dnStatuses = array($_dnStatuses);
        }

        $sql = " SELECT its4you_deliverynotes.deliverynoteid, its4you_deliverynotes.deliverynotes_no
                 FROM its4you_deliverynotes
                 INNER JOIN vtiger_crmentity
                     ON vtiger_crmentity.crmid=its4you_deliverynotes.deliverynoteid                   
                 WHERE vtiger_crmentity.deleted=0
                   AND its4you_deliverynotes.referenceid=?                   
                   AND its4you_deliverynotes.deliverynotestatus IN (" . generateQuestionMarks($_dnStatuses) . ")";
        $params = array_merge(array($this->parentId), $_dnStatuses);

        $result = $adb->pquery($sql, $params);
        $deliveryNoteIDs = array();
        if ($adb->num_rows($result) > 0) {
            while ($row = $adb->fetchByAssoc($result)) {
                $deliveryNoteIDs[$row["deliverynoteid"]] = $row["deliverynotes_no"];
            }
        }
        return $deliveryNoteIDs;
    }

    /**
     * @throws Exception
     */
    private function processRelatedInvoices()
    {
        global $adb;

        //first we need to find Invoices related directly to SO => INVso = INVall - INVdn
        //INVall - invoices with field salesorder_id
        $INVall = array();
        $sql = "SELECT vtiger_invoice.invoiceid 
                    FROM vtiger_invoice 
                    INNER JOIN vtiger_crmentity
                        ON vtiger_crmentity.crmid = vtiger_invoice.invoiceid
                    WHERE vtiger_crmentity.deleted = 0
                        AND vtiger_invoice.salesorderid = ?";
        $result = $adb->pquery($sql, array($this->parentId));
        while ($row = $adb->fetchByAssoc($result)) {
            $INVall[$row["invoiceid"]] = $row["invoiceid"];
        }
        if (count($INVall) == 0) {
            return;
        }

        //INVdn - related delivery notes which contains reference in vtiger_crmentityrel
        $INVdn = array();
        $relatedDN = $this->getAllDeliveryNotes();
        if (count($relatedDN) > 0) {
            $relatedDN = array_keys($relatedDN);
            $sql = "SELECT DISTINCT its4you_wh_crossinventoryrel.target_crmid AS invoiceid
                    FROM  its4you_wh_crossinventoryrel
                    INNER JOIN vtiger_crmentity
                        ON vtiger_crmentity.crmid = its4you_wh_crossinventoryrel.target_crmid
                    WHERE vtiger_crmentity.deleted = 0
                      AND vtiger_crmentity.setype= 'Invoice'                      
                      AND its4you_wh_crossinventoryrel.source_crmid IN (" . generateQuestionMarks($relatedDN) . ")";
            $result = $adb->pquery($sql, $relatedDN);
            while ($row = $adb->fetchByAssoc($result)) {
                $INVdn[$row["invoiceid"]] = $row["invoiceid"];
            }
        }

        $INVso = array_diff($INVall, $INVdn);
        foreach ($INVso as $invoiceID) {
            $invActionBlock = ITS4YouWarehouses_Base_ActionBlock::factory("Invoice", $invoiceID);
            $invActionBlockItems = $invActionBlock->getItems();

            foreach ($invActionBlockItems as $itemId => $invActionBlockItem) {
                /* @var $invActionBlockItem ITS4YouWarehouses_Item_ActionBlock */
                //skip non direct items
                if ($invActionBlockItem->getParentId() != $invoiceID) {
                    continue;
                }

                $parentItem = $this->getItem($itemId);
                if ($parentItem === null) {
                    $newItem = new ITS4YouWarehouses_Item_ActionBlock($invActionBlockItem->getId(), $invActionBlockItem->getName(), $invActionBlockItem->getParentId(), 0, $invActionBlockItem->getAvailableQty(), $invActionBlockItem->getItemInfo(), $invActionBlockItem->getSEType());
                    if ($this->addItem($newItem)) {
                        $parentItem = $newItem;
                    }
                }
                if ($parentItem !== null) {
                    $alreadyInvoicedQty = $parentItem->getStatusQty(ITS4YouWarehouses_ItemStatus_ActionBlock::STATUS_INVOICED);
                    $newInvoicedQty = $alreadyInvoicedQty + $invActionBlockItem->getParentQty();
                    $parentItem->modifyExplicitStatus(ITS4YouWarehouses_ItemStatus_ActionBlock::STATUS_INVOICED, $newInvoicedQty);
                    $parentItem->modifyExplicitStatus(ITS4YouWarehouses_ItemStatus_ActionBlock::STATUS_DELIVERED, $newInvoicedQty);

                    $toBeInvoicedQty = $parentItem->getParentQty() - $newInvoicedQty;
                    $toBeInvoicedQty = $toBeInvoicedQty < 0 ? 0 : $toBeInvoicedQty;
                    $parentItem->modifyExplicitStatus(ITS4YouWarehouses_ItemStatus_ActionBlock::STATUS_TOBEINVOICED, $toBeInvoicedQty);
                }
            }
        }
    }

    /**
     * @return array
     */
    public function getAllDeliveryNotes()
    {
        return $this->getRelatedDeliveryNoteIDs(array("Created", "Waiting Availability", "Ready to Deliver", "Delivered", "Invoiced"));
    }

    /**
     * @return bool
     */
    public function isCanceled()
    {
        return $this->sostatus === "Cancelled";
    }

    /**
     * @param array $_selectedProducts
     * @return array|void
     * @throws Exception
     */
    public function getOverviewModel($_selectedProducts)
    {
        //in order to avoid passing $_selectedProducts through all methods chain
        //we will use workaround so $this->items will contains only those in $_selectedProducts
        $parentItems = $this->getItems();
        foreach ($parentItems as $itemId => $item) {
            if (!in_array($itemId, $_selectedProducts)) {
                $this->removeItem($itemId);
            }
        }
        $mainViewModel = $this->getMainViewModel();

        //display/hide columns with only zero values
        //init $columns settings
        $columns = array(
            "product_col" => array(
                "visible" => true,
                "statuses" => array(false)
            ),
            "purchased_col" => array(
                "visible" => false,
                "statuses" => array(ITS4YouWarehouses_ItemStatus_ActionBlock::STATUS_AWAITING)
            ),
            "receipted_col" => array(
                "visible" => false,
                "statuses" => array(false)
            ),
            "available_col" => array(
                "visible" => true,
                "statuses" => array(ITS4YouWarehouses_ItemStatus_ActionBlock::STATUS_NOTAVAILABLE)
            ),
            //PT00773 Starts
            "overall_purchased_col" => array(
                "visible" => true,
                "statuses" => array(false)
            ),
            "overall_ordered_col" => array(
                "visible" => true,
                "statuses" => array(false)
            ),
            "reorder_level_col" => array(
                "visible" => true,
                "statuses" => array(false)
            ),
            "optimal_level_col" => array(
                "visible" => true,
                "statuses" => array(false)
            ),
            //PT00773 Ends
            "ordered_col" => array(
                "visible" => true,
                "statuses" => array(ITS4YouWarehouses_ItemStatus_ActionBlock::STATUS_AVAILABLE, ITS4YouWarehouses_ItemStatus_ActionBlock::STATUS_AVAILABLE_EXIST, ITS4YouWarehouses_ItemStatus_ActionBlock::STATUS_AVAILABLE_PART)
            ),
            "todeliver_col" => array(
                "visible" => true,
                "statuses" => array(ITS4YouWarehouses_ItemStatus_ActionBlock::STATUS_ISSUED)
            ),
            "delivered_col" => array(
                "visible" => true,
                "statuses" => array(false)
            ),
            "returned_col" => array(
                "visible" => false,
                "statuses" => array(false)
            ),
            "toinvoice_col" => array(
                "visible" => true,
                "statuses" => array(ITS4YouWarehouses_ItemStatus_ActionBlock::STATUS_TOBEINVOICED)
            ),
            "invoiced_col" => array(
                "visible" => true,
                "statuses" => array(false)
            )
        );

        $qtyInfoEntries = OverallQtyInfoHandler::getQtyInfoEntries(array_keys($this->getItems()));   // PT00773

        foreach ($mainViewModel["items"] as $itemId => $data) {
            $item = $this->getItem($itemId);
            /* @var $item ITS4YouWarehouses_Item_ActionBlock */

            $purchasedQty = $item->getRelatedInboundItemQty();
            $returnedQty = $item->getStatusQty(ITS4YouWarehouses_ItemStatus_ActionBlock::STATUS_RETURNED);
            $deliveredQty = $item->getStatusQty(ITS4YouWarehouses_ItemStatus_ActionBlock::STATUS_DELIVERED);
            $invoicedQty = $item->getStatusQty(ITS4YouWarehouses_ItemStatus_ActionBlock::STATUS_INVOICED);
            $toBeInvoicedQtyRaw = $deliveredQty - $invoicedQty - $returnedQty;
            $toBeInvoicedQty = $toBeInvoicedQtyRaw < 0 ? 0 : $toBeInvoicedQtyRaw;
            $issuedQty = $item->getStatusQty(ITS4YouWarehouses_ItemStatus_ActionBlock::STATUS_ISSUED);

            if ($purchasedQty > 0) {
                $columns["purchased_col"]["visible"] = true;
                $columns["receipted_col"]["visible"] = true;
            }
            if ($returnedQty > 0) {
                $columns["returned_col"]["visible"] = true;
            }
            //add color information for ordered column
            $mainViewModel["items"][$itemId]["ordered_col_color"] = "";
            if ($issuedQty == 0 && $deliveredQty == 0 && $toBeInvoicedQty == 0 && $invoicedQty == 0 && $item->getSEType() == "Products") {
                if ($item->getParentQty() > $item->getAvailableQty()) {
                    $mainViewModel["items"][$itemId]["ordered_col_color"] = "color: red;";
                } else {
                    $mainViewModel["items"][$itemId]["ordered_col_color"] = "color: green;";
                }
            }

            //PT00773 Starts
            $qtyInfoEntry = $qtyInfoEntries[$itemId];
            /* @var $qtyInfoEntry OverallQtyInfoEntry */
            $mainViewModel["items"][$itemId]["overall_purchased_qty"] = self::formatNumberForDetailView($qtyInfoEntry->getToBeReceiptedQty());
            $mainViewModel["items"][$itemId]["overall_ordered_qty"] = self::formatNumberForDetailView($qtyInfoEntry->getToBeDeliveredQty());
            $mainViewModel["items"][$itemId]["reorder_level_qty"] = ($item->getSEType() == "Products" ? self::formatNumberForDetailView($item->getItemInfo()->getReorderLevel()) : ITS4YouWarehouses_Item_ActionBlock::SERVICE_ITEM_AVAIL_QTY);
            $mainViewModel["items"][$itemId]["optimal_level_qty"] = ($item->getSEType() == "Products" ? self::formatNumberForDetailView($item->getItemInfo()->getOptimalLevel()) : ITS4YouWarehouses_Item_ActionBlock::SERVICE_ITEM_AVAIL_QTY);
            //PT00773 Ends
        }
        $mainViewModel["columnsSettings"] = $columns;

        foreach ($columns as $columnName => $columnSettings) {
            if ($columnSettings["visible"] === true) {
                foreach ($columnSettings["statuses"] as $statusId) {
                    if ($statusId !== false && isset($mainViewModel["actions"][$statusId])) {
                        $mainViewModel["overviewActions"][$columnName] = $mainViewModel["actions"][$statusId][0];
                        break;
                    } else {
                        $mainViewModel["overviewActions"][$columnName] = "";
                    }
                }
            }
        }

        //provide smarty template name
        $mainViewModel["overviewTemplate"] = "SOActionBlockOverview.tpl";

        return $mainViewModel;
    }

    /**
     * @return array
     * @throws Exception
     */
    public function getMainViewModel()
    {
        $viewModel = parent::getMainViewModel();

        //Project Simple DeliveryNotes4You and Single Warehouses4You specialities
        //invoicing is done based on DN records not based on their items, so we need to check this
        if (isset($viewModel["actions"][ITS4YouWarehouses_ItemStatus_ActionBlock::STATUS_TOBEINVOICED])) {
            if (count($this->getToBeInvoicedDeliveryNotes()) == 0) {
                unset($viewModel["actions"][ITS4YouWarehouses_ItemStatus_ActionBlock::STATUS_TOBEINVOICED]);
            }
        }

        return $viewModel;
    }

    /**
     * Return IDs of related Delivery Notes which can be Invoiced
     * @return array Array of Delivery Note IDs
     * @global PearDatabase $adb
     */
    public function getToBeInvoicedDeliveryNotes()
    {
        global $adb;

        $relatedDeliveryNoteIDs = $this->getRelatedDeliveryNoteIDs(array('Delivered', 'Invoiced'));
        $toBeInvoicedDeliveryNoteIDs = array();
        if (count($relatedDeliveryNoteIDs) > 0) {
            $sql = "SELECT its4you_deliverynotes.deliverynoteid, its4you_deliverynotes.deliverynotes_no
                    FROM its4you_deliverynotes                
                    LEFT JOIN vtiger_crmentity AS inv_crm
                        ON its4you_deliverynotes.invoice_id = inv_crm.crmid
                    WHERE its4you_deliverynotes.deliverynoteid IN (" . generateQuestionMarks($relatedDeliveryNoteIDs) . ")
                        AND (its4you_deliverynotes.invoice_id = 0 OR inv_crm.deleted = 1)
                    ORDER BY its4you_deliverynotes.deliverynoteid";
            $result = $adb->pquery($sql, array_keys($relatedDeliveryNoteIDs));
            if ($adb->num_rows($result) > 0) {
                while ($row = $adb->fetchByAssoc($result)) {
                    $toBeInvoicedDeliveryNoteIDs[$row["deliverynoteid"]] = $row["deliverynotes_no"];
                }
            }
        }
        return $toBeInvoicedDeliveryNoteIDs;
    }

    /**
     * Process all items of action block and check whether delivered quantity >= originally ordered quantity
     * @return bool
     */
    public function isCompletelyInvoiced()
    {
        /* @var $item ITS4YouWarehouses_Item_ActionBlock */
        $parentItems = $this->getItems();
        if (count($parentItems) == 0) {
            return false;
        }

        foreach ($parentItems as $itemId => $item) {
            if ($item->getStatusQty(ITS4YouWarehouses_ItemStatus_ActionBlock::STATUS_INVOICED) < $item->getParentQty()) {
                return false;
            }
        }

        return true;
    }

    /**
     * @param array $_selectedProducts
     * @return array
     * @throws AppException
     * @throws Exception
     */
    public function getWaitingAvailabilityActionViewModel($_selectedProducts)
    {
        $items = array();

        $relatedPurchaseOrderIDs = $this->getToBeReceiptedPurchaseOrders();
        $lineItems = array();
        foreach ($relatedPurchaseOrderIDs as $purchaseOrderId => $purchaseOrderNo) {
            $poActionBlock = ITS4YouWarehouses_Base_ActionBlock::factory("PurchaseOrder", $purchaseOrderId);
            /* @var $poActionBlock ITS4YouWarehouses_PO_ActionBlock */
            $lineItems = (array)$lineItems + (array)($poActionBlock->getToBeReceiptedLineItems());
        }

        foreach ($lineItems as $lineItemId => $lineItem) {
            /* @var $lineItem ITS4YouWarehouses_InventoryLineItem_Helper */
            $productId = $lineItem->getProductId();
            if (!in_array($productId, $_selectedProducts)) {
                continue;
            }

            $item = $this->getItem($productId);
            $items[$lineItemId]["productid"] = $productId;
            $items[$lineItemId]["name"] = $item->getName();
            $items[$lineItemId]["receipt_date"] = getNewDisplayDate();
            $items[$lineItemId]["price"] = self::formatNumberForEditView($lineItem->getPrice());
            $items[$lineItemId]["toreceipt_qty"] = self::formatNumberForEditView($lineItem->getQuantity());
        }

        $listViewModel = Vtiger_ListView_Model::getInstance("ITS4YouWarehouses");
        $pagingModel = new Vtiger_Paging_Model();
        $pagingModel->set("limit", 1000);
        $listViewEntries = $listViewModel->getListViewEntries($pagingModel);
        foreach ($listViewEntries as $recordId => $listViewRecordModel) {
            $other["warehouses"][$recordId] = $listViewRecordModel->get("warehouse_name");
        }

        $other["user_date_format"] = Users_Record_Model::getCurrentUserModel()->get("date_format");

        return array("items" => $items, "other" => $other);
    }

    /**
     * @return array
     */
    public function getAvailableDeliveryNotes()
    {
        return $this->getRelatedDeliveryNoteIDs(array("Created", "Waiting Availability", "Ready to Deliver"));
    }

    /**
     * Return array of related Delivery Notes which can be Delivered
     * @return array key => Delivery Note ID<br />value => Delivery Note No
     */
    public function getToBeDeliveredDeliveryNotes()
    {
        //delivered DN are taken as well in order to fix any possible incosistencies between status of DN and its IC
        return $this->getRelatedDeliveryNoteIDs(array("Ready to Deliver", "Delivered"));
    }
}