<?php
/* * *******************************************************************************
 * The content of this file is subject to the MultiWarehouses4You license.
 * ("License"); You may not use this file except in compliance with the License
 * The Initial Developer of the Original Code is IT-Solutions4You s.r.o.
 * Portions created by IT-Solutions4You s.r.o. are Copyright(C) IT-Solutions4You s.r.o.
 * All Rights Reserved.
 * ****************************************************************************** */

/**
 * Class ITS4YouWarehouses_INV_ActionBlock
 */
class ITS4YouWarehouses_INV_ActionBlock extends ITS4YouWarehouses_DNRelated_ActionBlock
{

    /**
     * @var
     */
    private $invoicestatus;

    /**
     * ITS4YouWarehouses_INV_ActionBlock constructor.
     * @param $_invoiceId
     * @throws Exception
     */
    public function __construct($_invoiceId)
    {
        parent::__construct($_invoiceId);
        $this->readParentAttributes();
        $this->processRelatedPurchaseOrders();
        $this->processRelatedDeliveryNotes();
    }

    /**
     * @throws Exception
     */
    private function readParentAttributes()
    {
        global $adb;

        $sql = "SELECT invoicestatus FROM vtiger_invoice WHERE invoiceid = ?";
        $result = $adb->pquery($sql, array($this->parentId));
        $this->invoicestatus = $adb->query_result($result, 0, "invoicestatus");
    }

    /**
     * @throws Exception
     */
    private function processRelatedPurchaseOrders()
    {
        /* @var $poActionBlockItem ITS4YouWarehouses_ItemInbound_ActionBlock */
        $relatedPurchaseOrderIDs = $this->getToBeReceiptedPurchaseOrders();

        foreach ($relatedPurchaseOrderIDs as $purchaseOrderId => $purchaseOrderNo) {
            $poActionBlock = ITS4YouWarehouses_Base_ActionBlock::factory("PurchaseOrder", $purchaseOrderId);
            $poActionBlockItems = $poActionBlock->getItems();

            foreach ($poActionBlockItems as $itemId => $poActionBlockItem) {
                //skip non direct items
                if ($poActionBlockItem->getParentId() != $purchaseOrderId) {
                    continue;
                }

                $parentItem = $this->getItem($itemId);
                if ($parentItem === null) {
                    $newItem = new ITS4YouWarehouses_Item_ActionBlock($poActionBlockItem->getId(), $poActionBlockItem->getName(), $poActionBlockItem->getParentId(), 0, $poActionBlockItem->getAvailableQty(), $poActionBlockItem->getItemInfo(), $poActionBlockItem->getSEType());
                    if ($this->addItem($newItem)) {
                        $parentItem = $newItem;
                    }
                }
                if ($parentItem !== null) {
                    $parentItem->mergeItems($poActionBlockItem);
                }
            }
        }
    }

    /**
     * @return array
     */
    private function getToBeReceiptedPurchaseOrders()
    {
        return $this->getRelatedPurchaseOrderIDs(array("Created", "Approved", "Delivered", "Received Shipment"));
    }

    /**
     * @param $_statuses
     * @return array
     */
    private function getRelatedPurchaseOrderIDs($_statuses)
    {
        global $adb;
        $sql = "SELECT vtiger_crmentityrel.relcrmid, vtiger_purchaseorder.purchaseorder_no
                FROM vtiger_crmentityrel 
                INNER JOIN vtiger_crmentity
                    ON vtiger_crmentity.crmid=vtiger_crmentityrel.relcrmid
                INNER JOIN vtiger_purchaseorder
                    ON vtiger_crmentity.crmid=vtiger_purchaseorder.purchaseorderid
                WHERE vtiger_crmentity.deleted = 0 
                    AND vtiger_crmentityrel.crmid=? 
                    AND vtiger_crmentityrel.module=? 
                    AND vtiger_crmentityrel.relmodule=?
                    AND vtiger_purchaseorder.postatus IN (" . generateQuestionMarks($_statuses) . ")";
        $params = array_merge(array($this->parentId, "Invoice", "PurchaseOrder"), $_statuses);
        $result = $adb->pquery($sql, $params);
        $purchaseOrderIDs = array();
        if ($adb->num_rows($result) > 0) {
            while ($row = $adb->fetchByAssoc($result)) {
                $purchaseOrderIDs[$row["relcrmid"]] = $row["purchaseorder_no"];
            }
        }

        return $purchaseOrderIDs;
    }

    /**
     * @throws Exception
     */
    private function processRelatedDeliveryNotes()
    {
        /* @var $dnActionBlock ITS4YouWarehouses_DN_ActionBlock */
        /* @var $dnActionBlockItem ITS4YouWarehouses_Item_ActionBlock */
        /* @var $itemId int */
        /* @var $item ITS4YouWarehouses_Item_ActionBlock */
        $deliveryNoteIDs = $this->getRelatedDeliveryNoteIDs(array('Delivered', 'Invoiced', 'Ready to Deliver', 'Canceled'));
        foreach ($deliveryNoteIDs as $deliveryNoteID => $deliveryNoteNo) {
            $dnActionBlock = ITS4YouWarehouses_Base_ActionBlock::factory("ITS4YouWHDeliveryNotes", $deliveryNoteID);
            $dnActionBlockItems = $dnActionBlock->getItems();
            foreach ($dnActionBlockItems as $itemId => $dnActionBlockItem) {
                $parentItem = $this->getItem($itemId);
                if ($parentItem === null) {
                    $newItem = new ITS4YouWarehouses_Item_ActionBlock($dnActionBlockItem->getId(), $dnActionBlockItem->getName(), $dnActionBlockItem->getParentId(), 0, $dnActionBlockItem->getAvailableQty(), $dnActionBlockItem->getItemInfo(), $dnActionBlockItem->getSEType());
                    if ($this->addItem($newItem)) {
                        $parentItem = $newItem;
                    }
                }
                if ($parentItem !== null) {
                    $parentItem->mergeItems($dnActionBlockItem);
                }
            }
        }

        //check if any related opened DN exists, if so then add status for possibility to add items to these DNs
        $openedDeliveryNoteIDs = $this->getRelatedDeliveryNoteIDs(array('Created', 'Waiting availability', 'Ready to Deliver'));
        if (count($openedDeliveryNoteIDs) > 0) {
            $parentItems = $this->getItems();
            foreach ($parentItems as $itemId => $item) {
                $availableQty = $item->getAvailableQty();
                if ($item->getRestQty() > 0 && $availableQty > 0) {
                    $item->modifyExplicitStatus(ITS4YouWarehouses_ItemStatus_ActionBlock::STATUS_AVAILABLE_EXIST, $availableQty);
                }
            }
        }
    }

    /**
     * Return IDs of related Delivery Notes
     * @param array $_dnStatuses Array of statuses used as a filter for selection related Delivery Notes
     * @return array IDs of related Delivery Notes
     * @global PearDatabase $adb
     */
    protected function getRelatedDeliveryNoteIDs($_dnStatuses)
    {
        global $adb;

        if (!is_array($_dnStatuses)) {
            $_dnStatuses = array($_dnStatuses);
        }

        $sql = " SELECT its4you_deliverynotes.deliverynoteid, its4you_deliverynotes.deliverynotes_no
                 FROM its4you_deliverynotes
                 INNER JOIN vtiger_crmentity
                     ON vtiger_crmentity.crmid=its4you_deliverynotes.deliverynoteid                   
                 WHERE vtiger_crmentity.deleted=0
                   AND its4you_deliverynotes.invoice_id=?                   
                   AND its4you_deliverynotes.deliverynotestatus IN (" . generateQuestionMarks($_dnStatuses) . ")";
        $params = array_merge(array($this->parentId), $_dnStatuses);
        $result = $adb->pquery($sql, $params);
        $deliveryNoteIDs = array();
        if ($adb->num_rows($result) > 0) {
            while ($row = $adb->fetchByAssoc($result)) {
                $deliveryNoteIDs[$row["deliverynoteid"]] = $row["deliverynotes_no"];
            }
        }

        return $deliveryNoteIDs;
    }

    /**
     * @return bool
     */
    public function isCanceled()
    {
        return $this->invoicestatus === "Cancel";
    }

    /**
     * Return array of related Delivery Notes which can be Delivered
     * @return array key => Delivery Note ID<br />value => Delivery Note No
     */
    public function getToBeDeliveredDeliveryNotes()
    {
        return $this->getRelatedDeliveryNoteIDs(array("Ready to Deliver"));
    }

    /**
     * @return array
     */
    public function getAvailableDeliveryNotes()
    {
        return $this->getRelatedDeliveryNoteIDs(array("Created", "Waiting Availability", "Ready to Deliver"));
    }

    /**
     * @throws Exception
     */
    public function getToBeInvoicedDeliveryNotes()
    {
        throw new Exception("Should not be implemented.");
    }

    /**
     * @param array $_selectedProducts
     * @return array
     * @throws Exception
     */
    public function getOverviewModel($_selectedProducts)
    {
        //in order to avoid passing $_selectedProducts through all methods chain
        //we will use workaround so $this->items will contains only those in $_selectedProducts
        $parentItems = $this->getItems();
        foreach ($parentItems as $itemId => $item) {
            if (!in_array($itemId, $_selectedProducts)) {
                $this->removeItem($itemId);
            }
        }

        $mainViewModel = parent::getMainViewModel();

        //display/hide columns with only zero values
        //init $columns settings
        $columns = array(
            'product_col' => array(
                'visible' => true,
                'statuses' => array(false),
                'label' => 'Product Name',
                'name' => 'name',
            ),
            'purchased_col' => array(
                'visible' => false,
                'statuses' => array(ITS4YouWarehouses_ItemStatus_ActionBlock::STATUS_AWAITING),
                'label' => 'HEAD_PURCHASED_QTY',
                'name' => 'purchased_qty',
            ),
            'receipted_col' => array(
                'visible' => false,
                'statuses' => array(false),
                'label' => 'HEAD_RECEIPTED_QTY',
                'name' => 'receipted_qty',
            ),
            'available_col' => array(
                'visible' => true,
                'statuses' => array(ITS4YouWarehouses_ItemStatus_ActionBlock::STATUS_NOTAVAILABLE),
                'label' => 'HEAD_QTY_STOCK',
                'name' => 'available_qty',
            ),
            //PT00773 Starts
            'overall_purchased_col' => array(
                'visible' => true,
                'statuses' => array(false),
                'label' => 'HEAD_OVERALL_PURCHASED',
                'name' => 'overall_purchased_qty',
            ),
            'overall_ordered_col' => array(
                'visible' => true,
                'statuses' => array(false),
                'label' => 'HEAD_OVERALL_ORDERED',
                'name' => 'overall_ordered_qty',
            ),
            'reorder_level_col' => array(
                'visible' => true,
                'statuses' => array(false),
                'label' => vtranslate('Reorder Level', 'Products'),
                'name' => 'reorder_level_qty',
            ),
            'optimal_level_col' => array(
                'visible' => true,
                'statuses' => array(false),
                'label' => vtranslate('OptimalLevel', 'Products'),
                'name' => 'optimal_level_qty',
            ),
            //PT00773 Ends
            'invoiced_col' => array(
                'visible' => true,
                'statuses' => array(ITS4YouWarehouses_ItemStatus_ActionBlock::STATUS_AVAILABLE, ITS4YouWarehouses_ItemStatus_ActionBlock::STATUS_AVAILABLE_EXIST, ITS4YouWarehouses_ItemStatus_ActionBlock::STATUS_AVAILABLE_PART),
                'label' => 'HEAD_INVOICED_QTY',
                'name' => 'ordered_qty',
            ),
            'todeliver_col' => array(
                'visible' => true,
                'statuses' => array(ITS4YouWarehouses_ItemStatus_ActionBlock::STATUS_ISSUED),
                'label' => 'HEAD_READY_QTY',
                'name' => 'issued_qty',
            ),
            'delivered_col' => array(
                'visible' => true,
                'statuses' => array(false),
                'label' => 'HEAD_DELIVERED_QTY',
                'name' => 'delivered_qty',
            ),
            'canceled_col' => array(
                'visible' => false,
                'statuses' => array(false),
                'label' => 'HEAD_CANCELED_QTY',
                'name' => 'canceled_qty',
            )
        );

        $qtyInfoEntries = OverallQtyInfoHandler::getQtyInfoEntries(array_keys($this->getItems()));   // PT00773

        foreach ($mainViewModel["items"] as $itemId => $data) {
            $item = $this->getItem($itemId);
            /* @var $item ITS4YouWarehouses_Item_ActionBlock */

            $purchasedQty = $item->getRelatedInboundItemQty();
            $canceledQty = $item->getStatusQty(ITS4YouWarehouses_ItemStatus_ActionBlock::STATUS_CANCELED);
            $issuedQty = $item->getStatusQty(ITS4YouWarehouses_ItemStatus_ActionBlock::STATUS_ISSUED);
            $deliveredQty = $item->getStatusQty(ITS4YouWarehouses_ItemStatus_ActionBlock::STATUS_DELIVERED);

            if ($purchasedQty > 0) {
                $columns["purchased_col"]["visible"] = true;
                $columns["receipted_col"]["visible"] = true;
            }
            if ($canceledQty > 0) {
                $columns["canceled_col"]["visible"] = true;
            }

            $mainViewModel["items"][$itemId]["ordered_col_color"] = "";
            if ($issuedQty == 0 && $deliveredQty == 0 && $item->getSEType() == "Products") {
                if ($item->getParentQty() > $item->getAvailableQty()) {
                    $mainViewModel["items"][$itemId]["ordered_col_color"] = "color: red;";
                } else {
                    $mainViewModel["items"][$itemId]["ordered_col_color"] = "color: green;";
                }
            }

            //PT00773 Starts
            $qtyInfoEntry = $qtyInfoEntries[$itemId];
            /* @var $qtyInfoEntry OverallQtyInfoEntry */
            $mainViewModel["items"][$itemId]["overall_purchased_qty"] = self::formatNumberForDetailView($qtyInfoEntry->getToBeReceiptedQty());
            $mainViewModel["items"][$itemId]["overall_ordered_qty"] = self::formatNumberForDetailView($qtyInfoEntry->getToBeDeliveredQty());
            $mainViewModel["items"][$itemId]["reorder_level_qty"] = ($item->getSEType() == "Products" ? self::formatNumberForDetailView($item->getItemInfo()->getReorderLevel()) : ITS4YouWarehouses_Item_ActionBlock::SERVICE_ITEM_AVAIL_QTY);
            $mainViewModel["items"][$itemId]["optimal_level_qty"] = ($item->getSEType() == "Products" ? self::formatNumberForDetailView($item->getItemInfo()->getOptimalLevel()) : ITS4YouWarehouses_Item_ActionBlock::SERVICE_ITEM_AVAIL_QTY);
            //PT00773 Ends
        }
        $mainViewModel["columnsSettings"] = $columns;

        foreach ($columns as $columnName => $columnSettings) {
            if ($columnSettings["visible"] === true) {
                foreach ($columnSettings["statuses"] as $statusId) {
                    if ($statusId !== false && isset($mainViewModel["actions"][$statusId])) {
                        $mainViewModel["overviewActions"][$columnName] = $mainViewModel["actions"][$statusId][0];
                        break;
                    } else {
                        $mainViewModel["overviewActions"][$columnName] = "";
                    }
                }
            }
        }

        //provide smarty template name
        $mainViewModel["overviewTemplate"] = "INVActionBlockOverview.tpl";
        return $mainViewModel;
    }

    /**
     * @param array $_selectedProducts
     * @return array
     * @throws AppException
     * @throws Exception
     */
    public function getWaitingAvailabilityActionViewModel($_selectedProducts)
    {
        $items = array();

        $relatedPurchaseOrderIDs = $this->getToBeReceiptedPurchaseOrders();
        $lineItems = array();
        foreach ($relatedPurchaseOrderIDs as $purchaseOrderId => $purchaseOrderNo) {
            $poActionBlock = ITS4YouWarehouses_Base_ActionBlock::factory("PurchaseOrder", $purchaseOrderId);
            /* @var $poActionBlock ITS4YouWarehouses_PO_ActionBlock */
            $lineItems = (array)$lineItems + (array)($poActionBlock->getToBeReceiptedLineItems());
        }

        foreach ($lineItems as $lineItemId => $lineItem) {
            /* @var $lineItem ITS4YouWarehouses_InventoryLineItem_Helper */
            $productId = $lineItem->getProductId();
            if (!in_array($productId, $_selectedProducts)) {
                continue;
            }

            $item = $this->getItem($productId);
            $items[$lineItemId]["productid"] = $productId;
            $items[$lineItemId]["name"] = $item->getName();
            $items[$lineItemId]["receipt_date"] = getNewDisplayDate();
            $items[$lineItemId]["price"] = self::formatNumberForEditView($lineItem->getPrice());
            $items[$lineItemId]["toreceipt_qty"] = self::formatNumberForEditView($lineItem->getQuantity());
        }

        $listViewModel = Vtiger_ListView_Model::getInstance("ITS4YouWarehouses");
        $pagingModel = new Vtiger_Paging_Model();
        $pagingModel->set("limit", 1000);
        $listViewEntries = $listViewModel->getListViewEntries($pagingModel);
        foreach ($listViewEntries as $recordId => $listViewRecordModel) {
            $other["warehouses"][$recordId] = $listViewRecordModel->get("warehouse_name");
        }

        $other["user_date_format"] = Users_Record_Model::getCurrentUserModel()->get("date_format");

        return array("items" => $items, "other" => $other);
    }

    /**
     *
     */
    protected function initStatusActionsPool()
    {
        $this->statusActionsPool = ITS4YouWarehouses_INVStatusActionsPool_ActionBlock::getInstance();
    }

}