<?php
/* * *******************************************************************************
 * The content of this file is subject to the MultiWarehouses4You license.
 * ("License"); You may not use this file except in compliance with the License
 * The Initial Developer of the Original Code is IT-Solutions4You s.r.o.
 * Portions created by IT-Solutions4You s.r.o. are Copyright(C) IT-Solutions4You s.r.o.
 * All Rights Reserved.
 * ****************************************************************************** */

/**
 * Class ITS4YouWarehouses_DN_ActionBlock
 */
class ITS4YouWarehouses_DN_ActionBlock extends ITS4YouWarehouses_Base_ActionBlock implements ITS4YouWarehouses_IStatusActionBlock_Sequential
{
    /**
     *
     */
    const ACTION_MODE_DELIVER = "deliver";
    /**
     *
     */
    const ACTION_MODE_INVOICE = "invoice";
    /**
     *
     */
    const ACTION_MODE_CANCEL = "cancel";

    /**
     * @var string[]
     */
    protected static $initialStatuses = array("Created", "Waiting Availability", "Ready to Deliver");
    /**
     * @var
     */
    protected $deliveryNoteStatus;
    /**
     * @var
     */
    protected $relatedInvoiceId;
    /**
     * @var
     */
    protected $salesOrderId;
    /**
     * @var
     */
    protected $successiveQtyResidues;

    /**
     * ITS4YouWarehouses_DN_ActionBlock constructor.
     * @param $_delivertNoteId
     * @throws Exception
     */
    protected function __construct($_delivertNoteId)
    {
        parent::__construct($_delivertNoteId);

        $this->processReturnedProducts();
        $this->processRelatedIssueCards();
        $this->processRelatedInvoice();
    }

    /**
     * @throws Exception
     */
    protected function processReturnedProducts()
    {
        $this->getLineItems();
        $relatedReceiptCards = ITS4YouWarehouses_DNUtil_ActionBlock::getRelatedReceiptcards($this->parentId, array("Created", "Delivered"));
        CrossInventoryRelEventHandler::addCrossInventoryRels($this->lineItems, array_keys($relatedReceiptCards));

        foreach ($this->lineItems as $lineItem) {
            $parentItem = $this->getItem($lineItem->getProductId());
            if ($parentItem === null) {
                continue;
            }

            $returnedQty = $parentItem->getStatusQty(ITS4YouWarehouses_ItemStatus_ActionBlock::STATUS_RETURNED);
            $returnedQty += $lineItem->getQuantity() - $lineItem->getAdjustedQuantity();

            $parentItem->modifyExplicitStatus(ITS4YouWarehouses_ItemStatus_ActionBlock::STATUS_RETURNED, $returnedQty);
        }
    }

    /**
     * @return array
     * @throws Exception
     */
    public function getLineItems()
    {
        if (empty($this->lineItems)) {
            $this->lineItems = ITS4YouWarehouses_InventoryLineItem_Helper::getParentLineItems($this->parentId);
            if ($this->salesOrderId > 0) {
                $salesOrderLineItems = ITS4YouWarehouses_InventoryLineItem_Helper::getParentLineItems($this->salesOrderId);

                foreach ($this->lineItems as $lineItemId => $lineItem) {
                    /* @var $lineItem ITS4YouWarehouses_InventoryLineItem_Helper */

                    //search price from first lineItem from SalesOrder based on productId
                    $found = false;
                    foreach ($salesOrderLineItems as $salesOrderLineItemId => $salesOrderLineItem) {
                        /* @var $salesOrderLineItem ITS4YouWarehouses_InventoryLineItem_Helper */
                        if ($salesOrderLineItem->getProductId() == $lineItem->getProductId()) {
                            $lineItem->setPrice($salesOrderLineItem->getPrice());
                            $lineItem->setDiscountAmount($salesOrderLineItem->getDiscountAmount());
                            $lineItem->setDiscountPercent($salesOrderLineItem->getDiscountPercent());
                            break;
                        }
                    }
                    //if SalesOrder does not contain any lineitem with the productId, use unit price
                    if (!$found) {
                        $item = $this->getItem($lineItem->getProductId());
                        /* @var $item ITS4YouWarehouses_Item_ActionBlock */
                        if ($item != null) {
                            $lineItem->setPrice($item->getItemInfo()->getUnitPrice());
                        }
                    }
                }
            } else {
                foreach ($this->lineItems as $lineItemId => $lineItem) {
                    /* @var $lineItem ITS4YouWarehouses_InventoryLineItem_Helper */
                    if ($lineItem->getPrice() == 0) {
                        $item = $this->getItem($lineItem->getProductId());
                        /* @var $item ITS4YouWarehouses_Item_ActionBlock */
                        if ($item != null) {
                            $lineItem->setPrice($item->getItemInfo()->getUnitPrice());
                        }
                    }
                }
            }
        }

        return $this->lineItems;
    }

    /**
     *
     */
    protected function processRelatedIssueCards()
    {
        global $adb;

        $relatedIssueCardIDs = $this->getRelatedIssueCards(array("Transferred from Warehouse", "Delivered"));

        if (ITS4YouWarehouses_Utils_Helper::count($relatedIssueCardIDs) > 0) {
            $sql = "SELECT  CASE WHEN vtiger_products.productid != '' THEN vtiger_products.productname ELSE vtiger_service.servicename END AS 'productname',
                        CASE WHEN vtiger_products.productid != '' THEN vtiger_products.qtyinstock ELSE '0' END AS 'available_qty',            
                        " . ITS4YouWarehouses_ItemInfo_ActionBlock::getItemInfoQuery() . ",                        
                    its4you_issuecards.issuecardstatus,
                    issuecard_rel.id,                    
                    issuecard_rel.productid,
                    SUM(issuecard_rel.quantity) AS qty,
                    vtiger_crmentity.setype
                FROM vtiger_inventoryproductrel AS issuecard_rel
                INNER JOIN vtiger_crmentity
                    ON vtiger_crmentity.crmid = issuecard_rel.id
                INNER JOIN its4you_issuecards
                    ON its4you_issuecards.issuecardid = issuecard_rel.id
                LEFT JOIN vtiger_products
                    ON vtiger_products.productid=issuecard_rel.productid
                LEFT JOIN vtiger_service
                    ON vtiger_service.serviceid=issuecard_rel.productid                
                WHERE vtiger_crmentity.deleted = 0
                    AND issuecard_rel.id IN (" . generateQuestionMarks($relatedIssueCardIDs) . ")                 
                GROUP BY its4you_issuecards.issuecardstatus, issuecard_rel.productid 
                ORDER BY issuecard_rel.sequence_no";
            $result = $adb->pquery($sql, array_keys($relatedIssueCardIDs));
            if ($adb->num_rows($result) > 0) {
                while ($row = $adb->fetchByAssoc($result)) {
                    $parentItem = $this->getItem($row["productid"]);

                    if ($parentItem === null) {
                        $newItem = new ITS4YouWarehouses_Item_ActionBlock($row["productid"], $row["productname"], $row["id"], 0, $row["available_qty"], ITS4YouWarehouses_ItemInfo_ActionBlock::initialize($row), $row["setype"]);
                        if ($this->addItem($newItem)) {
                            $parentItem = $newItem;
                        }
                    }
                    if ($parentItem !== null) {
                        switch ($row["issuecardstatus"]) {
                            case "Transferred from Warehouse":
                                $parentItem->modifyExplicitStatus(ITS4YouWarehouses_ItemStatus_ActionBlock::STATUS_ISSUED, $row["qty"]);
                                break;

                            case "Delivered":
                                $parentItem->modifyExplicitStatus(ITS4YouWarehouses_ItemStatus_ActionBlock::STATUS_DELIVERED, $row["qty"]);

                                $toBeInvoicedQty = $row["qty"] - $parentItem->getStatusQty(ITS4YouWarehouses_ItemStatus_ActionBlock::STATUS_RETURNED);
                                $parentItem->modifyExplicitStatus(ITS4YouWarehouses_ItemStatus_ActionBlock::STATUS_TOBEINVOICED, $toBeInvoicedQty);
                                break;
                        }
                    }
                }
            }
        }

        //it's needed to handle Services since they are not at ICs
        foreach ($this->getItems() as $item) {
            /* @var $item ITS4YouWarehouses_Item_ActionBlock */
            if ($item->getSEType() == "Products") {
                continue;
            }

            switch ($this->deliveryNoteStatus) {
                case "Ready to Deliver":
                    $item->modifyExplicitStatus(ITS4YouWarehouses_ItemStatus_ActionBlock::STATUS_ISSUED, $item->getParentQty());
                    break;

                case "Delivered":
                    $item->modifyExplicitStatus(ITS4YouWarehouses_ItemStatus_ActionBlock::STATUS_DELIVERED, $item->getParentQty());
                    $toBeInvoicedQty = $item->getParentQty() - $item->getStatusQty(ITS4YouWarehouses_ItemStatus_ActionBlock::STATUS_RETURNED);
                    $item->modifyExplicitStatus(ITS4YouWarehouses_ItemStatus_ActionBlock::STATUS_TOBEINVOICED, $toBeInvoicedQty);
                    break;
            }
        }
    }

    /**
     * @param $_statuses
     * @return array
     */
    protected function getRelatedIssueCards($_statuses)
    {
        return ITS4YouWarehouses_DNUtil_ActionBlock::getRelatedIssueCards($this->parentId, $_statuses);
    }

    /**
     *
     */
    protected function processRelatedInvoice()
    {
        global $adb;

        /* @var $item ITS4YouWarehouses_Item_ActionBlock */
        if ($this->deliveryNoteStatus != "Invoiced") {
            return;
        }

        $sql = "SELECT  CASE WHEN vtiger_products.productid != '' THEN vtiger_products.productname ELSE vtiger_service.servicename END AS 'productname',
                        CASE WHEN vtiger_products.productid != '' THEN vtiger_products.qtyinstock ELSE '0' END AS 'available_qty',
                        " . ITS4YouWarehouses_ItemInfo_ActionBlock::getItemInfoQuery() . ",
                        invoice_rel.productid,
                        SUM(invoice_rel.quantity) AS qty,
                        invoice_rel.id
                FROM vtiger_inventoryproductrel AS invoice_rel
                INNER JOIN " . CrossInventoryRelEventHandler::TABLE_NAME . " AS cross_rel
                    ON  invoice_rel.productid = cross_rel.productid
                    AND invoice_rel.sequence_no = cross_rel.target_sequence_no
                    AND invoice_rel.id = cross_rel.target_crmid
                INNER JOIN vtiger_crmentity
                    ON invoice_rel.productid=vtiger_crmentity.crmid
                LEFT JOIN vtiger_products
                    ON vtiger_products.productid=invoice_rel.productid                                
                LEFT JOIN vtiger_service
                    ON vtiger_service.serviceid=invoice_rel.productid
                WHERE vtiger_crmentity.deleted = 0
                  AND cross_rel.target_crmid = ?
                  AND cross_rel.source_crmid = ?
                GROUP BY invoice_rel.productid 
                ORDER BY invoice_rel.sequence_no";
        $params = array($this->relatedInvoiceId, $this->parentId);
        $result = $adb->pquery($sql, $params);

        if ($adb->num_rows($result) > 0) {
            while ($row = $adb->fetchByAssoc($result)) {
                $parentItem = $this->getItem($row["productid"]);
                if ($parentItem === null) {
                    $newItem = new ITS4YouWarehouses_Item_ActionBlock($row["productid"], $row["productname"], $this->relatedInvoiceId, 0, $row["available_qty"], ITS4YouWarehouses_ItemInfo_ActionBlock::initialize($row), $row["setype"]);
                    if ($this->addItem($newItem)) {
                        $parentItem = $newItem;
                    }
                }
                if ($parentItem !== null) {
                    /* @var $item ITS4YouWarehouses_Item_ActionBlock */
                    $parentItem->modifyExplicitStatus(ITS4YouWarehouses_ItemStatus_ActionBlock::STATUS_INVOICED, $row["qty"]);

//decrease to-be-invoiced qty
//In future realeases quantity difference (to-be-invoiced qty - invoiced qty) will be used, but for project simple DeliveryNotes4You it's forced to unset to-be-invoiced status
//by setting itemQty = 0. Special case is when there is DN in state Invoiced (see $this->createDeliveryNoteItems) but there is no related
//Invoice, then to-be-invoiced status remains set - thus we avoid situation of having none displayable status (see ITS4YouWarehouses_Item_ActionBlock::getDisplayStatus())
//forced unsetting to-be-invoiced status
                    $toBeInvoicedQty = 0;

//quantity difference (to-be-invoiced qty - invoiced qty)
//$toBeInvoicedQty = $this->items[$row["productid"]]->getStatusQty(ITS4YouWarehouses_ItemStatus_ActionBlock::STATUS_TOBEINVOICED) - $row["qty"];

                    $parentItem->modifyExplicitStatus(ITS4YouWarehouses_ItemStatus_ActionBlock::STATUS_TOBEINVOICED, $toBeInvoicedQty);
                }
            }
        }
    }

    /**
     * Method for returning initial DN status based on inventory products and comparison
     * requested quantity vs. quantity in stock
     * @param Vtiger_Request $request
     * @param bool $useAllLineItems
     * @return string
     * @throws AppException
     * @throws Exception
     */
    public static function getInitialStatus(Vtiger_Request $request, $useAllLineItems = false)
    {
        $initialStatuses = self::$initialStatuses;
        if ($request->has("totalProductCount") == false) {
            throw new Exception("MultiWarehouses4You :: Incorrect usage of method ITS4YouWarehouses_InventoryUtils_Helper::getInitialStatus");
        }

        $totalProductCount = $request->get("totalProductCount");
        $status = $initialStatuses[2];
        //cumulated quantities because of situation, i.e. qtyInStock = 4 and we have two times same product with qty = 3
        $globalQuantities = array();
        $warehouseQuantities = array();
        $globalQuantitiesInStock = array();
        $warehouseQuantitiesInStock = array();
        for ($productIdx = 1; $productIdx <= $totalProductCount; $productIdx++) {
            //do not take Services into consideration
            if ($request->get("lineItemType" . $productIdx) == "Services") {
                continue;
            }

            $totalDistributionCount = $request->get("totalDistributionCount" . $productIdx);
            $productId = $request->get("hdnProductId" . $productIdx);
            for ($distributionIdx = 1; $distributionIdx <= $totalDistributionCount; $distributionIdx++) {
                $sufix = $productIdx . "_" . $distributionIdx;

                if ($request->has("hdnWarehouseId" . $sufix) && $request->get("hdnWarehouseId" . $sufix)) {
                    $warehouseId = $request->get("hdnWarehouseId" . $sufix);
                    //only newly added line items (which are line items without information about related issuecardid) should be taken into account
                    //because otherwise we would sum up quantities which have been already issued
                    if (!$useAllLineItems && $request->get("hdnIssuecardId" . $sufix) != "") {
                        continue;
                    }

                    if (!isset($warehouseQuantities[$warehouseId][$productId])) {
                        $warehouseQuantities[$warehouseId][$productId] = $request->get("qty" . $sufix);
                    } else {
                        $warehouseQuantities[$warehouseId][$productId] += $request->get("qty" . $sufix);
                    }
                    $qty = $warehouseQuantities[$warehouseId][$productId];

                    //performance
                    if (!isset($warehouseQuantitiesInStock[$warehouseId][$productId])) {
                        /** @var $warehousesModel ITS4YouWarehouses_Record_Model */
                        $warehousesModel = Vtiger_Record_Model::getInstanceById($warehouseId, 'ITS4YouWarehouses');
                        $warehouseQuantitiesInStock[$warehouseId][$productId] = $warehousesModel->getProductQtyInStock($productId);
                    }
                    $qtyInStock = $warehouseQuantitiesInStock[$warehouseId][$productId];
                } else {
                    if (!isset($globalQuantities[$productId])) {
                        $globalQuantities[$productId] = $request->get("qty" . $sufix);
                    } else {
                        $globalQuantities[$productId] += $request->get("qty" . $sufix);
                    }
                    $qty = $globalQuantities[$productId];

                    //performance
                    if (!isset($globalQuantitiesInStock[$productId])) {
                        $globalQuantitiesInStock[$productId] = Vtiger_Record_Model::getInstanceById($productId)->get("qtyinstock");
                    }
                    $qtyInStock = $globalQuantitiesInStock[$productId];
                }

                if ($qtyInStock < $qty || $qtyInStock <= 0) {
                    $status = $initialStatuses[1];
                    break 2;
                }
            }
        }

        return $status;
    }

    /**
     * @return array
     * @throws Exception
     */
    public function getSeqStatusViewModel()
    {
        $actions = array();
        $moduleName = "ITS4YouWHDeliveryNotes";
        $predecessors1 = array("Waiting Availability");
        if (in_array($this->deliveryNoteStatus, $predecessors1)) {
            $actions[] = array(
                "url" => "javascript: void(0);",
                "label" => vtranslate("LBL_CHECK_AVAILABILITY", $moduleName),
                "class" => "btn-success action" . ITS4YouWarehouses_DNStatusPool_Sequential::ACTION_MODE_READY_TO_DELIVER
            );
        } elseif ($this->deliveryNoteStatus == "Ready to Deliver") {
            $actions[] = array(
                "url" => "javascript: void(0);",
                "label" => vtranslate("LBL_DELIVER_CUSTOMER", $moduleName),
                "class" => "btn-success action" . ITS4YouWarehouses_DNStatusPool_Sequential::ACTION_MODE_DELIVER
            );
        } elseif ($this->deliveryNoteStatus == "Delivered") {
            $actions[] = array(
                "url" => "index.php?module=ITS4YouWHDeliveryNotes&view=SeqStatusActionHandler&mode=changeStatus&sourceModule=ITS4YouWHDeliveryNotes&sourceRecord=" . $this->parentId . "&record=" . $this->parentId . "&statusMode=" . ITS4YouWarehouses_DNStatusPool_Sequential::ACTION_MODE_INVOICE,
                "label" => vtranslate("LBL_CREATE_INV", $moduleName),
                "class" => "btn-success action" . ITS4YouWarehouses_DNStatusPool_Sequential::ACTION_MODE_INVOICE
            );

            if ($this->isReturnable()) {
                $actions[] = array(
                    "url" => "javascript: ITS4YouUtils_Js.returnProducts(\'" . $moduleName . "\', \'" . $this->parentId . "\');",
                    "label" => vtranslate("LBL_RETURN_PRODUCTS", $moduleName),
                    "class" => "btn-info action"
                );
            }
        }

        if ($this->deliveryNoteStatus != "Canceled") {
            $actions[] = array(
                "url" => "javascript: void(0);",
                "label" => vtranslate("LBL_CANCEL_DN", $moduleName),
                "class" => "btn-warning action" . ITS4YouWarehouses_DNStatusPool_Sequential::ACTION_MODE_CANCEL
            );
        }

        $seqStatusInstance = ITS4YouWarehouses_DNStatusPool_Sequential::getInstance()->getFirst();
        /* @var $seqStatusInstance ITS4YouWarehouses_IStatus_Sequential */
        $sequentialStatuses = array();
        while ($seqStatusInstance != null) {
            if ($this->deliveryNoteStatus != "Canceled") {
                if ($seqStatusInstance->getId() != "Canceled") {
                    $sequentialStatuses[$seqStatusInstance->getId()] = $this->deliveryNoteStatus == $seqStatusInstance->getId();
                }
            } else {
                $sequentialStatuses[$seqStatusInstance->getId()] = $this->deliveryNoteStatus == $seqStatusInstance->getId();
                if ($seqStatusInstance->getId() == $this->getPreCanceledStatus()) {
                    $sequentialStatuses["Canceled"] = true;
                    break;
                }
            }
            $seqStatusInstance = $seqStatusInstance->getNext();
        }

        return array(
            "actions" => $actions,
            "sequential_statuses" => $sequentialStatuses
        );
    }

    /**
     * @return bool
     */
    protected function isReturnable()
    {
        foreach ($this->lineItems as $lineItem) {
            /* @var $lineItem ITS4YouWarehouses_InventoryLineItem_Helper */
            //do not take Services into considaretion in Return products process
            if ($this->getItem($lineItem->getProductId()) != null && $this->getItem($lineItem->getProductId())->getSEType() == "Services") {
                continue;
            }
            return true;
        }

        return false;
    }

    /**
     * @return mixed|null
     * @throws Exception
     */
    protected function getPreCanceledStatus()
    {
        global $adb;
        if (!file_exists("modules/ModTracker/ModTracker.php")) {
            return null;
        }
        require_once "modules/ModTracker/ModTracker.php";
// modifiedtime could be better for performance but Edit must be disabled once the DN is canceled
        $sql = "SELECT createdtime FROM vtiger_crmentity WHERE crmid = ?";
        $result = $adb->pquery($sql, array($this->parentId));
        $time = $adb->query_result($result, 0, "createdtime");
        $fieldChanges = ModTracker::getRecordFieldChanges($this->parentId, strtotime($time));

        if (isset($fieldChanges["deliverynotestatus"]) && $fieldChanges["deliverynotestatus"]["postvalue"] == "Canceled") {
            return $fieldChanges["deliverynotestatus"]["prevalue"];
        } else {
            return null;
        }
    }

    /**
     * @param $relatedProducts
     * @return mixed
     * @throws Exception
     */
    public function enhanceRelProductsWithDistribution($relatedProducts)
    {
        switch ($this->deliveryNoteStatus) {
            case "Canceled":
                $relatedIssuecards = $this->getRelatedIssueCards(array("Canceled"));
                break;

            default:
                $relatedIssuecards = $this->getRelatedIssueCards(array("Approved", "Transferred from Warehouse", "Delivered"));
        }

        $productsCount = ITS4YouWarehouses_Utils_Helper::count($relatedProducts);
        //get source lineItems
        $lineItems = $this->getLineItems();

        //add lineItems of related ICs to source lineItems for non merged items, which contains record
        //in cross inventory table
        if (ITS4YouWarehouses_Utils_Helper::count($relatedIssuecards) > 0) {
            CrossInventoryRelEventHandler::addCrossInventoryRels($lineItems, array_keys($relatedIssuecards));
        }
        //for merged or manually added items, that is items without any record in cross inventory table
        //we need to get lineItems of all related ICs
        $icAllLineItems = array();
        foreach ($relatedIssuecards as $issuecardId => $issueCardName) {
            $icActionBlock = ITS4YouWarehouses_Base_ActionBlock::factory("ITS4YouIssuecards", $issuecardId);
            /* @var $icActionBlock ITS4YouWarehouses_IC_ActionBlock */
            $icAllLineItems = (array)$icAllLineItems + (array)$icActionBlock->getLineItems();
        }
        //process related products and add distribution data
        for ($productIdx = 1; $productIdx <= $productsCount; $productIdx++) {
            $lineItemId = $relatedProducts[$productIdx]["lineItemId" . $productIdx];
            if (!isset($lineItems[$lineItemId])) {
                continue;
            }
            $lineItem = $lineItems[$lineItemId];
            /* @var $lineItem ITS4YouWarehouses_InventoryLineItem_Helper */
            $icCrossLineItems = $lineItem->getCrossInventoryLineItems();
            $icCrossLineItemsCount = ITS4YouWarehouses_Utils_Helper::count($icCrossLineItems);

            if ($icCrossLineItemsCount > 0) {
                //handle cross inventory rel items
                $totalDistributedQty = 0;
                foreach ($icCrossLineItems as $icCrossLineItemId => $icCrossLineItem) {
                    /* @var $icCrossLineItem ITS4YouWarehouses_InventoryLineItem_Helper */

                    $relatedProducts = $this->adjustRelProducts($relatedProducts, $productIdx, $icCrossLineItem);
                    $totalDistributedQty += $icCrossLineItem->getQuantity();
                    //it's needed to adjust lineItems of all related ICs ($icAllLineItems) in order to keep the quantity remainder
                    //for non cross items
                    if (isset($icAllLineItems[$icCrossLineItemId])) {
                        unset($icAllLineItems[$icCrossLineItemId]);
                    }
                }
                $relatedProducts[$productIdx]["totalDistributedQty" . $productIdx] = ITS4YouWarehouses_Base_ActionBlock::formatNumberForDetailView($totalDistributedQty);
            } else {
                //handle non-cross inventory rel items, this branch can occur only in exceptional cases, in standard process there is cross inventory rel
                //between DN and IC
                //this branch can also occur when lineItem is Service
                //find first lineItem based on current iteration's hdnProductId
                $productId = $relatedProducts[$productIdx]["hdnProductId" . $productIdx];
                $productQuantity = $relatedProducts[$productIdx]["qty" . $productIdx];
                $isProductHandled = false;
                $totalDistributedQty = 0;

                if ($relatedProducts[$productIdx]["entityType" . $productIdx] == "Services") {
                    $totalDistributedQty = $productQuantity;
                    $isProductHandled = true;
                } else {
                    foreach ($icAllLineItems as $icAllLineItem) {
                        /* @var $icAllLineItem ITS4YouWarehouses_InventoryLineItem_Helper */
                        $lineItemQty = $icAllLineItem->getQuantity();
                        if ($icAllLineItem->getProductId() == $productId && $lineItemQty > 0 /* && $productQuantity >= $icAllLineItem->getQuantity() */) {
                            if ($lineItemQty > $productQuantity) {
                                //set qty for the adjustRelProducts method pruposes, in order to display max DN qty level
                                $icAllLineItem->setQuantity($productQuantity);
                                $relatedProducts = $this->adjustRelProducts($relatedProducts, $productIdx, $icAllLineItem);
                                //keep items separated as they are at DN, even if items at IC are grouped because of the same warehouse
                                //(they are by default, when merge was selected or item was manually added in AvailableSave)
                                $newLineItemQty = $lineItemQty - $productQuantity;
                                $totalDistributedQty += $productQuantity;
                                $icAllLineItem->setQuantity($newLineItemQty);
                                $productQuantity = 0;
                                $isProductHandled = true;
                            } else {
                                $relatedProducts = $this->adjustRelProducts($relatedProducts, $productIdx, $icAllLineItem);
                                $productQuantity -= $lineItemQty;
                                $totalDistributedQty += $lineItemQty;
                                $icAllLineItem->setQuantity(0);
                                $isProductHandled = true;
                            }
                        }

                        if ($productQuantity <= 0) {
                            break;
                        }
                    }
                }

                if ($isProductHandled) {
                    $relatedProducts[$productIdx]["totalDistributedQty" . $productIdx] = ITS4YouWarehouses_Base_ActionBlock::formatNumberForDetailView($totalDistributedQty);
                    $relatedProducts[$productIdx]["totalDistributedQtyForEdit" . $productIdx] = ITS4YouWarehouses_Base_ActionBlock::formatNumberForEditView($totalDistributedQty);
                } else {
                    //handle possible cancelation of IC, which results in that there are more line items in vtiger_inventoryproductrel table for DN
                    //than in suitable related ICs
                    unset($relatedProducts[$productIdx]);
                }
            }
        }

        return $relatedProducts;
    }

    /**
     * @param $relatedProducts
     * @param $productIdx
     * @param $icLineItem
     * @return mixed
     * @throws AppException
     */
    protected function adjustRelProducts($relatedProducts, $productIdx, $icLineItem)
    {
        static $icRecordModels = array();
        static $whRecordModels = array();

        $c_warehouseNA = "-";

        //IC record model pool
        if (!isset($icRecordModels[$icLineItem->getParentId()])) {
            $icRecordModels[$icLineItem->getParentId()] = Vtiger_Record_Model::getInstanceById($icLineItem->getParentId());
        }
        $icRecordModel = $icRecordModels[$icLineItem->getParentId()];
        /* @var $icRecordModel ITS4YouIssuecards_Record_Model */
        $warehouseId = (int)$icRecordModel->get('warehouseid');
        //WH record model pool
        if (!isset($whRecordModels[$warehouseId]) && 0 < $warehouseId) {
            $whRecordModels[$warehouseId] = Vtiger_Record_Model::getInstanceById($warehouseId);
        }

        /* @var $whRecordModel ITS4YouWarehouses_Record_Model */
        if (0 < $warehouseId) {
            $whRecordModel = $whRecordModels[$warehouseId];
        } else {
            $whRecordModel = null;
        }

        $relatedProducts[$productIdx]["distribution" . $productIdx][$warehouseId]["qty"] = $icLineItem->getQuantity();
        $relatedProducts[$productIdx]["distribution" . $productIdx][$warehouseId]["hdnIssuecardId"] = $icRecordModel->getId();
        $relatedProducts[$productIdx]["distribution" . $productIdx][$warehouseId]["issuecardName"] = $icRecordModel->getName();
        $relatedProducts[$productIdx]["distribution" . $productIdx][$warehouseId]["issuecardurl"] = $icRecordModel->getDetailViewUrl();
        if ($whRecordModel !== null) {
            $relatedProducts[$productIdx]["distribution" . $productIdx][$warehouseId]["hdnWarehouseId"] = $whRecordModel->getId();
            $relatedProducts[$productIdx]["distribution" . $productIdx][$warehouseId]["warehouseName"] = $whRecordModel->getName();
            $productQtyInStock = $whRecordModel->getProductQtyInStock($icLineItem->getProductId());
            $relatedProducts[$productIdx]["distribution" . $productIdx][$warehouseId]["qtyInStockRaw"] = $productQtyInStock;
            $relatedProducts[$productIdx]["distribution" . $productIdx][$warehouseId]["qtyInStock"] = ITS4YouWarehouses_Base_ActionBlock::formatNumberForDetailView($productQtyInStock);
        } else {
            $relatedProducts[$productIdx]["distribution" . $productIdx][$warehouseId]["hdnWarehouseId"] = "";
            $relatedProducts[$productIdx]["distribution" . $productIdx][$warehouseId]["warehouseName"] = $c_warehouseNA;
            $relatedProducts[$productIdx]["distribution" . $productIdx][$warehouseId]["qtyInStockRaw"] = $c_warehouseNA;
            $relatedProducts[$productIdx]["distribution" . $productIdx][$warehouseId]["qtyInStock"] = $c_warehouseNA;
        }

        return $relatedProducts;
    }

    /**
     * @return array
     */
    public function getChangeStatusDeliverViewModel()
    {
        //In order to avoid incosistencies when user changes quantity via this action, since there are IC at background related
//        $disabled = "false";
//        if ($this->relatedInvoiceId > 0) {
        $disabled = "true";
//        }

        $items = array();
        $parentItems = $this->getItems();
        foreach ($parentItems as $item) {
            /* @var $item ITS4YouWarehouses_Item_ActionBlock */
            if ($item->getParentId() != $this->parentId) {
                continue;
            }

            $itemId = $item->getId();
            $items[$itemId]["name"] = $item->getName();
            $items[$itemId]["parent_qty"] = self::formatNumberForEditView($item->getStatusQty(ITS4YouWarehouses_ItemStatus_ActionBlock::STATUS_ISSUED));
            $items[$itemId]["disabled"] = $disabled;
        }

        return array(
            "items" => $items,
            "title" => vtranslate("LBL_DELIVER_CUSTOMER", "ITS4YouWHDeliveryNotes"),
            "qty_col_title" => vtranslate("HEAD_DELIVER_QTY", "ITS4YouWHDeliveryNotes")
        );
    }

    /**
     * @return array
     */
    public function getReturnProductsViewModel()
    {
        return [
            'items' => $this->getReturnProductsLineItems(),
            'isReturnable' => $this->isReturnable,
            'target_module' => 'ITS4YouReceiptcards',
        ];
    }

    /**
     * @return string
     * @throws Exception
     */
    public function checkInitialStatus()
    {
        $initialStatuses = self::$initialStatuses;
        $groupedItems = array();
        $status = $initialStatuses[2];
        $relatedIssuecards = $this->getRelatedIssueCards(array("Approved"));

        foreach ($relatedIssuecards as $issuecardId => $issuecardNo) {
            $ICActionBlock = ITS4YouWarehouses_Base_ActionBlock::factory("ITS4YouIssuecards", $issuecardId);
            $ICActionBlockItems = $ICActionBlock->getItems();
            foreach ($ICActionBlockItems as $itemId => $item) {
                $warehouseId = $ICActionBlock->getWarehouseId();
                /* @var $item ITS4YouWarehouses_Item_ActionBlock */
                if (!isset($groupedItems[$warehouseId][$itemId])) {
                    $groupedItems[$warehouseId][$itemId]["qty"] = $item->getParentQty();
                    $groupedItems[$warehouseId][$itemId]["qtyInStock"] = $item->getAvailableQty();
                } else {
                    $groupedItems[$warehouseId][$itemId]["qty"] += $item->getParentQty();
                }
            }
        }

        foreach ($groupedItems as $warehouseId => $warehouseData) {
            foreach ($warehouseData as $itemId => $itemData) {
                if ($itemData["qty"] > $itemData["qtyInStock"]) {
                    $status = $initialStatuses[1];
                    break 2;
                }
            }
        }

        return $status;
    }

    /**
     *
     */
    protected function initStatusActionsPool()
    {
        $this->statusActionsPool = ITS4YouWarehouses_DNStatusActionsPool_ActionBlock::getInstance();
    }

    /**
     * Overriden method for creating parent items
     * @throws Exception
     * @global PearDatabase $adb
     */
    protected function createParentItems()
    {
        global $adb;

        $this->readParrentAttributes();

        $sql = "SELECT  CASE WHEN vtiger_products.productid != '' THEN vtiger_products.productname ELSE vtiger_service.servicename END AS 'productname',
                        CASE WHEN vtiger_products.productid != '' THEN vtiger_products.qtyinstock ELSE '0' END AS 'available_qty',
                        " . ITS4YouWarehouses_ItemInfo_ActionBlock::getItemInfoQuery() . ",
                        delnote_rel.productid,
                        SUM(delnote_rel.quantity) AS deliver_qty,
                        vtiger_crmentity.setype                        
                FROM vtiger_inventoryproductrel AS delnote_rel
                INNER JOIN vtiger_crmentity
                    ON vtiger_crmentity.crmid=delnote_rel.productid                    
                LEFT JOIN vtiger_products
                    ON vtiger_products.productid=delnote_rel.productid                                
                LEFT JOIN vtiger_service
                    ON vtiger_service.serviceid=delnote_rel.productid                    
                WHERE vtiger_crmentity.deleted = 0
                    AND delnote_rel.id=?                 
                GROUP BY delnote_rel.productid 
                ORDER BY delnote_rel.sequence_no";
        $result = $adb->pquery($sql, array($this->parentId));
        if ($adb->num_rows($result) > 0) {
            while ($row = $adb->fetchByAssoc($result)) {
                $parentItem = new ITS4YouWarehouses_Item_ActionBlock($row["productid"], $row["productname"], $this->parentId, $row["deliver_qty"], $row["available_qty"], ITS4YouWarehouses_ItemInfo_ActionBlock::initialize($row), $row["setype"]);
                if ($this->addItem($parentItem)) {
                    switch ($this->deliveryNoteStatus) {
//this branch is handled within the method processRelatedIssueCards
//                    case "Delivered":
//                        $this->items[$row["productid"]]->modifyExplicitStatus(ITS4YouWarehouses_ItemStatus_ActionBlock::STATUS_DELIVERED, $row["deliver_qty"]);
//                        $this->items[$row["productid"]]->modifyExplicitStatus(ITS4YouWarehouses_ItemStatus_ActionBlock::STATUS_TOBEINVOICED, $row["deliver_qty"]);
//                        break;
//
                        case "Invoiced":
                            $parentItem->modifyExplicitStatus(ITS4YouWarehouses_ItemStatus_ActionBlock::STATUS_DELIVERED, $row["deliver_qty"]);
//increase to-be-invoiced qty, it is decreased by invoiced qty in processRelatedInvoice
                            $parentItem->modifyExplicitStatus(ITS4YouWarehouses_ItemStatus_ActionBlock::STATUS_TOBEINVOICED, $row["deliver_qty"]);
                            break;

//this branch is handled within the method processRelatedIssueCards
//                    case "Ready to Deliver":
//                        $this->items[$row["productid"]]->modifyExplicitStatus(ITS4YouWarehouses_ItemStatus_ActionBlock::STATUS_ISSUED, $row["deliver_qty"]);
//                        break;

                        case "Canceled":
                            $parentItem->modifyExplicitStatus(ITS4YouWarehouses_ItemStatus_ActionBlock::STATUS_CANCELED, $row["deliver_qty"]);
                            break;
                    }
                }
            }
        }
    }

    /*
     * Overrided version of the method for purpose of non-deterministic price determination
     */

    /**
     * @throws Exception
     */
    protected function readParrentAttributes()
    {
        global $adb;

        $sql = "SELECT  deliverynotestatus, 
                        invoice_id,
                        referenceid
                FROM its4you_deliverynotes 
                WHERE deliverynoteid=?";
        $result = $adb->pquery($sql, array($this->parentId));
        if ($adb->num_rows($result) > 0) {
            $this->deliveryNoteStatus = $adb->query_result($result, 0, "deliverynotestatus");
            $this->relatedInvoiceId = $adb->query_result($result, 0, "invoice_id");
            $this->salesOrderId = $adb->query_result($result, 0, "referenceid");
        }
    }

}