<?php
/* * *******************************************************************************
 * The content of this file is subject to the MultiWarehouses4You license.
 * ("License"); You may not use this file except in compliance with the License
 * The Initial Developer of the Original Code is IT-Solutions4You s.r.o.
 * Portions created by IT-Solutions4You s.r.o. are Copyright(C) IT-Solutions4You s.r.o.
 * All Rights Reserved.
 * ****************************************************************************** */

class ITS4YouWHDeliveryNotes_Save_Action extends Inventory_Save_Action
{

    /**
     * @param Vtiger_Request $request
     * @throws Exception
     */
    public function process(Vtiger_Request $request)
    {
        $recordModel = $this->getRecordModelFromRequest($request);
        $existingStatus = $recordModel->get('deliverynotestatus');

        //set distribution
        $distribution = array();
        $totalProductCount = $request->get('totalProductCount');
        $stockQuantities = array();

        //never merge products because we need to keep reference at items level between DN and IC
        //if the $products are merged then they'll come already merged from AvailableSave
        //$mergeProducts = false;

        for ($productIdx = 1; $productIdx <= $totalProductCount; $productIdx++) {
            if ('Services' === $request->get('lineItemType' . $productIdx)) {
                continue;
            }

            $totalDistributionCount = $request->get('totalDistributionCount' . $productIdx);
            $productId = $request->get('hdnProductId' . $productIdx);

            for ($distributionIdx = 1; $distributionIdx <= $totalDistributionCount; $distributionIdx++) {
                $sufix = $productIdx . "_" . $distributionIdx;
                $warehouseId = $request->get('hdnWarehouseId' . $sufix);
                $quantity = $request->get('qty' . $sufix);

                if (!isset($stockQuantities[$warehouseId][$productId])) {
                    $stockQuantities[$warehouseId][$productId] = Vtiger_Record_Model::getInstanceById($warehouseId, 'ITS4YouWarehouses')->getProductQtyInStock($productId);
                }

                if (empty($request->get('hdnIssuecardId' . $sufix))) {
                    //if products are not merged, then we will group quantities per unique pair - warehouseId x productIdx
                    if (!isset($distribution[$warehouseId][$productIdx])) {
                        $distribution[$warehouseId][$productIdx] = $quantity;
                    } else {
                        $distribution[$warehouseId][$productIdx] += $quantity;
                    }
                }
            }
        }

        //only if there are any newly added line items
        if (0 < count($distribution) && 'Canceled' !== $existingStatus) {
            $initialStatus = ITS4YouWarehouses_DN_ActionBlock::getInitialStatus($request);

            if (in_array($existingStatus, ['Ready to Deliver', 'Created']) && 'Waiting Availability' === $initialStatus) {
                //this branch is in edit mode of already existing DN, because during creation of new DN the $existingStatus and $initialStatus
                //are the same
                //analyse quantities and create duplicate DN in status Waiting Availability with the products and their
                //shortage quantities, if there is not enough quantity at stock
                //duplicate is created because the original DN is already in status Ready to Deliver (Created
                //is currently not used - see ITS4YouWarehouses_DN_ActionBlock::getInitialStatus)

                if (empty($request->get('record'))) {
                    throw new Exception('MultiWarehouses4You :: Unexpected situation occured.');
                }

                $newLineItems = array();
                $newDistribution = array();
                $newProductIdx = 1;

                for ($productIdx = 1; $productIdx <= $totalProductCount; $productIdx++) {
                    $productId = $request->get('hdnProductId' . $productIdx);
                    $totalDistributionCount = $request->get('totalDistributionCount' . $productIdx);
                    $totalNewlyDistributedQuantity = 0;

                    for ($distributionIdx = 1; $distributionIdx <= $totalDistributionCount; $distributionIdx++) {
                        $sufix = $productIdx . '_' . $distributionIdx;
                        //items with existing IC we skip
                        if (!empty($request->get('hdnIssuecardId' . $sufix))) {
                            continue;
                        }
                        $quantity = $request->get('qty' . $sufix);
                        $warehouseId = $request->get('hdnWarehouseId' . $sufix);
                        $quantityInStock = $stockQuantities[$warehouseId][$productId];

                        if ($quantity > $quantityInStock) {
                            $quantityDiff = $quantity - $quantityInStock;
                            $totalNewlyDistributedQuantity += $quantityDiff;

                            if ($quantityInStock > 0) {
                                //set quantityInStock as quantity at original DN because it's status Ready to Deliver
                                $request->setGlobal('qty' . $sufix, $quantityInStock);
                            } else {
                                //else set quantity for distributionItem to 0 for request for original DN
                                $request->setGlobal('qty' . $sufix, 0);
                            }
                            //adjust quantity distribution because we have modified original DN quantity
                            $distribution[$warehouseId][$productIdx] -= $quantityDiff;
                            //fill distribution for ICs of new DN

                            if (!isset($newDistribution[$warehouseId][$newProductIdx])) {
                                $newDistribution[$warehouseId][$newProductIdx] = $quantityDiff;
                            } else {
                                $newDistribution[$warehouseId][$newProductIdx] += $quantityDiff;
                            }
                            $stockQuantities[$warehouseId][$productId] = 0;
                        } else {
                            //decrease qauntity in $stockQuantities
                            $stockQuantities[$warehouseId][$productId] -= $quantity;
                        }
                    }

                    //store the lineItems with all the item details for new distribution
                    //$newLineItems and $newDistribution together create complete information for items of new DN and new ICs
                    if ($totalNewlyDistributedQuantity > 0) {
                        $newLineItems[$newProductIdx] = new ITS4YouWarehouses_InventoryLineItem_Helper(0, $productId, $totalNewlyDistributedQuantity, 0, $request->get("comment" . $productIdx));
                        $newProductIdx++;
                    }
                }
                $newDNStatus = 'Ready to Deliver';
                $newICStatus = 'Transferred from Warehouse';
                //in status Created there might be a situation when user uses products with 0 stock quantity
                //then we need to swap a logic and create original DN in status Waiting Availability

                if ('Created' === $existingStatus && !$this->isDistributionValid($distribution)) {
                    //this branch ought not occur since initial $existingStatus is at least Waiting availability
                    $newDNStatus = 'Waiting Availability';
                    $newICStatus = 'Approved';

                    unset($_REQUEST);
                    ITS4YouWarehouses_InventoryUtils_Helper::prepareRequestForAssociatedLineItems($newLineItems);
                    //save original DN
                    $request->set('deliverynotestatus', $newDNStatus);
                    list ($recordModel, $loadUrl) = $this->saveSource($request);
                    //create ICs for original DN
                    $this->createIssuecards($newDistribution, $recordModel, $newICStatus);
                } else {
                    //save original DN
                    $request->set('deliverynotestatus', $newDNStatus);
                    list ($recordModel, $loadUrl) = $this->saveSource($request);
                    //create ICs for original DN
                    $this->createIssuecards($distribution, $recordModel, $newICStatus);

                    if ($this->isDistributionValid($newDistribution)) {
                        //create new DN in status Waiting Availability
                        $newRecordModel = $this->createDuplicate($recordModel, $newLineItems, 'Waiting Availability');
                        //create ICs for new DN in status Approved
                        $this->createIssuecards($newDistribution, $newRecordModel, 'Approved');
                        //add information to user about new DN
                        #displayNotifyMsg
                        $loadUrl .= '#ALERT_NEW_AWAITING_DN_CREATED1|' . $newRecordModel->getName() . '|ALERT_NEW_AWAITING_DN_CREATED2';
                    }
                }
            } elseif (in_array($existingStatus, ['Delivered', 'Invoiced'])) {
                $newLineItems = array();
                $newProductIdx = 1;
                //we will use $newDistribution since we need to keep correct indexing ($newProductIdx),
                //because it must match with $newLineItems array
                $newDistribution = array();
                //save original DN without newly added line items
                for ($productIdx = 1; $productIdx <= $totalProductCount; $productIdx++) {
                    $productId = $request->get('hdnProductId' . $productIdx);

                    $totalDistributionCount = $request->get('totalDistributionCount' . $productIdx);
                    $totalNewlyDistributedQuantity = 0;
                    for ($distributionIdx = 1; $distributionIdx <= $totalDistributionCount; $distributionIdx++) {
                        $sufix = $productIdx . '_' . $distributionIdx;
                        //items with existing IC we skip
                        if (!empty($request->get('hdnIssuecardId' . $sufix))) {
                            continue;
                        }

                        $warehouseId = $request->get('hdnWarehouseId' . $sufix);
                        $quantity = $request->get('qty' . $sufix);
                        $totalNewlyDistributedQuantity += $quantity;
                        //set quantity for distributionItem to 0 for request for original DN
                        $request->setGlobal('qty' . $sufix, 0);

                        //fill distribution for ICs of new DN
                        if (!isset($newDistribution[$warehouseId][$newProductIdx])) {
                            $newDistribution[$warehouseId][$newProductIdx] = $quantity;
                        } else {
                            $newDistribution[$warehouseId][$newProductIdx] += $quantity;
                        }
                    }
                    //store the lineItems with all the item details for new distribution
                    //$newLineItems and $newDistribution together create complete information for items of new DN and new ICs
                    if ($totalNewlyDistributedQuantity > 0) {
                        $newLineItems[$newProductIdx] = new ITS4YouWarehouses_InventoryLineItem_Helper(0, $productId, $totalNewlyDistributedQuantity, 0, $request->get('comment' . $productIdx));
                        $newProductIdx++;
                    }
                }

                $request->set('deliverynotestatus', $existingStatus);
                list ($recordModel, $loadUrl) = $this->saveSource($request);
                //do not create any additional ICs for original DN, instead create new DN and ICs for it

                if ($this->isDistributionValid($newDistribution)) {
                    //create new DN in status Waiting Availability
                    $newRecordModel = $this->createDuplicate($recordModel, $newLineItems, $initialStatus);
                    //create ICs for new DN in status Approved
                    $this->createIssuecards($newDistribution, $newRecordModel, $this->getICStatus($initialStatus));
                    //add information to user about new DN
                    #displayNotifyMsg
                    $loadUrl .= '#ALERT_NEW_AWAITING_DN_CREATED1|' . $newRecordModel->getName() . '|ALERT_NEW_AWAITING_DN_CREATED2';
                }
            } elseif ('Waiting Availability' === $existingStatus) {
                //save original DN
                $request->set('deliverynotestatus', $existingStatus);
                list($recordModel, $loadUrl) = $this->saveSource($request);

                //create ICs
                $this->createIssuecards($distribution, $recordModel, $this->getICStatus($existingStatus));
            } else {
                //save original DN
                $request->set('deliverynotestatus', $initialStatus);
                list($recordModel, $loadUrl) = $this->saveSource($request);

                //create ICs
                $this->createIssuecards($distribution, $recordModel, $this->getICStatus($initialStatus));
            }
        } else {
            if ('ITS4YouIssuecards' === $request->get('sourceModule')) {
                //if we are coming from CreateDNFromIC view -
                //assumption is that new IC are created in status Transferred from Warehouse therefore new DN is created in status Ready to Deliver
                $request->set('deliverynotestatus', 'Ready to Deliver');
                list($recordModel, $loadUrl) = $this->saveSource($request);
                //edit ICs, add DN to them
                $this->relateIssuecards($recordModel, $request->get('sourceRecord'));
            } else {
                list($recordModel, $loadUrl) = $this->saveSource($request);
            }
        }

        header("Location: $loadUrl");
    }

    /**
     * @param Vtiger_Request $request
     * @return Inventory_Record_Model|Vtiger_Record_Model
     * @throws Exception
     */
    protected function getRecordModelFromRequest(Vtiger_Request $request)
    {
        $recordModel = parent::getRecordModelFromRequest($request);
        /* @var $recordModel Inventory_Record_Model */

        if (empty($recordModel->get('mode')) && empty($recordModel->get('deliverynotestatus'))) {
            $recordModel->set('deliverynotestatus', ITS4YouWarehouses_DN_ActionBlock::getInitialStatus($request));
        }

        return $recordModel;
    }

    /**
     * @param $distribution
     * @return bool
     */
    private function isDistributionValid($distribution)
    {
        foreach ($distribution as $itemQuantities) {
            //check if productQuantities contains at least one product with qty > 0
            foreach ($itemQuantities as $qty) {
                if ($qty) {
                    return true;
                }
            }
        }

        return false;
    }

    /**
     * @param Vtiger_Request $request
     * @return array
     */
    private function saveSource(Vtiger_Request $request)
    {
        $recordModel = $this->saveRecord($request);

        if ($request->get('relationOperation')) {
            $parentModuleName = $request->get('sourceModule');
            $parentRecordId = $request->get('sourceRecord');

            if (!is_array($parentRecordId)) {
                $parentRecordModel = Vtiger_Record_Model::getInstanceById($parentRecordId, $parentModuleName);
                $loadUrl = $parentRecordModel->getDetailViewUrl();
            } elseif (count($parentRecordId) > 1) {
                $loadUrl = Vtiger_Module_Model::getInstance($parentModuleName)->getListViewUrl();
            } else {
                $parentRecordModel = Vtiger_Record_Model::getInstanceById(reset($parentRecordId), $parentModuleName);
                $loadUrl = $parentRecordModel->getDetailViewUrl();
            }
        } else {
            if ($request->get('returnToList')) {
                $loadUrl = $recordModel->getModule()->getListViewUrl();
            } else {
                $loadUrl = $recordModel->getDetailViewUrl();
            }
        }

        return array($recordModel, $loadUrl);
    }

    /**
     * @param $request
     * @return |Inventory_Record_Model|Vtiger_Record_Model
     * @throws Exception
     */
    public function saveRecord($request)
    {
        $recordModel = $this->getRecordModelFromRequest($request);

        if ($request->has('deliverynotestatus')) {
            $recordModel->set('deliverynotestatus', $request->get('deliverynotestatus'));
        }

        $recordModel->save();

        if ($request->get('relationOperation')) {
            if ('ITS4YouIssuecards' !== $request->get('sourceModule')) {
                $parentModuleName = $request->get('sourceModule');
                $parentModuleModel = Vtiger_Module_Model::getInstance($parentModuleName);
                $parentRecordId = $request->get('sourceRecord');
                $relatedModule = $recordModel->getModule();
                $relatedRecordId = $recordModel->getId();
                $relationModel = Vtiger_Relation_Model::getInstance($parentModuleModel, $relatedModule);
                $relationModel->addRelation($parentRecordId, $relatedRecordId);
            }
        }

        if ($request->get('imgDeleted')) {
            $imageIds = $request->get('imageid');

            foreach ($imageIds as $imageId) {
                $recordModel->deleteImage($imageId);
            }
        }

        return $recordModel;
    }

    /**
     *
     * @param array $distribution
     * @param Inventory_Record_Model $sourceRecordModel
     * @param string $status
     */
    private function createIssuecards($distribution, $sourceRecordModel, $status)
    {
        foreach ($distribution as $warehouseId => $items) {
            $lineItems = array();
            //prepare lineItems in order to unify processing of items as lineItems
            foreach ($items as $sequenceNo => $quantity) {
                $lineItem = ITS4YouWarehouses_InventoryLineItem_Helper::getLineItemFromDBBySequence($sourceRecordModel->getId(), $sequenceNo);

                if (null != $lineItem && 0 < $quantity) {
                    $lineItem->setQuantity($quantity);
                    /* @var $lineItem ITS4YouWarehouses_InventoryLineItem_Helper */
                    $lineItems[$lineItem->getItemId()] = $lineItem;
                }
            }
            //check if there is any lineItem, if not then skip creation for such warehouse (i.e. quantity for all lineItems was equal to 0 )
            if (empty($lineItems)) {
                continue;
            }

            $originalRequest = $_REQUEST;
            unset($_REQUEST);
            $icRecordModel = Vtiger_Record_Model::getCleanInstance('ITS4YouIssuecards');
            $icRecordModel->set('warehouseid', $warehouseId);
            $icRecordModel->set('assigned_user_id', $sourceRecordModel->get('assigned_user_id'));
            $issueDate = ($sourceRecordModel->get('issue_date') != '' ? $sourceRecordModel->get('issue_date') : DateTimeField::convertToDBFormat(getNewDisplayDate()));
            $icRecordModel->set('issue_date', $issueDate);
            $icRecordModel->set('parent_id', $sourceRecordModel->getId());
            $icRecordModel->set('issuecardstatus', $status);
            $icRecordModel->set('issuecard_carrier', $sourceRecordModel->get('deliverynote_carrier'));
            $icRecordModel->set('tracking_no', $sourceRecordModel->get('tracking_no'));
            $icRecordModel->set('issuecardtype', $sourceRecordModel->get('deliverynotetype'));
            $icRecordModel->set('currency_id', $sourceRecordModel->get('currency_id'));
            $icRecordModel->set('conversion_rate', $sourceRecordModel->get('conversion_rate'));
            ITS4YouWarehouses_InventoryUtils_Helper::prepareRequestForAssociatedLineItems($lineItems);

            $icRecordModel->save();
        }

        $_REQUEST = $originalRequest;
    }

    /**
     * @param ITS4YouWHDeliveryNotes_Record_Model $sourceRecordModel
     * @param $newLineItems
     * @param $status
     * @return Inventory_Record_Model|Vtiger_Record_Model
     */
    private function createDuplicate(ITS4YouWHDeliveryNotes_Record_Model $sourceRecordModel, $newLineItems, $status)
    {
        $moduleName = $sourceRecordModel->getModuleName();
        $recordModel = Inventory_Record_Model::getCleanInstance($moduleName);
        /* @var $recordModel Inventory_Record_Model */
        $recordModel->setRecordFieldValues($sourceRecordModel);
        $recordModel->set('deliverynotestatus', $status);

        $originalRequest = $_REQUEST;
        unset($_REQUEST);
        ITS4YouWarehouses_InventoryUtils_Helper::prepareRequestForAssociatedLineItems($newLineItems);
        $recordModel->save();
        $_REQUEST = $originalRequest;

        return $recordModel;
    }

    /**
     * @param $DNStatus
     * @return string
     */
    private function getICStatus($DNStatus)
    {
        return ($DNStatus == 'Ready to Deliver' ? 'Transferred from Warehouse' : 'Approved');
    }

    /**
     * @param Vtiger_Record_Model $dnRecordModel
     * @param $icRecordIds
     */
    private function relateIssuecards(Vtiger_Record_Model $dnRecordModel, $icRecordIds)
    {
        foreach ($icRecordIds as $icRecordId) {
            $icRecordModel = Vtiger_Record_Model::getInstanceById($icRecordId, 'ITS4YouIssuecards');
            $icRecordModel->set('mode', 'edit');
            $icRecordModel->set('parent_id', $dnRecordModel->getId());
            unset($_REQUEST['totalProductCount']);  //it's needed to force unsetting the totalProductCount in order to skip saving inventoryProducts for IC
            $icRecordModel->save();
        }
    }
}
