<?php

/* * *******************************************************************************
 * The content of this file is subject to the Delivery Notes 4 You license.
 * ("License"); You may not use this file except in compliance with the License
 * The Initial Developer of the Original Code is IT-Solutions4You s.r.o.
 * Portions created by IT-Solutions4You s.r.o. are Copyright(C) IT-Solutions4You s.r.o.
 * All Rights Reserved.
 * ****************************************************************************** */

include_once 'modules/Vtiger/CRMEntity.php';

class ITS4YouWHDeliveryNotes extends CRMEntity
{

    public static $statusFieldName = 'deliverynotestatus';
    public $log;
    public $db;
    public $table_name = 'its4you_deliverynotes';
    public $table_index = 'deliverynoteid';
    public $moduleName = 'ITS4YouWHDeliveryNotes';
    public $parentName = 'Inventory';
    public $related_tables = array(
        'its4you_deliverynoteshipads' => array(
            'deliverynoteid'
        ),
        'its4you_deliverynotescf' => array(
            'deliverynoteid'
        )
    );

    /**
     * Mandatory table for supporting custom fields.
     */
    public $customFieldTable = ['its4you_deliverynotescf', 'deliverynoteid'];

    /**
     * Mandatory for Saving, Include tables related to this module.
     */
    public $tab_name = ['vtiger_crmentity', 'its4you_deliverynotes', 'its4you_deliverynotescf', 'its4you_deliverynoteshipads', 'vtiger_inventoryproductrel'];

    /**
     * Mandatory for Saving, Include tablename and tablekey columnname here.
     */
    public $tab_name_index = [
        'vtiger_crmentity' => 'crmid',
        'its4you_deliverynotes' => 'deliverynoteid',
        'its4you_deliverynotescf' => 'deliverynoteid',
        'its4you_deliverynoteshipads' => 'deliverynoteid',
        'vtiger_inventoryproductrel' => 'id'
    ];

    /**
     * Mandatory for Listing (Related listview)
     */
    public $list_fields = [
        'Delivery Note No' => ['deliverynotes' => 'deliverynotes_no'],
        'Related To' => ['deliverynotes' => 'parent_id'],
        'Contact Name' => ['deliverynotes' => 'contact_id'],
        'Description' => ['crmentity' => 'description'],
        'Delivery Date' => ['deliverynotes' => 'issue_date'],
        'Status' => ['deliverynotes' => 'deliverynotestatus'],
        'Assigned To' => ['crmentity' => 'smownerid']
    ];
    public $list_fields_name = [
        /* Format: Field Label => fieldname */
        'Delivery Note No' => 'deliverynotes_no',
        'Related To' => 'parent_id',
        'Contact Name' => 'contact_id',
        'Description' => 'description',
        'Delivery Date' => 'issue_date',
        'Status' => 'deliverynotestatus',
        'Assigned To' => 'assigned_user_id'
    ];
    public $list_link_field = 'deliverynotes_no';
    public $search_fields = [
        'Delivery Note No' => ['deliverynotes', 'deliverynotes_no']
    ];
    public $search_fields_name = [
        'Delivery Note No' => 'deliverynotes_no'
    ];
    public $popup_fields = ['deliverynotes_no'];
    public $def_basicsearch_col = 'deliverynotes_no';
    public $def_detailview_recname = 'deliverynotes_no';
    public $mandatory_fields = ['createdtime', 'modifiedtime', 'deliverynotes_no'];
    public $default_order_by = 'deliverynotes_no';
    public $default_sort_order = 'ASC';
    public $isLineItemUpdate = true;
    public $additional_column_fields = ['assigned_user_name', 'smownerid', 'currency_id'];
    public $column_fields;
    public $relatedModules = ["SalesOrder", "Invoice", "Accounts", "Products"];

    /**
     * [module, type, label, url, icon, sequence, handlerInfo]
     * @return array
     */
    public $registerCustomLinks = array(
        ['ITS4YouWHDeliveryNotes', 'HEADERSCRIPT', 'ITS4YouWHDeliveryNotesSummaryWidgets', 'layouts/v7/modules/ITS4YouWHDeliveryNotes/resources/SummaryWidgets.js'],
        ['PurchaseOrder', 'DETAILVIEWSIDEBARWIDGET', 'WHActionBlockBase', 'module=ITS4YouWHDeliveryNotes&view=ActionBlockBase'],
    );

    /**
     * @var array
     * [module, related module, relate label, related actions, related function]
     * $moduleInstance->setRelatedList($relatedListModule, $relatedLabel, $relatedActions, $relatedFunction);
     */
    public $registerRelatedLists = [
        ['ITS4YouWHDeliveryNotes', 'ITS4YouIssuecards'],
        ['SalesOrder', 'ITS4YouWHDeliveryNotes'],
        ['Invoice', 'ITS4YouWHDeliveryNotes'],
        ['Accounts', 'ITS4YouWHDeliveryNotes'],
        ['Products', 'ITS4YouWHDeliveryNotes'],
        ['ITS4YouWHDeliveryNotes', 'ITS4YouReceiptcards'],
        ['ITS4YouWHDeliveryNotes', 'Documents', 'Documents', 'add,select', 'get_attachments'],
    ];

    public function __construct()
    {
        global $log;

        $this->log = $log;
        $this->db = PearDatabase::getInstance();
        $this->column_fields = getColumnFields('ITS4YouWHDeliveryNotes');
    }

    /**
     * function to add activate Inventory Workflow for Invoice Module
     * and deactivate Inventory Workflow for Delivery Notes Module
     *
     * @param $moduleName
     * @param $event_type
     *
     * @return bool
     */
    public static function deactivateWorkflowForDn($moduleName, $event_type)
    {
        self::changeWorkflowStatus(1, 0);

        return true;
    }

    /**
     * function to change Invoice and DeliveryNotes Inventory update workflow status
     *
     * @param $invoiceWfStatus
     * @param $deliveryNoteWfStatus
     *
     * @return bool
     */
    private static function changeWorkflowStatus($invoiceWfStatus, $deliveryNoteWfStatus)
    {
        $adb = PearDatabase::getInstance();

        if ($invoiceWfStatus == $deliveryNoteWfStatus) {
            die('Only one Inventory workflow should be enabled to prevent multiplication of warehouse update !');
        }

        $InvoiceId = 0;
        $ITS4YouWHDeliveryNotesWorkflowId = 0;

        $result = $adb->pquery('SELECT workflow_id, module_name 
                          FROM com_vtiger_workflows 
                          WHERE summary IN (?,?) 
                          AND module_name IN (?,?)',
            [
                'UpdateInventoryProducts On Every Save',
                'Update Inventory Products On Delivery',
                'Invoice',
                'ITS4YouWHDeliveryNotes',
            ]);
        if ($result) {
            while ($row = $adb->fetchByAssoc($result)) {
                $comWorkflowId = $row['workflow_id'];
                $comModuleName = $row['module_name'];
                if ('Invoice' === $comModuleName) {
                    Settings_Workflows_Record_Model::updateWorkflowStatus($comWorkflowId, $invoiceWfStatus);
                } else {
                    Settings_Workflows_Record_Model::updateWorkflowStatus($comWorkflowId, $deliveryNoteWfStatus);
                }
            }
        }

        return true;
    }

    /**
     * function to add activate Inventory Workflow for Delivery Notes Module
     * and deactivate Inventory Workflow for Invoice Module
     *
     * @param $moduleName
     * @param $event_type
     *
     * @return bool
     */
    public static function activateWorkflowForDn($moduleName, $event_type)
    {
        self::changeWorkflowStatus(0, 1);

        return true;
    }

    public function insertIntoEntityTable($table_name, $module, $fileid = '')
    {
        //Ignore relation table insertions while saving of the record
        if ($table_name == 'vtiger_inventoryproductrel') {
            return;
        }
        parent::insertIntoEntityTable($table_name, $module, $fileid);
    }

    public function save_module()
    {
        //in ajax save we should not call this function, because this will delete all the existing product values
        if (isset($_REQUEST['totalProductCount'])) {
            if ($_REQUEST['action'] != 'ITS4YouWHDeliveryNotesAjax' && $_REQUEST['ajxaction'] != 'DETAILVIEW' && $_REQUEST['action'] != 'MassEditSave' && $_REQUEST['action'] != 'ProcessDuplicates' && $_REQUEST['action'] != 'SaveAjax' && $this->isLineItemUpdate != false) {
                //Based on the total Number of rows we will save the product relationship with this entity
//                saveInventoryProductDetails($this, 'ITS4YouWHDeliveryNotes');
                ITS4YouWarehouses_InventoryUtils_Helper::saveDeliveryNotesInventoryProductDetails($this);
            }
        }
        $this->db->pquery('UPDATE its4you_deliverynotes SET deliverylaststatus = deliverynotestatus WHERE deliverynoteid=?', [$this->id]);

        //update status manually because of displaytype
        $status_update = "";
        if ($this->column_fields["deliverynotestatus"] != "") {
            $status_update = ", deliverynotestatus='" . $this->column_fields["deliverynotestatus"] . "'";
        }

        // Update the currency id and the conversion rate for the invoice
        $update_query = "update its4you_deliverynotes set currency_id=?, conversion_rate=?" . $status_update . " where deliverynoteid=?";
        $update_params = [$this->column_fields['currency_id'], $this->column_fields['conversion_rate'], $this->id];
        $this->db->pquery($update_query, $update_params);

        //save relationship to Accounts
        if ($this->column_fields["parent_id"] != "") {
            $parentModuleName = getSalesEntityType($this->column_fields["parent_id"]);
            $relationModel = Vtiger_Relation_Model::getInstance(Vtiger_Module_Model::getInstance($parentModuleName), Vtiger_Module_Model::getInstance("ITS4YouWHDeliveryNotes"));
            if ($relationModel) {
                $relationModel->addRelation($this->column_fields["parent_id"], $this->id);
            }
        }
        //save relationship to Sales Order
        if ($this->column_fields["referenceid"] > 0) {
            $parentModuleName = getSalesEntityType($this->column_fields["referenceid"]);
            $relationModel = Vtiger_Relation_Model::getInstance(Vtiger_Module_Model::getInstance($parentModuleName), Vtiger_Module_Model::getInstance("ITS4YouWHDeliveryNotes"));
            $relationModel->addRelation($this->column_fields["referenceid"], $this->id);
        }

        //save relationship to Invoice
        if ($this->column_fields["invoice_id"] > "0") {
            $parentModuleName = getSalesEntityType($this->column_fields["invoice_id"]);
            $relationModel = Vtiger_Relation_Model::getInstance(Vtiger_Module_Model::getInstance($parentModuleName), Vtiger_Module_Model::getInstance("ITS4YouWHDeliveryNotes"));
            if ($relationModel) {
                $relationModel->addRelation($this->column_fields["invoice_id"], $this->id);
            }
        }
    }

    public $eventType = '';

    public function vtlib_handler($moduleName, $eventType)
    {
        require_once 'vtlib/Vtiger/Module.php';
        require_once 'vtlib/Vtiger/Event.php';
        require_once 'modules/ModTracker/ModTracker.php';
        require_once 'modules/ModComments/ModComments.php';

        $this->eventType = $eventType;

        switch ($eventType) {
            case 'module.postinstall':
            case 'module.enabled':
            case 'module.postupdate':
                self::addWorkflowForDn($moduleName, $eventType);
                self::activateWorkflowForDn($moduleName, $eventType);
                $this->addCustomLinks();
                break;
            case 'module.preuninstall':
            case 'module.preupdate':
            case 'module.disabled':
                self::removeWorkflowForDn($moduleName, $eventType);
                self::deactivateWorkflowForDn($moduleName, $eventType);
                $this->deleteCustomLinks();
                break;
        }
    }

    public function clearCustomLinks()
    {
        $params = array('ITS4YouWHDeliveryNotesSummaryWidgets', 'ITS4YouSeqStatusHandler', 'ITS4YouUtils', 'ITS4YouProductMovements');
        $sql = sprintf('DELETE FROM vtiger_links WHERE linklabel IN (%s)', generateQuestionMarks($params));
        $this->db->pquery($sql, $params);
    }

    public function updateNumbering()
    {
        $this->setModuleSeqNumber('configure', $this->moduleName, 'DN', 1);
        $this->updateMissingSeqNumber($this->moduleName);
    }

    public function addCustomLinks()
    {
        self::runSelfFormUpdates();

        $this->updateNumbering();
        $this->clearCustomLinks();
        $this->updateToWarehouses();
        $this->updateTables();
        $this->deletePreferences('updateInventoryOnDelivery');
        $this->updateFields();
        $this->updateRelatedList();
        $this->updateCustomLinks();

        Settings_MenuEditor_Module_Model::addModuleToApp($this->moduleName, $this->parentName);

        ModComments::addWidgetTo([$this->moduleName]);
        ModTracker::enableTrackingForModule(getTabid($this->moduleName));
    }


    public function updateFields()
    {
        $this->updateItemDetailFields();
    }

    public function updateItemDetailFields()
    {
        $fieldNames = array('txtAdjustment', 'hdnSubTotal', 'hdnGrandTotal', 'hdnDiscountAmount', 'hdnS_H_Amount', 'pre_tax_total');
        $this->db->pquery('UPDATE vtiger_field SET uitype=? WHERE tabid=? AND fieldname IN(' . generateQuestionMarks($fieldNames) . ')',
            array(72, getTabid($this->moduleName), $fieldNames)
        );
    }

    public function deletePreferences($name)
    {
        $this->db->pquery('DELETE FROM `its4you_preferences` WHERE `its4you_preferences`.`name` = ?', [$name]);
    }

    /**
     * @param string $table
     * @return array
     */
    public function getTableColumns($table)
    {
        $resultColumns = $this->db->pquery('SELECT COLUMN_NAME FROM INFORMATION_SCHEMA.COLUMNS WHERE TABLE_NAME = ?', array($table));
        $columns = array();
        while ($column = $this->db->fetchByAssoc($resultColumns)) {
            $columns[] = $column['COLUMN_NAME'];
        }

        return $columns;
    }

    public function updateTables()
    {
        $fields = array(
            'deliverylaststatus' => 'ALTER TABLE its4you_deliverynotes ADD deliverylaststatus varchar(200) DEFAULT NULL',
            'tags' => 'ALTER TABLE its4you_deliverynotes ADD tags varchar(1) DEFAULT NULL',
            'pre_tax_total' => 'ALTER TABLE its4you_deliverynotes ADD pre_tax_total decimal(25,8) DEFAULT NULL',
            'compound_taxes_info' => 'ALTER TABLE its4you_deliverynotes ADD compound_taxes_info text DEFAULT NULL',
            'region_id' => 'ALTER TABLE its4you_deliverynotes ADD region_id int(19) DEFAULT NULL',
        );

        foreach ($fields as $field => $sql) {
            preg_match('/ALTER\ TABLE\ ([a-z0-9\_]+)\ ADD/', $sql, $matches);

            if ($matches[1] && !columnExists($field, $matches[1])) {
                $this->db->query($sql);
            }
        }
    }

    public function updateToWarehouses()
    {
        $oldDN = 'ITS4YouDeliveryNotes';
        $params = ['ITS4YouWHDeliveryNotes', $oldDN];
        $this->db->pquery('UPDATE vtiger_crmentityrel SET relmodule=? WHERE relmodule=?', $params);
        $this->db->pquery('UPDATE vtiger_crmentity SET setype=? WHERE setype=?', $params);
        $this->db->pquery('UPDATE vtiger_modentity_num SET semodule=? WHERE semodule=?', $params);
        $this->db->pquery('UPDATE vtiger_modtracker_basic SET module=? WHERE module=?', $params);
        $this->db->pquery('UPDATE vtiger_modtracker_relations SET targetmodule=? WHERE targetmodule=?', $params);
        $this->db->pquery('DELETE FROM vtiger_links WHERE linkurl LIKE ?', ['%' . $oldDN . '%']);
        $this->db->pquery('DELETE FROM vtiger_fieldmodulerel WHERE module = ? OR relmodule = ?', [$oldDN, $oldDN]);

        $oldInstance = Vtiger_Module::getInstance($oldDN);

        if ($oldInstance) {
            $oldInstance->deleteRelatedLists();
            $oldInstance->deleteLinks();
            $oldInstance->delete();
        }
    }

    /**
     * function to add Inactive Inventory Workflow for Delivery Notes Module
     *
     * @param $moduleName
     * @param $event_type
     *
     * @return bool
     * @throws Exception
     */
    public static function addWorkflowForDn($moduleName, $event_type)
    {
        $adb = PearDatabase::getInstance();

        vimport("~~modules/com_vtiger_workflow/include.inc");
        vimport("~~modules/com_vtiger_workflow/tasks/VTEntityMethodTask.inc");
        vimport("~~modules/com_vtiger_workflow/VTEntityMethodManager.inc");

        $moduleName = 'ITS4YouWHDeliveryNotes';

        $emm = new VTEntityMethodManager($adb);
        $emm->addEntityMethod($moduleName, 'SetBeforeStatus', 'modules/' . $moduleName . '/handlers/InventoryHandler.php', 'handleSetStatusBeforeSave');
        $emm->addEntityMethod($moduleName, 'UpdateInventory', 'modules/' . $moduleName . '/handlers/InventoryHandler.php', 'handleInventoryProductRelDeliveryNotes');

        $vtWorkFlow = new VTWorkflowManager($adb);
        $newWorkFlow = $vtWorkFlow->newWorkFlow($moduleName);
        $newWorkFlow->name = 'Update Inventory Products On Delivery';
        $newWorkFlow->description = 'Update Inventory Products On Delivery';
        $newWorkFlow->defaultworkflow = 1;
        $newWorkFlow->executionCondition = 3;
        $newWorkFlow->id = self::getWorkflowId($newWorkFlow->name);

        $vtWorkFlow->save($newWorkFlow);

        $tm = new VTTaskManager($adb);
        $task = $tm->createTask('VTEntityMethodTask', $newWorkFlow->id);
        $task->active = 1;
        $task->summary = 'Update Inventory Products';
        $task->methodName = 'UpdateInventory';
        $task->id = self::getWorkflowTaskId($task->summary, $newWorkFlow->id);

        $tm->saveTask($task);

        return true;
    }

    public static function isEntityMethodExists($moduleName, $methodName)
    {
        $adb = PearDatabase::getInstance();
        $result = $adb->pquery(
            'SELECT workflowtasks_entitymethod_id FROM com_vtiger_workflowtasks_entitymethod WHERE module_name=? AND method_name=?',
            [$moduleName, $methodName]
        );

        return (bool)$adb->num_rows($result);
    }

    /**
     * @param string $name
     * @return int
     * @throws Exception
     */
    public static function getWorkflowId($workflowName)
    {
        $adb = PearDatabase::getInstance();
        $result = $adb->pquery(
            'SELECT workflow_id FROM com_vtiger_workflows WHERE workflowname=?',
            [$workflowName]
        );
        $id = (int)$adb->query_result($result, 0, 'workflow_id');

        return !empty($id) ? $id : null;
    }

    /**
     * @param string $summary
     * @param int $workflowId
     * @return int
     * @throws Exception
     */
    public static function getWorkflowTaskId($summary, $workflowId)
    {
        $adb = PearDatabase::getInstance();
        $result = $adb->pquery(
            'SELECT task_id FROM com_vtiger_workflowtasks WHERE summary=? AND workflow_id=?',
            [$summary, $workflowId]
        );

        $id = (int)$adb->query_result($result, 0, 'task_id');

        return !empty($id) ? $id : null;
    }

    public function changeWfTaskStatuses($newStatus) {
        if (class_exists("Settings_Workflows_Record_Model")) {
            $wfTaskManager = new VTTaskManager($this->db);
            $wfTasks = $wfTaskManager->getTasks();

            foreach ($wfTasks as $wfTaskObject) {
                if ($wfTaskObject->methodName == "UpdateInventory") {
                    $wfTaskObject->active = $newStatus;
                    $wfTaskManager->saveTask($wfTaskObject);
                }
            }
        }
    }

    public static function runSelfFormUpdates()
    {
        $db = PearDatabase::getInstance();

        $moduleName = 'ITS4YouWHDeliveryNotes';
        $moduleInstance = Vtiger_Module::getInstance($moduleName);
        $block = Vtiger_Block::getInstance('LBL_ITEM_DETAILS', $moduleInstance);

        $tableName = 'its4you_deliverynotes';
        $fieldName = 'pre_tax_total';
        $fieldInstance = Vtiger_Field::getInstance($fieldName, $moduleInstance);

        if (!$fieldInstance) {
            $preTaxTotalField = new Vtiger_Field();

            $preTaxTotalField->name = $fieldName;
            $preTaxTotalField->label = 'Pre Tax Total';
            $preTaxTotalField->table = $tableName;
            $preTaxTotalField->column = $fieldName;
            $preTaxTotalField->columntype = 'decimal(25,8)';
            $preTaxTotalField->typeofdata = 'N~O';
            $preTaxTotalField->uitype = '72';
            $preTaxTotalField->masseditable = '1';
            $preTaxTotalField->displaytype = '3';

            $block->addField($preTaxTotalField);
        }

        //Updating existing inventory tax tables
        $columns = $db->getColumnNames($tableName);
        if (!in_array('compound_taxes_info', $columns)) {
            $db->pquery("ALTER TABLE $tableName ADD COLUMN compound_taxes_info TEXT", []);
        }
        $db->pquery('UPDATE ' . $tableName . ' SET compound_taxes_info=?', ['[]']);

        //creating new field in entity tables
        $moduleInstance = Vtiger_Module::getInstance($moduleName);
        $blockInstance = Vtiger_Block::getInstance('LBL_ITEM_DETAILS', $moduleInstance);
        $fieldName = 'region_id';
        $fieldInstance = Vtiger_Field::getInstance($fieldName, $moduleInstance);

        if (!$fieldInstance) {
            $fieldInstance = new Vtiger_Field();

            $fieldInstance->name = $fieldName;
            $fieldInstance->column = $fieldName;
            $fieldInstance->table = $tableName;
            $fieldInstance->label = 'Tax Region';
            $fieldInstance->columntype = 'int(19)';
            $fieldInstance->typeofdata = 'N~O';
            $fieldInstance->uitype = '16';
            $fieldInstance->readonly = '0';
            $fieldInstance->displaytype = '5';
            $fieldInstance->masseditable = '0';

            $blockInstance->addField($fieldInstance);
        }

        return true;
    }

    public function deleteCustomLinks()
    {
        $this->deleteSidebarWidgets();
        $this->deletePreferences('updateInventoryOnDelivery');
        //enable standard workflow in order to avoid diplicitly stock manipulation
        $this->changeWfTaskStatuses(true);
        $this->retrieveRelatedListForDelete();
        $this->updateRelatedList(false);
        $this->updateCustomLinks(false);

        ModTracker::disableTrackingForModule(getTabid($this->moduleName));
        ModComments::removeWidgetFrom([$this->moduleName]);
    }

    public function retrieveRelatedListForDelete()
    {
        $this->registerRelatedLists[] = ['Products', 'ITS4YouWHDeliveryNotes', 'ProductMovements'];
        $this->registerRelatedLists[] = ['Products', 'ITS4YouWHDeliveryNotes', 'ProductInWarehouses'];
    }

    public function deleteSidebarWidgets()
    {
        foreach ($this->relatedModules as $relatedModule) {
            $moduleInstance = Vtiger_Module::getInstance($relatedModule);

            if ($moduleInstance) {
                $type = 'DETAILVIEWSIDEBARWIDGET';

                $moduleInstance->deleteLink($type, 'ITS4YouWHDeliveryNotes');
                $moduleInstance->deleteLink($type, 'ITS4YouDeliveryNotes');
            }
        }
    }

    /**
     * function to completely remove Inactive Inventory Workflow for Delivery Notes Module
     *
     * @param $moduleName
     * @param $event_type
     *
     * @return bool
     * @throws Exception
     */
    public static function removeWorkflowForDn($moduleName, $event_type)
    {
        $adb = PearDatabase::getInstance();

        vimport("~~modules/com_vtiger_workflow/include.inc");
        vimport("~~modules/com_vtiger_workflow/tasks/VTEntityMethodTask.inc");
        vimport("~~modules/com_vtiger_workflow/VTEntityMethodManager.inc");

        $moduleName = 'ITS4YouWHDeliveryNotes';

        $emm = new VTEntityMethodManager($adb);
        $emm->removeEntityMethod($moduleName, 'UpdateInventory');
        $emm->removeEntityMethod($moduleName, 'SetBeforeStatus');

        $taskSummary = 'Update Inventory Products';
        $workflowSummary = 'Update Inventory Products On Delivery';
        $adb->pquery(
            'DELETE FROM com_vtiger_workflowtasks WHERE summary=? AND workflow_id IN (SELECT workflow_id FROM com_vtiger_workflows WHERE module_name=? AND summary=?)',
            [$taskSummary, $moduleName, $workflowSummary]
        );
        $adb->pquery(
            'DELETE FROM com_vtiger_workflows WHERE module_name=? AND summary=?',
            [$moduleName, $workflowSummary]
        );

        return true;
    }

    /** Overriden function to delete an entity with given Id */
    public function trash($module, $id)
    {
        parent::trash($module, $id);
        ITS4YouWarehouses_DNUtil_ActionBlock::updateRelatedSalesOrderStatus($id);
        ITS4YouWarehouses_DNUtil_ActionBlock::trashRelatedIssueCards($id);
    }

    /**
     * @param bool $register
     */
    public function updateRelatedList($register = true)
    {
        foreach ($this->registerRelatedLists as $relatedList) {
            $module = Vtiger_Module::getInstance($relatedList[0]);
            $relatedModule = Vtiger_Module::getInstance($relatedList[1]);

            if ($module && $relatedModule) {
                $relatedLabel = isset($relatedList[2]) ? $relatedList[2] : $relatedModule->name;
                $relatedActions = isset($relatedList[3]) ? $relatedList[3] : '';
                $relatedFunction = isset($relatedList[4]) ? $relatedList[4] : 'get_related_list';

                if ($register) {
                    if ($this->checkRelatedList($module->getId(), $relatedModule->getId(), $relatedLabel, $relatedFunction) == 0) {
                        $module->setRelatedList($relatedModule, $relatedLabel, $relatedActions, $relatedFunction);
                    }
                } else {
                    $module->unsetRelatedList($relatedModule, $relatedLabel, $relatedFunction);
                }
            }
        }
    }

    /**
     * @param int $tabid
     * @param int $related_tabid
     * @param string $label
     * @param string $name
     * @return mixed
     */
    private function checkRelatedList($tabid, $related_tabid, $label, $name = "get_related_list")
    {
        $query = "SELECT relation_id FROM vtiger_relatedlists WHERE tabid=? AND related_tabid=? AND label=? AND name=?";
        $result = $this->db->pquery($query, array($tabid, $related_tabid, $label, $name));
        return $this->db->num_rows($result);
    }

    /**
     * @param string $module
     * @param object $queryPlanner
     * @return string
     * @throws Exception
     */
    public function generateReportsQuery($module, $queryPlanner)
    {
        $query = parent::generateReportsQuery($module, $queryPlanner);

        // Define the dependency matrix ahead
        $matrix = $queryPlanner->newDependencyMatrix();
        $matrix->setDependency('vtiger_crmentity' . $this->moduleName, array('vtiger_users' . $this->moduleName, 'vtiger_groups' . $this->moduleName, 'vtiger_lastModifiedBy' . $this->moduleName));
        $matrix->setDependency('vtiger_inventoryproductrel' . $this->moduleName, array('vtiger_products' . $this->moduleName, 'vtiger_service' . $this->moduleName));

        if ($queryPlanner->requireTable('its4you_deliverynoteshipads')) {
            $query .= sprintf(' LEFT JOIN its4you_deliverynoteshipads AS its4you_deliverynoteshipads ON its4you_deliverynoteshipads.deliverynoteid=%s.%s ', $this->table_name, $this->table_index);
        }

        $requiredTables = [
            'vtiger_inventoryproductrel' => 'id',
        ];

        $this->generateCustomReportsQuery($this->table_index, $this->table_name, $requiredTables, $queryPlanner, $query, 'tmp');

        $requiredTables = [
            'vtiger_currency_info' => 'id',
        ];

        $this->generateCustomReportsQuery('currency_id', $this->table_name, $requiredTables, $queryPlanner, $query, '');

        $requiredTables = [
            'vtiger_contactdetails' => 'contactid',
        ];

        $this->generateCustomReportsQuery('contactid', $this->table_name, $requiredTables, $queryPlanner, $query);

        $tableName = 'vtiger_inventoryproductreltmp' . $this->moduleName;
        $requiredTables = [
            'vtiger_products' => 'productid',
            'vtiger_service' => 'serviceid',
        ];

        $this->generateCustomReportsQuery('productid', $tableName, $requiredTables, $queryPlanner, $query, '');

        return $query;
    }

    /**
     * @param string $fieldName
     * @param string $tableName
     * @param array $requiredTables
     * @param object $queryPlanner
     * @param string $query
     * @param string $rel
     * @throws Exception
     */
    public function generateCustomReportsQuery($fieldName, $tableName, $requiredTables, &$queryPlanner, &$query, $rel = 'Rel')
    {
        $thisModule = $this->moduleName;
        $relFieldId = '';

        if ('Rel' === $rel) {
            $result = $this->db->pquery('SELECT fieldid FROM vtiger_field WHERE columnname = ? AND tablename = ?',
                array($fieldName, $tableName));
            $relFieldId = $this->db->query_result($result, 0, 'fieldid');
        }

        foreach ($requiredTables as $requiredTable => $requiredField) {
            $customRequiredTableName = $requiredTable . $rel . $thisModule . $relFieldId;

            if ($queryPlanner->requireTable($customRequiredTableName)) {
                $query .= sprintf(
                    ' LEFT JOIN %s AS %s ON %s.%s = %s.%s ',
                    $requiredTable,
                    $customRequiredTableName,
                    $customRequiredTableName,
                    $requiredField,
                    $tableName,
                    $fieldName
                );
            }
        }
    }

    /**
     * @param bool $register
     */
    public function updateCustomLinks($register = true)
    {
        foreach ($this->registerCustomLinks as $customLink) {
            list($moduleName, $type, $label, $url, $icon, $sequence, $handlerInfo) = $customLink;

            $module = Vtiger_Module::getInstance($moduleName);
            $url = str_replace('$LAYOUT$', Vtiger_Viewer::getDefaultLayoutName(), $url);

            if ($module) {
                $module->deleteLink($type, $label);

                if ($register) {
                    $module->addLink($type, $label, $url, $icon, $sequence, $handlerInfo);
                }
            }
        }
    }
}
