<?php
/*********************************************************************************
 * The content of this file is subject to the ITS4YouImport license.
 * ("License"); You may not use this file except in compliance with the License
 * The Initial Developer of the Original Code is IT-Solutions4You s.r.o.
 * Portions created by IT-Solutions4You s.r.o. are Copyright(C) IT-Solutions4You s.r.o.
 * All Rights Reserved.
 * ****************************************************************************** */

require_once 'modules/ITS4YouImport/libraries/PHPExcel.php';

abstract class ITS4YouImport_PHPExcelReader_Reader extends ITS4YouImport_FileReader_Reader
{

    /** @var PHPExcel */
    protected $phpExcel;

    /** $var ITS4YouImport_ChunkReadFilter_Filter */
//    protected $chunkFilter;

    public function __construct($request, $user)
    {
        parent::__construct($request, $user);

//        $this->chunkFilter = new ITS4YouImport_ChunkReadFilter_Filter();
//        $this->chunkFilter->setStartRow(1);

        try {
            $this->createPHPExcelObject();
        } catch (Exception $ex) {
            $this->createPHPExcelObjectAuto();
        }
    }

    /**
     * @throws Exception
     */
    abstract protected function createPHPExcelObject();

    protected function createPHPExcelObjectAuto()
    {
        $this->phpExcel = PHPExcel_IOFactory::load($this->getFilePath());
        $this->phpExcel->setActiveSheetIndex(0);
    }

    /** Function returns the database column type of the field
     * @param $fieldObject <Vtiger_Field_Model>
     * @param $fieldTypes <Array> - fieldnames with column type
     * @return <String> - column name with type for sql creation of table
     */
    public function getDBColumnType($fieldObject, $fieldTypes)
    {
        $columnsListQuery = '';
        $fieldName = $fieldObject->getName();
        $dataType = $fieldObject->getFieldDataType();
        //added additional data types in order to handle type check at later phase (avoid)
        //automatic DB conversion rules wo any message
        $skipDataType = array('reference', 'owner', 'currencyList', 'date', 'datetime', 'integer', 'double', 'currency', 'percentage');
        if (in_array($dataType, $skipDataType)) {
            $columnsListQuery .= ',' . $fieldName . ' varchar(250)';
        } else {
            $columnsListQuery .= ',' . $fieldName . ' ' . $fieldTypes[$fieldObject->get('column')];
        }

        return $columnsListQuery;
    }

    public function getFirstRowData($hasHeader = true)
    {
        $worksheet = $this->phpExcel->getActiveSheet();
        $highestColumn = $worksheet->getHighestColumn();
        $highestColumnIndex = PHPExcel_Cell::columnIndexFromString($highestColumn);
        $firstRowValues = array();
        $secondRowValues = array();

        if ($hasHeader) {
            for ($col = 0; $col <= $highestColumnIndex; $col++) {
                $firstRowCell = $worksheet->getCellByColumnAndRow($col, 1);
                $secondRowCell = $worksheet->getCellByColumnAndRow($col, 2);
                $firstRowCellValue = $this->getPHPExcelCellValue($firstRowCell, false, true);
                if (!empty($firstRowCellValue)) {
                    $firstRowValues[] = $firstRowCellValue;
                    $secondRowValues[] = $this->getPHPExcelCellValue($secondRowCell);
                }
            }

            return array_combine($firstRowValues, $secondRowValues);
        } else {
            for ($col = 0; $col <= $highestColumnIndex; $col++) {
                $firstRowCell = $worksheet->getCellByColumnAndRow($col, 1);
                $firstRowCellValue = $this->getPHPExcelCellValue($firstRowCell);
                if (!empty($firstRowCellValue)) {
                    $firstRowValues[] = $this->getPHPExcelCellValue($firstRowCell);
                }
            }
            return $firstRowValues;
        }
    }

    /**
     * @param PHPExcel_Cell $cell
     * @param bool $inInternalFormat
     * @param bool $isHeaderCell
     * @return string
     */
    protected function getPHPExcelCellValue(PHPExcel_Cell $cell, $inInternalFormat = false, $isHeaderCell = false)
    {
        $cellValue = trim($cell->getFormattedValue());

        if (!$isHeaderCell) {
            if (PHPExcel_Shared_Date::isDateTime($cell)) {
                $cellValue = self::getDateCellValue($cellValue);
            }
        } else {
            $cellValue = trim($cellValue, '"');
        }

        return $cellValue;
    }

    protected static function getDateCellValue($cellValue)
    {
        $dateFromString = self::getDateFromString($cellValue);

        if (!empty($dateFromString)) {
            return date('d/m/Y', strtotime($dateFromString));
        }

        return $cellValue;
    }

    /**
     * @param $value
     * @return string
     */
    protected static function getDateFromString($value)
    {
        $date = null;

        if (self::isValidDate($value)) {
            if (strpos($value, '-')) {
                $delimiter = '-';
            } elseif (strpos($value, '.')) {
                $delimiter = '.';
            } else {
                $delimiter = '/';
            }

            $dateParts = explode($delimiter, $value);

            if (4 === strlen($dateParts[0])) {
                $date = date('Y-m-d', strtotime(sprintf('%s-%s-%d', $dateParts[0], $dateParts[1], $dateParts[2])));
            } else {
                $date = date('Y-m-d', strtotime(sprintf('%s-%s-%d', $dateParts[2], $dateParts[1], $dateParts[0])));
            }
        }

        return $date;
    }

    public static function isValidDate($value)
    {
        preg_match_all(
            '/^[0-9]{1,4}[.\-\/][0-9]{1,2}[.\-\/][0-9]{1,4}$/',
            decode_html($value),
            $matches,
            PREG_SET_ORDER
            , 0
        );

        return !empty($matches);
    }

    public function getRowByNumber($rows = array())
    {
        $result = array();

        foreach ($rows as $row) {
            $worksheet = $this->phpExcel->getActiveSheet();
            $highestColumn = $worksheet->getHighestColumn();
            $highestColumnIndex = PHPExcel_Cell::columnIndexFromString($highestColumn);
            $rowValues = array();

            for ($col = 0; $col <= $highestColumnIndex; $col++) {
                $rowCell = $worksheet->getCellByColumnAndRow($col, $row);
                $rowValues[] = $this->getPHPExcelCellValue($rowCell);
            }

            $result[$row] = $rowValues;
        }

        return $result;
    }

    /**
     * Returns number of records which are to be read - before full reading
     * @return int Number of records to be read
     */
    public function getNumberOfRecordsToBeRead()
    {
        $activeSheet = $this->phpExcel->getActiveSheet();
        /* @var $activeSheet PHPExcel_Worksheet */
        $highestRow = $activeSheet->getHighestRow("A");
        if ($this->hasHeader()) {
            $highestRow--;
        }
        return $highestRow;
    }

    public function read()
    {
        $status = $this->createTable();
        if (!$status) {
            return false;
        }

        $fieldMapping = $this->request->get('field_mapping');

        $rowIterator = $this->phpExcel->getActiveSheet()->getRowIterator();
        /* @var $rowIterator PHPExcel_Worksheet_RowIterator */
        $activeSheet = $this->phpExcel->getActiveSheet();
        /* @var $activeSheet PHPExcel_Worksheet */
        while ($rowIterator->valid()) {
            if ($this->hasHeader() && $rowIterator->key() === 1) {
                $rowIterator->next();
                continue;
            }

            $mappedData = array();
            $allValuesEmpty = true;

            $row = $rowIterator->current();
            /* @var $row PHPExcel_Worksheet_Row */
            foreach ($fieldMapping as $fieldName => $index) {
                $cell = $activeSheet->getCellByColumnAndRow($index, $row->getRowIndex());
                /* @var $cell PHPExcel_Cell */
                $cellValue = $this->getPHPExcelCellValue($cell, true);
                $mappedData[$fieldName] = $cellValue;
                if (!empty($cellValue)) {
                    $allValuesEmpty = false;
                }
            }
            if ($allValuesEmpty) {
                $rowIterator->next();
                continue;
            }

            $fieldNames = array_keys($mappedData);
            $fieldValues = array_values($mappedData);

            $this->addRecordToDB($fieldNames, $fieldValues);

            $rowIterator->next();
        }
    }

    public function __destruct()
    {
        $this->phpExcel->disconnectWorksheets();
        unset($this->phpExcel);
    }

}
