<?php
/*********************************************************************************
 * The content of this file is subject to the ITS4YouImport license.
 * ("License"); You may not use this file except in compliance with the License
 * The Initial Developer of the Original Code is IT-Solutions4You s.r.o.
 * Portions created by IT-Solutions4You s.r.o. are Copyright(C) IT-Solutions4You s.r.o.
 * All Rights Reserved.
 * ****************************************************************************** */

class ITS4YouImport_Logger_Model
{

    /** @var ITS4YouImport_Logger_Model singleton instance reference */
    private static $instance = null;
    protected $importEntityId;

    /** @var int User record model */
    protected $userId;

    /** @var array Array of ITS4YouImport_LogEntryBase_Model */
    protected $logEntries;

    /** @var int Identifier of lastly added entry.
     * It's used for performance optimization (i.e. when only unsuccess entries
     * should be stored in entry repository). */
    private $lastAddedEntryId;

    /** @var bool Flag for keeping all entries or only the those which are unsuccessfull */
    private $keepAllEntries;

    private function __construct($importEntityId, $userId, $keepAllEntries)
    {
        $this->importEntityId = $importEntityId;
        $this->userId = $userId;
        $this->lastAddedEntryId = null;
        $this->keepAllEntries = $keepAllEntries;
    }

    /**
     * Returns instance of logger
     * @param int $importEntityId
     * @param int $userId
     * @param bool $keepAllEntries Optional parameter for performance optimization
     * By setting this to true all entries are stored in entry repository, otherwise
     * only unsuccessfull entries are stored.
     */
    public static function factory($importEntityId, $userId, $keepAllEntries = false)
    {
        if ((self::$instance == null || self::$instance->importEntityId != $importEntityId || self::$instance->userId != $userId || self::$instance->keepAllEntries != self::$instance->keepAllEntries)) {
            self::createInstance($importEntityId, $userId, $keepAllEntries);
        }

        return self::$instance;
    }

    private static function createInstance($importEntityId, $userId, $keepAllEntries = false)
    {
        self::$instance = self::restore($importEntityId);
        if (self::$instance == null) {
            self::$instance = new ITS4YouImport_Logger_Model($importEntityId, $userId, $keepAllEntries);
        }
    }

    private static function restore($importEntityId)
    {
        $db = PearDatabase::getInstance();
        if (Vtiger_Utils::CheckTable('its4you_import_log')) {
            $sql = "SELECT * FROM its4you_import_log WHERE importid = ?";
            $result = $db->pquery($sql, array($importEntityId));
            if ($db->num_rows($result) > 0) {
                return unserialize(base64_decode($db->query_result($result, 0, "logdata")));
            }
        }
        return null;
    }

    /**
     * Adds log entry to the log repository
     * @param ITS4YouImport_LogEntryBase_Model $logEntry
     */
    public function addLogEntry(ITS4YouImport_LogEntryBase_Model $logEntry)
    {
        $this->checkLastAddedEntry();
        $this->logEntries[$logEntry->getId()] = $logEntry;
        $this->lastAddedEntryId = $logEntry->getId();
    }

    /**
     * Checks and adjusts entry repository with taking the $keepAllEntries flag
     * into consideration
     */
    private function checkLastAddedEntry()
    {
        if (!$this->keepAllEntries && $this->lastAddedEntryId != null) {
            $lastAddedEntry = $this->logEntries[$this->lastAddedEntryId];
            /* @var $lastAddedEntry ITS4YouImport_LogEntryBase_Model */
            //updated 600.700.1.1 -> import drop prevention
            if (is_object($lastAddedEntry) && !$lastAddedEntry->isNotOK()) {
                unset($this->logEntries[$this->lastAddedEntryId]);
            }
        }
    }

    /**
     *
     * @param type $logEntryId
     * @return ITS4YouImport_LogEntryBase_Model
     */
    public function getLogEntry($logEntryId)
    {
        return (isset($this->logEntries[$logEntryId]) ? $this->logEntries[$logEntryId] : null);
    }

    public function store()
    {
        $this->checkLastAddedEntry();

        $db = PearDatabase::getInstance();

        if (!Vtiger_Utils::CheckTable('its4you_import_log')) {
            Vtiger_Utils::CreateTable(
                'its4you_import_log',
                "(importid INT NOT NULL PRIMARY KEY, logdata LONGTEXT)", true);
        }

        //update 600.700.1.1 insert changed
        //$db->pquery("INSERT INTO its4you_import_log(importid, logdata) VALUES(?,?)", array($this->importEntityId, base64_encode(serialize($this))));
        $logData = base64_encode(serialize($this));
        $db->pquery('INSERT INTO its4you_import_log(importid, logdata) VALUES(?,?)
                    ON DUPLICATE KEY UPDATE logdata=?', [$this->importEntityId, $logData, $logData]);
    }

    /**
     * Optimize repository of log entries and remove correct one
     */
    public function purifyRepository()
    {
        foreach ($this->logEntries as $logEntryId => $logEntry) {
            /* @var $logEntry ITS4YouImport_LogEntryBase_Model */
            if (!$logEntry->isNotOK()) {
                unset($this->logEntries[$logEntryId]);
            }
        }
    }

    public function getLogEntries()
    {
        return $this->logEntries;
    }

}
