<?php

/* * *******************************************************************************
 * The content of this file is subject to the CreditNotes4You license.
 * ("License"); You may not use this file except in compliance with the License
 * The Initial Developer of the Original Code is IT-Solutions4You s.r.o.
 * Portions created by IT-Solutions4You s.r.o. are Copyright(C) IT-Solutions4You s.r.o.
 * All Rights Reserved.
 * ****************************************************************************** */

require_once('data/CRMEntity.php');
require_once('data/Tracker.php');
include_once('vtlib/Vtiger/Utils.php');

class CreditNotes4You extends CRMEntity
{

    const MODULE_NAME = 'CreditNotes4You';
    public static $statusFieldName = 'creditnotes4youstatus';

    public $db, $log; // Used in class functions of CRMEntity
    public $table_name = "vtiger_creditnotes4you";
    public $table_index = 'creditnotes4you_id';
    public $column_fields = array();
    public $moduleName = 'CreditNotes4You';
    public $parentName = 'Sales';
    public $module;

    /** Indicator if this is a custom module or standard module */
    public $IsCustomModule = true;

    /**
     * Mandatory table for supporting custom fields.
     */
    public $customFieldTable = array('vtiger_creditnotes4youcf', 'creditnotes4you_id');

    /**
     * Mandatory for Saving, Include tables related to this module.
     */
    public $tab_name = array(
        'vtiger_crmentity',
        'vtiger_creditnotes4you',
        'vtiger_creditnotes4youcf',
        'vtiger_inventoryproductrel'
    );

    /**
     * Mandatory for Saving, Include tablename and tablekey columnname here.
     */
    public $tab_name_index = array(
        'vtiger_crmentity' => 'crmid',
        'vtiger_creditnotes4you' => 'creditnotes4you_id',
        'vtiger_creditnotes4youcf' => 'creditnotes4you_id',
        'vtiger_inventoryproductrel' => 'id'
    );

    /**
     * Mandatory for Listing (Related listview)
     */
    public $list_fields = array(
        'creditnotes4you_no' => array('creditnotes4you' => 'creditnotes4you_no'),
        'CreditNotes4You Name' => array('creditnotes4you' => 'creditnotes4you_name'),
        'Assigned To' => array('crmentity' => 'smownerid')
    );
    public $list_fields_name = array(
        'creditnotes4you_no' => 'creditnotes4you_no',
        'CreditNotes4You Name' => 'creditnotes4you_name',
        'Assigned To' => 'assigned_user_id'
    );
    // Make the field link to detail view
    public $list_link_field = 'creditnotes4you_name';
    // For Popup listview and UI type support
    public $search_fields = array(
        'CreditNotes4You Name' => array('creditnotes4you' => 'creditnotes4you_name'),
        'Assigned To' => array('crmentity' => 'smownerid')
    );
    public $search_fields_name = array(
        'CreditNotes4You Name' => 'creditnotes4you_name',
        'Assigned To' => 'assigned_user_id'
    );
    // For Popup window record selection
    public $popup_fields = array('creditnotes4you_name', 'creditnotes4you_no', 'account_id');
    // Placeholder for sort fields - All the fields will be initialized for Sorting through initSortFields
    public $sortby_fields = array('creditnotes4you_name', 'creditnotes4you_no');
    // For Alphabetical search
    public $def_basicsearch_col = 'creditnotes4you_name';
    // Required Information for enabling Import feature
    public $required_fields = array('creditnotes4you_name' => 1);
    // Used when enabling/disabling the mandatory fields for the module.
    // Refers to vtiger_field.fieldname values.
    public $mandatory_fields = array('creditnotes4you_name');
    // Callback function list during Importing
    //public $special_functions = Array('set_import_assigned_user');

    public $default_order_by = 'createdtime';
    public $default_sort_order = 'DESC';
    public $relatedModulesList = ['Invoice', 'Accounts', 'Contacts'];
    private $version_type;
    private $license_key;
    private $version_no;

    /**    Constructor which will set the column_fields in this object
     */
    function __construct()
    {
        global $log;
        $this->column_fields = getColumnFields(get_class($this));
        $this->db = PearDatabase::getInstance();
        $this->log = $log;
    }

    /**
     * function to completely remove Inactive Inventory Workflow for Credit Notes Module
     *
     * @param $moduleName
     * @param $event_type
     *
     * @return bool
     */
    public static function removeWofkflowForCn($moduleName, $event_type)
    {
        $adb = PearDatabase::getInstance();

        vimport("~~modules/com_vtiger_workflow/include.inc");
        vimport("~~modules/com_vtiger_workflow/tasks/VTEntityMethodTask.inc");
        vimport("~~modules/com_vtiger_workflow/VTEntityMethodManager.inc");

        $moduleName = self::MODULE_NAME;

        $emm = new VTEntityMethodManager($adb);
        $emm->removeEntityMethod($moduleName, 'UpdateInventory');

        $result = $adb->pquery('SELECT workflow_id FROM com_vtiger_workflows where module_name=?', [$moduleName]);
        while ($row = $adb->fetchByAssoc($result)) {
            $workflowId = $row['workflow_id'];
            if ($workflowId) {
                $adb->pquery('DELETE FROM com_vtiger_workflowtasks
                                        WHERE workflow_id=?', [$workflowId]);
            }
        }

        $adb->pquery('DELETE FROM com_vtiger_workflows where module_name=?', [$moduleName]);

        return true;
    }

    function save_module($module)
    {
        if (isset($_REQUEST)) {
            if ('CreditNotes4YouAjax' !== $_REQUEST['action'] && 'DETAILVIEW' !== $_REQUEST['ajxaction'] && 'MassEditSave' !== $_REQUEST['action'] && 'ProcessDuplicates' !== $_REQUEST['action'] && 'SaveAjax' !== $_REQUEST['action'] && false !== $this->isLineItemUpdate) {
                //Based on the total Number of rows we will save the product relationship with this entity
                //CreditNotes4You_Inventory_Helper::saveCreditNotes4YouInventoryProductDetails($this, 'CreditNotes4You');
                saveInventoryProductDetails($this, 'CreditNotes4You');
            }
        }
        // Inserting into attachments table
        $this->insertIntoAttachment($this->id, $module);
        // Update the currency id and the conversion rate for the credit note
        $this->updateItemDetails();

        $this->updateRelations();
    }

    public function updateRelations()
    {
        foreach (array('invoice_id', 'account_id', 'contact_id') as $field) {
            if (!empty($this->column_fields[$field])) {
                $this->createRelationFromRecord($this->column_fields[$field]);
            }
        }
    }

    public function createRelationFromRecord($recordId)
    {
        $module = Vtiger_Module_Model::getInstance($this->moduleName);
        $parentModuleName = getSalesEntityType($recordId);
        $parentModule = $parentModuleName ? Vtiger_Module_Model::getInstance($parentModuleName) : false;

        if ($parentModule) {
            $relationModel = Vtiger_Relation_Model::getInstance($parentModule, $module);

            if ($relationModel) {
                $relationModel->addRelation($recordId, $this->id);
            }
        }
    }


    public function updateItemDetails()
    {
        $params = [
            'currency_id' => $this->column_fields['currency_id'],
            'conversion_rate' => $this->column_fields['conversion_rate'],
        ];

        $this->db->pquery(
            sprintf('UPDATE %s SET %s=? WHERE %s=?', $this->table_name, implode('=?,', array_keys($params)), $this->table_index),
            array($params, $this->id)
        );
    }

    /**
     * This function is used to add the vtiger_attachments. This will call the function uploadAndSaveFile which will upload the attachment into the server and save that attachment information in the database.
     * @param int $id - entity id to which the vtiger_files to be uploaded
     * @param string $module - the current module name
     */
    function insertIntoAttachment($id, $module)
    {
        $file_saved = false;

        foreach ($_FILES as $files) {
            if ($files['name'] != '' && $files['size'] > 0) {
                $file_saved = $this->uploadAndSaveFile($id, $module, $files);
            }
        }
    }

    /**
     * Return query to use based on given modulename, fieldname
     * Useful to handle specific case handling for Popup
     */
    function getQueryByModuleField($module, $fieldname, $srcrecord)
    {
        // $srcrecord could be empty
    }

    /**
     * Get list view query.
     */
    function getListQuery($module, $where = '')
    {
        $query = "SELECT vtiger_crmentity.*, $this->table_name.*";

        // Select Custom Field Table Columns if present
        if (!empty($this->customFieldTable)) {
            $query .= ", " . $this->customFieldTable[0] . ".* ";
        }

        $query .= " FROM $this->table_name";

        $query .= "	INNER JOIN vtiger_crmentity ON vtiger_crmentity.crmid = $this->table_name.$this->table_index";

        // Consider custom table join as well.
        if (!empty($this->customFieldTable)) {
            $query .= " INNER JOIN " . $this->customFieldTable[0] . " ON " . $this->customFieldTable[0] . '.' . $this->customFieldTable[1] .
                " = $this->table_name.$this->table_index";
        }
        $query .= " LEFT JOIN vtiger_users ON vtiger_users.id = vtiger_crmentity.smownerid";
        $query .= " LEFT JOIN vtiger_groups ON vtiger_groups.groupid = vtiger_crmentity.smownerid";


        $linkedModulesQuery = $this->db->pquery("SELECT distinct fieldname, columnname, relmodule FROM vtiger_field" .
            " INNER JOIN vtiger_fieldmodulerel ON vtiger_fieldmodulerel.fieldid = vtiger_field.fieldid" .
            " WHERE uitype='10' AND vtiger_fieldmodulerel.module=?", array($module));
        $linkedFieldsCount = $this->db->num_rows($linkedModulesQuery);

        for ($i = 0; $i < $linkedFieldsCount; $i++) {
            $related_module = $this->db->query_result($linkedModulesQuery, $i, 'relmodule');
            $fieldname = $this->db->query_result($linkedModulesQuery, $i, 'fieldname');
            $columnname = $this->db->query_result($linkedModulesQuery, $i, 'columnname');

            $other = CRMEntity::getInstance($related_module);
            vtlib_setup_modulevars($related_module, $other);

            $query .= " LEFT JOIN $other->table_name ON $other->table_name.$other->table_index = $this->table_name.$columnname";
        }

        $query .= "	WHERE vtiger_crmentity.deleted = 0 " . $where;
        $query .= $this->getListViewSecurityParameter($module);
        return $query;
    }

    /**
     * Apply security restriction (sharing privilege) query part for List view.
     */
    function getListViewSecurityParameter($module)
    {
        global $current_user;
        require('user_privileges/user_privileges_' . $current_user->id . '.php');
        require('user_privileges/sharing_privileges_' . $current_user->id . '.php');

        $sec_query = '';
        $tabid = getTabid($module);

        if ($is_admin == false && $profileGlobalPermission[1] == 1 && $profileGlobalPermission[2] == 1 && $defaultOrgSharingPermission[$tabid] == 3) {

            $sec_query .= " AND (vtiger_crmentity.smownerid in($current_user->id) OR vtiger_crmentity.smownerid IN
					(
						SELECT vtiger_user2role.userid FROM vtiger_user2role
						INNER JOIN vtiger_users ON vtiger_users.id=vtiger_user2role.userid
						INNER JOIN vtiger_role ON vtiger_role.roleid=vtiger_user2role.roleid
						WHERE vtiger_role.parentrole LIKE '" . $current_user_parent_role_seq . "::%'
					)
					OR vtiger_crmentity.smownerid IN
					(
						SELECT shareduserid FROM vtiger_tmp_read_user_sharing_per
						WHERE userid=" . $current_user->id . " AND tabid=" . $tabid . "
					)
					OR
						(";

            // Build the query based on the group association of current user.
            if (sizeof($current_user_groups) > 0) {
                $sec_query .= " vtiger_groups.groupid IN (" . implode(",", $current_user_groups) . ") OR ";
            }
            $sec_query .= " vtiger_groups.groupid IN
						(
							SELECT vtiger_tmp_read_group_sharing_per.sharedgroupid
							FROM vtiger_tmp_read_group_sharing_per
							WHERE userid=" . $current_user->id . " and tabid=" . $tabid . "
						)";
            $sec_query .= ")
				)";
        }
        return $sec_query;
    }

    /**
     * Create query to export the records.
     */
    function create_export_query($where)
    {
        global $current_user;

        include("include/utils/ExportUtils.php");

        //To get the Permitted fields query and the permitted fields list
        $sql = getPermittedFieldsQuery('Assets', "detail_view");

        $fields_list = getFieldsListFromQuery($sql);

        $query = "SELECT $fields_list, vtiger_users.user_name AS user_name
					FROM vtiger_crmentity INNER JOIN $this->table_name ON vtiger_crmentity.crmid=$this->table_name.$this->table_index";

        if (!empty($this->customFieldTable)) {
            $query .= " INNER JOIN " . $this->customFieldTable[0] . " ON " . $this->customFieldTable[0] . '.' . $this->customFieldTable[1] .
                " = $this->table_name.$this->table_index";
        }

        $query .= " LEFT JOIN vtiger_groups ON vtiger_groups.groupid = vtiger_crmentity.smownerid";
        $query .= " LEFT JOIN vtiger_users ON vtiger_crmentity.smownerid = vtiger_users.id and vtiger_users.status='Active'";

        $where_auto = " vtiger_crmentity.deleted=0";

        if ($where != '') {
            $query .= " WHERE ($where) AND $where_auto";
        } else {
            $query .= " WHERE $where_auto";
        }

        require('user_privileges/user_privileges_' . $current_user->id . '.php');
        require('user_privileges/sharing_privileges_' . $current_user->id . '.php');

        // Security Check for Field Access
        if ($is_admin == false && $profileGlobalPermission[1] == 1 && $profileGlobalPermission[2] == 1 && $defaultOrgSharingPermission[getTabid('Assets')] == 3) {
            //Added security check to get the permitted records only
            $query = $query . " " . getListViewSecurityParameter($thismodule);
        }
        return $query;
    }

    /**
     * Handle saving related module information.
     * NOTE: This function has been added to CRMEntity (base class).
     * You can override the behavior by re-defining it here.
     */
    // function save_related_module($module, $crmid, $with_module, $with_crmid) { }

    /**
     * Handle deleting related module information.
     * NOTE: This function has been added to CRMEntity (base class).
     * You can override the behavior by re-defining it here.
     */
    //function delete_related_module($module, $crmid, $with_module, $with_crmid) { }

    /**
     * Handle getting related list information.
     * NOTE: This function has been added to CRMEntity (base class).
     * You can override the behavior by re-defining it here.
     */
    //function get_related_list($id, $cur_tab_id, $rel_tab_id, $actions=false) { }


    /*
     * Function to get the primary query part of a report
     * @param - $module primary module name
     * returns the query string formed on fetching the related data for report for secondary module
     */
    // function generateReportsQuery($module){ }


    /*
     * Function to get the secondary query part of a report
     * @param - $module primary module name
     * @param - $secmodule secondary module name
     * returns the query string formed on fetching the related data for report for secondary module
     */
    /**
     * Transform the value while exporting
     */
    function transform_export_value($key, $value)
    {
        if ($key == 'owner') {
            return getOwnerName($value);
        }
        return parent::transform_export_value($key, $value);
    }

    // Function to unlink all the dependent entities of the given Entity by Id

    /**
     * Function which will give the basic query to find duplicates
     */
    function getDuplicatesQuery($module, $table_cols, $field_values, $ui_type_arr, $select_cols = '')
    {
        $select_clause = "SELECT " . $this->table_name . "." . $this->table_index . " AS recordid, vtiger_users_last_import.deleted," . $table_cols;

        // Select Custom Field Table Columns if present
        if (isset($this->customFieldTable)) {
            $query .= ", " . $this->customFieldTable[0] . ".* ";
        }

        $from_clause = " FROM $this->table_name";

        $from_clause .= "	INNER JOIN vtiger_crmentity ON vtiger_crmentity.crmid = $this->table_name.$this->table_index";

        // Consider custom table join as well.
        if (isset($this->customFieldTable)) {
            $from_clause .= " INNER JOIN " . $this->customFieldTable[0] . " ON " . $this->customFieldTable[0] . '.' . $this->customFieldTable[1] .
                " = $this->table_name.$this->table_index";
        }
        $from_clause .= " LEFT JOIN vtiger_users ON vtiger_users.id = vtiger_crmentity.smownerid
						LEFT JOIN vtiger_groups ON vtiger_groups.groupid = vtiger_crmentity.smownerid";

        $where_clause = "	WHERE vtiger_crmentity.deleted = 0";
        $where_clause .= $this->getListViewSecurityParameter($module);

        if (isset($select_cols) && trim($select_cols) != '') {
            $sub_query = "SELECT $select_cols FROM  $this->table_name AS t " .
                " INNER JOIN vtiger_crmentity AS crm ON crm.crmid = t." . $this->table_index;
            // Consider custom table join as well.
            if (isset($this->customFieldTable)) {
                $sub_query .= " INNER JOIN " . $this->customFieldTable[0] . " tcf ON tcf." . $this->customFieldTable[1] . " = t.$this->table_index";
            }
            $sub_query .= " WHERE crm.deleted=0 GROUP BY $select_cols HAVING COUNT(*)>1";
        } else {
            $sub_query = "SELECT $table_cols $from_clause $where_clause GROUP BY $table_cols HAVING COUNT(*)>1";
        }


        $query = $select_clause . $from_clause .
            " LEFT JOIN vtiger_users_last_import ON vtiger_users_last_import.bean_id=" . $this->table_name . "." . $this->table_index .
            " INNER JOIN (" . $sub_query . ") AS temp ON " . get_on_clause($field_values, $ui_type_arr, $module) .
            $where_clause .
            " ORDER BY $table_cols," . $this->table_name . "." . $this->table_index . " ASC";

        return $query;
    }


    function generateReportsSecQuery($module, $secmodule, $queryPlanner)
    {

        // Define the dependency matrix ahead
        $matrix = $queryPlanner->newDependencyMatrix();
        $matrix->setDependency('vtiger_crmentityCreditNotes4You', array(
            'vtiger_usersCreditNotes4You',
            'vtiger_groupsCreditNotes4You',
            'vtiger_lastModifiedByCreditNotes4You'
        ));
        $matrix->setDependency('vtiger_inventoryproductrelCreditNotes4You',
            array('vtiger_productsCreditNotes4You', 'vtiger_serviceCreditNotes4You'));

        /*if (!$queryPlanner->requireTable('vtiger_cre', $matrix)) {
            return '';
        }*/
        if (!$queryPlanner->requireTable('vtiger_creditnotes4you', $matrix)) {
            return '';
        }

        $matrix->setDependency('vtiger_creditnotes4you', array(
            'vtiger_crmentityCreditNotes4You',
            "vtiger_currency_info$secmodule",
            'vtiger_creditnotes4youcf',
            'vtiger_salesorderCreditNotes4You',
            'vtiger_invoicebillads',
            'vtiger_invoiceshipads',
            'vtiger_inventoryproductrelCreditNotes4You',
            'vtiger_contactdetailsCreditNotes4You',
            'vtiger_accountCreditNotes4You',
            'vtiger_accountRelCreditNotes4You',
            'vtiger_contactdetailsRelCreditNotes4You'
        ));

        $query = $this->getRelationQuery($module, $secmodule, "vtiger_creditnotes4you", "creditnotes4you_id",
            $queryPlanner);

        if ($queryPlanner->requireTable('vtiger_crmentityCreditNotes4You', $matrix)) {
            $query .= " left join vtiger_crmentity as vtiger_crmentityCreditNotes4You on vtiger_crmentityCreditNotes4You.crmid=vtiger_creditnotes4you.creditnotes4you_id and vtiger_crmentityCreditNotes4You.deleted=0";
        }
        if ($queryPlanner->requireTable('vtiger_creditnotes4youcf')) {
            $query .= " left join vtiger_creditnotes4youcf on vtiger_creditnotes4you.creditnotes4you_id = vtiger_creditnotes4youcf.creditnotes4you_id";
        }
        if ($queryPlanner->requireTable("vtiger_currency_info$secmodule")) {
            $query .= " left join vtiger_currency_info as vtiger_currency_info$secmodule on vtiger_currency_info$secmodule.id = vtiger_creditnotes4you.currency_id";
        }
        if ($queryPlanner->requireTable('vtiger_salesorderCreditNotes4You')) {
            $query .= " left join vtiger_salesorder as vtiger_salesorderCreditNotes4You on vtiger_salesorderCreditNotes4You.salesorderid=vtiger_creditnotes4you.salesorderid";
        }
        if ($queryPlanner->requireTable('vtiger_invoicebillads')) {
            $query .= " left join vtiger_invoicebillads on vtiger_creditnotes4you.creditnotes4you_id=vtiger_invoicebillads.invoicebilladdressid";
        }
        if ($queryPlanner->requireTable('vtiger_invoiceshipads')) {
            $query .= " left join vtiger_invoiceshipads on vtiger_creditnotes4you.creditnotes4you_id=vtiger_invoiceshipads.invoiceshipaddressid";
        }
        if ($queryPlanner->requireTable('vtiger_inventoryproductrelCreditNotes4You', $matrix)) {
        }
        if ($queryPlanner->requireTable('vtiger_productsCreditNotes4You')) {
            $query .= " left join vtiger_products as vtiger_productsCreditNotes4You on vtiger_productsCreditNotes4You.productid = vtiger_inventoryproductreltmpCreditNotes4You.productid";
        }
        if ($queryPlanner->requireTable('vtiger_serviceCreditNotes4You')) {
            $query .= " left join vtiger_service as vtiger_serviceCreditNotes4You on vtiger_serviceCreditNotes4You.serviceid = vtiger_inventoryproductreltmpCreditNotes4You.productid";
        }
        if ($queryPlanner->requireTable('vtiger_groupsCreditNotes4You')) {
            $query .= " left join vtiger_groups as vtiger_groupsCreditNotes4You on vtiger_groupsCreditNotes4You.groupid = vtiger_crmentityCreditNotes4You.smownerid";
        }
        if ($queryPlanner->requireTable('vtiger_usersCreditNotes4You')) {
            $query .= " left join vtiger_users as vtiger_usersCreditNotes4You on vtiger_usersCreditNotes4You.id = vtiger_crmentityCreditNotes4You.smownerid";
        }
        if ($queryPlanner->requireTable('vtiger_contactdetailsCreditNotes4You')) {
            $query .= " left join vtiger_contactdetails as vtiger_contactdetailsCreditNotes4You on vtiger_creditnotes4you.contactid = vtiger_contactdetailsCreditNotes4You.contactid";
        }
        if ($queryPlanner->requireTable('vtiger_accountCreditNotes4You')) {
            $query .= " left join vtiger_account as vtiger_accountCreditNotes4You on vtiger_accountCreditNotes4You.accountid = vtiger_creditnotes4you.accountid";
        }
        if ($queryPlanner->requireTable('vtiger_accountRelCreditNotes4You')) {
            $query .= " left join vtiger_account as vtiger_accountRelCreditNotes4You on vtiger_accountRelCreditNotes4You.accountid = vtiger_creditnotes4you.accountid";
        }
        if ($queryPlanner->requireTable('vtiger_contactdetailsRelCreditNotes4You')) {
            $query .= " left join vtiger_contactdetails as vtiger_contactdetailsRelCreditNotes4You on vtiger_creditnotes4you.contactid = vtiger_contactdetailsRelCreditNotes4You.contactid";
        }
        if ($queryPlanner->requireTable('vtiger_lastModifiedByCreditNotes4You')) {
            $query .= " left join vtiger_users as vtiger_lastModifiedByCreditNotes4You on vtiger_lastModifiedByCreditNotes4You.id = vtiger_crmentityCreditNotes4You.modifiedby ";
        }
        if ($queryPlanner->requireTable("vtiger_createdbyCreditNotes4You")) {
            $query .= " left join vtiger_users as vtiger_createdbyCreditNotes4You on vtiger_createdbyCreditNotes4You.id = vtiger_crmentityCreditNotes4You.smcreatorid ";
        }

        //if secondary modules custom reference field is selected
        $query .= parent::getReportsUiType10Query($secmodule, $queryPlanner);

        $query .= $this->getReferenceQuery($module, $queryPlanner);

        return $query;
    }

    public function getReferenceQuery($module, $queryPlanner)
    {
        $relatedModules = [
            'Accounts' => 'account_id',
            'Contacts' => 'contact_id',
        ];
        $moduleInstance = Vtiger_Module::getInstance($module);

        foreach ($relatedModules as $relModule => $field) {
            $fieldInstance = Vtiger_Field::getInstance($field, $moduleInstance);
            $relModuleInstance = Vtiger_Module::getInstance($relModule);

            if ($fieldInstance && $relModuleInstance) {
                $relTable = $relModuleInstance->basetable . 'Rel' . $moduleInstance->name . $fieldInstance->id;

                if ($queryPlanner->requireTable($relTable) && 0 < stripos($query, 'as ' . $relTable . ' on')) {
                    $query .= " left join $relModuleInstance->basetable as $relTable on $relTable.$relModuleInstance->basetableid = $moduleInstance->basetable.$fieldInstance->column ";
                }
            }
        }

        return $query;
    }

    public function generateReportsQuery($module, $queryPlanner)
    {
        $query = parent::generateReportsQuery($module, $queryPlanner);

        if ($queryPlanner->requireTable('vtiger_currency_infoCreditNotes4You')) {
            $query .= " left join vtiger_currency_info as vtiger_currency_infoCreditNotes4You on vtiger_currency_infoCreditNotes4You.id = vtiger_creditnotes4you.currency_id ";
        }

        $query .= $this->getReferenceQuery($module, $queryPlanner);

        return $query;
    }

    function unlinkDependencies($module, $id)
    {
        global $log;
        parent::unlinkDependencies($module, $id);
    }

    /**
     * Invoked when special actions are performed on the module.
     * @param String Module name
     * @param String Event Type
     */
    function vtlib_handler($modulename, $event_type)
    {
        include_once 'vtlib/Vtiger/Module.php';
        include_once 'modules/ModComments/ModComments.php';
        include_once 'modules/ModTracker/ModTracker.php';

        switch ($event_type) {
            case 'module.postupdate':
            case 'module.postinstall':
            case 'module.enabled':
                $this->setVtigerCompatible();
                $this->addCustomLinks();
                self::addWofkflowForCn($this->moduleName, $event_type);
                self::activateWofkflowForCn($this->moduleName, $event_type);
                break;
            case 'module.preuninstall':
            case 'module.disabled':
            case 'module.preupdate':
                $this->deleteCustomLinks();
                self::deactivateWofkflowForCn($this->moduleName, $event_type);
                break;
        }
    }

    public function updateNumbering()
    {
        $this->setModuleSeqNumber('configure', $this->moduleName, 'CN', 1);
        $this->updateMissingSeqNumber($this->moduleName);
    }

    /**
     * @var array
     * [Module, RelatedModule, RelatedLabel, RelatedActions, RelatedFunction]
     */
    public $registerRelatedLists = array(
        ['CreditNotes4You', 'Documents', 'Documents', 'ADD,SELECT', 'get_attachments'],
        ['Invoice', 'CreditNotes4You', 'CreditNotes4You', '', 'get_dependents_list', 'invoice_id'],
        ['Accounts', 'CreditNotes4You', 'CreditNotes4You', '', 'get_dependents_list', 'account_id'],
        ['Contacts', 'CreditNotes4You', 'CreditNotes4You', '', 'get_dependents_list', 'contact_id'],
    );

    /**
     * [module, type, label, url, icon, sequence, handlerInfo]
     * @return array
     */
    public $registerCustomLinks = array(
        ['Invoice', 'DETAILVIEW', 'Generate Credit Note', 'index.php?module=CreditNotes4You&action=EditView&return_module=$MODULE$&return_action=DetailView&return_id=$RECORD$&convertmode=invtocn'],
    );

    /**
     * @param bool $register
     */
    public function updateRelatedList($register = true)
    {
        foreach ($this->registerRelatedLists as $relatedList) {
            $module = Vtiger_Module::getInstance($relatedList[0]);
            $relatedModule = Vtiger_Module::getInstance($relatedList[1]);

            if ($module && $relatedModule) {
                $relatedLabel = isset($relatedList[2]) ? $relatedList[2] : $relatedModule->name;
                $relatedActions = isset($relatedList[3]) ? $relatedList[3] : '';
                $relatedFunction = isset($relatedList[4]) ? $relatedList[4] : 'get_related_list';
                $field = isset($relatedList[5]) ? Vtiger_Field_Model::getInstance($relatedList[5], $relatedModule) : '';
                $fieldId = $field ? $field->getId() : '';

                $module->unsetRelatedList($relatedModule, $relatedLabel);
                $module->unsetRelatedList($relatedModule, $relatedLabel, $relatedFunction);

                if ($register) {
                    $module->setRelatedList($relatedModule, $relatedLabel, $relatedActions, $relatedFunction, $fieldId);
                }
            }
        }
    }

    public static function setVtigerCompatible()
    {
        $layout = Vtiger_Viewer::getDefaultLayoutName();

        if ('v7' === $layout) {
            $db = PearDatabase::getInstance();

            $modulename = self::MODULE_NAME;
            $table_name = 'vtiger_creditnotes4you';

            $inventoryModules = array($modulename);

            $actions = array('Import', 'Export');

            // add missing inventory block and fields !- 540 - 600
            for ($i = 0; $i < count($inventoryModules); $i++) {
                $moduleName = $inventoryModules[$i];
                $moduleInstance = Vtiger_Module::getInstance($moduleName);

                $result = $db->pquery('SELECT 1 FROM vtiger_blocks WHERE tabid=? AND blocklabel=?',
                    array(getTabid($moduleName), 'LBL_ITEM_DETAILS'));
                if (!($db->num_rows($result))) {
                    $blockInstance = new Vtiger_Block();

                    $blockInstance->label = 'LBL_ITEM_DETAILS';
                    $blockInstance->sequence = '5';
                    $blockInstance->showtitle = '0';

                    $moduleInstance->addBlock($blockInstance);

                    foreach ($actions as $actionName) {
                        Vtiger_Access::updateTool($moduleInstance, $actionName, true, '');
                    }
                }
            }

            $itemFieldsName = array(
                'productid',
                'quantity',
                'listprice',
                'comment',
                'discount_amount',
                'discount_percent',
                'tax1',
                'tax2',
                'tax3'
            );
            $itemFieldsLabel = array(
                'Item Name',
                'Quantity',
                'List Price',
                'Item Comment',
                'Item Discount Amount',
                'Item Discount Percent',
                'Tax1',
                'Tax2',
                'Tax3'
            );
            $itemFieldsTypeOfData = array('V~M', 'V~M', 'V~M', 'V~O', 'V~O', 'V~O', 'V~O', 'V~O', 'V~O');
            $itemFieldsDisplayType = array('10', '7', '19', '19', '7', '7', '83', '83', '83');

            for ($i = 0; $i < count($inventoryModules); $i++) {
                $moduleName = $inventoryModules[$i];
                $moduleInstance = Vtiger_Module::getInstance($moduleName);
                $blockInstance = Vtiger_Block::getInstance('LBL_ITEM_DETAILS', $moduleInstance);

                $relatedmodules = array('Products', 'Services');

                for ($j = 0; $j < count($itemFieldsName); $j++) {
                    $field = new Vtiger_Field();

                    $field->name = $itemFieldsName[$j];
                    $field->label = $itemFieldsLabel[$j];
                    $field->column = $itemFieldsName[$j];
                    $field->table = 'vtiger_inventoryproductrel';
                    $field->uitype = $itemFieldsDisplayType[$j];
                    $field->typeofdata = $itemFieldsTypeOfData[$j];
                    $field->readonly = '0';
                    $field->displaytype = '5';
                    $field->masseditable = '0';

                    $blockInstance->addField($field);

                    if ($itemFieldsName[$j] == 'productid') {
                        $field->setRelatedModules($relatedmodules);
                    }
                }
            }

            $inventoryModuleBlocks = array(
                $modulename => array('LBL_CREDITNOTES4YOU_INFO', 'vtiger_creditnotes4you', 'creditnotes4you_id')
            );

            foreach ($inventoryModuleBlocks as $module => $details) {
                $tableName = $details[1];
                $moduleInstance = Vtiger_Module::getInstance($module);
                $block = Vtiger_Block::getInstance($details[0], $moduleInstance);

                $result = $db->pquery('SELECT 1 FROM vtiger_field WHERE tabid=? AND columnname=?',
                    array(getTabid($moduleName), 'pre_tax_total'));
                if (!($db->num_rows($result))) {
                    $preTaxTotalField = new Vtiger_Field();
                    $preTaxTotalField->name = 'pre_tax_total';
                    $preTaxTotalField->label = 'Pre Tax Total';
                    $preTaxTotalField->table = $tableName;
                    $preTaxTotalField->column = 'pre_tax_total';
                    $preTaxTotalField->columntype = 'decimal(25,8)';
                    $preTaxTotalField->typeofdata = 'N~O';
                    $preTaxTotalField->uitype = '72';
                    $preTaxTotalField->masseditable = '1';
                    $preTaxTotalField->displaytype = '3';
                    $block->addField($preTaxTotalField);
                }

                $tableId = $details[2];

                $result = $db->pquery("SELECT $tableId, subtotal, s_h_amount, discount_percent, discount_amount FROM $tableName",
                    array());
                $numOfRows = $db->num_rows($result);

                for ($i = 0; $i < $numOfRows; $i++) {
                    $id = $db->query_result($result, $i, $tableId);
                    $subTotal = (float)$db->query_result($result, $i, "subtotal");
                    $shAmount = (float)$db->query_result($result, $i, "s_h_amount");
                    $discountAmount = (float)$db->query_result($result, $i, "discount_amount");
                    $discountPercent = (float)$db->query_result($result, $i, "discount_percent");

                    if ($discountPercent != '0') {
                        $discountAmount = ($subTotal * $discountPercent) / 100;
                    }
                    $preTaxTotalValue = $subTotal + $shAmount - $discountAmount;

                    $db->pquery("UPDATE $tableName set pre_tax_total = ? WHERE $tableId = ?",
                        array($preTaxTotalValue, $id));
                }
            }

            $modulesList = array($moduleName => $table_name);
            $fieldName = 'region_id';

            foreach ($modulesList as $moduleName => $tableName) {
                //Updating existing inventory tax tables
                $columns = $db->getColumnNames($tableName);
                if (!in_array('compound_taxes_info', $columns)) {
                    $db->pquery("ALTER TABLE $tableName ADD COLUMN compound_taxes_info TEXT", array());
                }
                $db->pquery('UPDATE ' . $tableName . ' SET compound_taxes_info=?', array('[]'));

                //creating new field in entity tables
                $moduleInstance = Vtiger_Module::getInstance($moduleName);
                $blockInstance = Vtiger_Block::getInstance('LBL_ITEM_DETAILS', $moduleInstance);

                $fieldInstance = Vtiger_Field::getInstance($fieldName, $moduleInstance);
                if (!$fieldInstance) {
                    $fieldInstance = new Vtiger_Field();

                    $fieldInstance->name = $fieldName;
                    $fieldInstance->column = $fieldName;
                    $fieldInstance->table = $tableName;
                    $fieldInstance->label = 'Tax Region';
                    $fieldInstance->columntype = 'int(19)';
                    $fieldInstance->typeofdata = 'N~O';
                    $fieldInstance->uitype = '16';
                    $fieldInstance->readonly = '0';
                    $fieldInstance->displaytype = '5';
                    $fieldInstance->masseditable = '0';

                    $blockInstance->addField($fieldInstance);
                }
            }
            //End: Tax Enhancements - Compound Taxes, Regional Taxes, Deducted Taxes, Other Charges

            // vtmigrate 600 - 700
            foreach ($inventoryModules as $moduleName) {
                $tabId = getTabid($moduleName);
                $blockId = getBlockId($tabId, 'LBL_ITEM_DETAILS');
                $db->pquery('UPDATE vtiger_field SET displaytype=?, block=? WHERE tabid=? AND fieldname IN (?, ?)',
                    array(5, $blockId, $tabId, 'hdnDiscountAmount', 'hdnDiscountPercent'));
            }

            $itemFieldsName = array('image', 'purchase_cost', 'margin');
            $itemFieldsLabel = array('Image', 'Purchase Cost', 'Margin');
            $itemFieldsTypeOfData = array('V~O', 'N~O', 'N~O');
            $itemFieldsDisplayType = array('56', '71', '71');
            $itemFieldsDataType = array('VARCHAR(2)', 'decimal(27,8)', 'decimal(27,8)');

            $fieldIdsList = array();
            foreach ($inventoryModules as $moduleName) {
                $moduleInstance = Vtiger_Module::getInstance($moduleName);
                $blockInstance = Vtiger_Block::getInstance('LBL_ITEM_DETAILS', $moduleInstance);

                for ($i = 0; $i < count($itemFieldsName); $i++) {
                    $fieldName = $itemFieldsName[$i];

                    if ($moduleName === 'PurchaseOrder' && $fieldName !== 'image') {
                        continue;
                    }

                    $fieldInstance = Vtiger_Field::getInstance($fieldName, $moduleInstance);
                    if (!$fieldInstance) {
                        $fieldInstance = new Vtiger_Field();

                        $fieldInstance->name = $fieldName;
                        $fieldInstance->column = $fieldName;
                        $fieldInstance->label = $itemFieldsLabel[$i];
                        $fieldInstance->columntype = $itemFieldsDataType[$i];
                        $fieldInstance->typeofdata = $itemFieldsTypeOfData[$i];
                        $fieldInstance->uitype = $itemFieldsDisplayType[$i];
                        $fieldInstance->table = 'vtiger_inventoryproductrel';
                        $fieldInstance->presence = '1';
                        $fieldInstance->readonly = '0';
                        $fieldInstance->displaytype = '5';
                        $fieldInstance->masseditable = '0';

                        $blockInstance->addField($fieldInstance);
                        $fieldIdsList[] = $fieldInstance->id;
                    }
                }
            }
        }
        return true;
    }

    public function addCustomLinks()
    {
        $this->updateNumbering();
        $this->updateFields();
        $this->updateTables();
        $this->updateRelatedList();
        $this->updateCustomLinks();
        $this->updateMenuLink();

        ModComments::addWidgetTo([$this->moduleName]);
        ModTracker::enableTrackingForModule(getTabid($this->moduleName));
    }

    public function updateMenuLink()
    {
        if (method_exists('Settings_MenuEditor_Module_Model', 'addModuleToApp')) {
            Settings_MenuEditor_Module_Model::addModuleToApp($this->moduleName, $this->parentName);
        }
    }

    public function updateTables()
    {
        $fields = [
            'pre_tax_total' => 'ALTER TABLE vtiger_creditnotes4you ADD pre_tax_total decimal(25,8) DEFAULT NULL',
            'contactid' => 'ALTER TABLE vtiger_creditnotes4you ADD contactid int(19) DEFAULT NULL',
            's_h_percent' => 'ALTER TABLE vtiger_creditnotes4you ADD s_h_percent DECIMAL (25,3) NOT NULL DEFAULT 0.000;',
            'terms_conditions' => 'ALTER TABLE vtiger_creditnotes4you ADD terms_conditions TEXT',
        ];

        foreach ($fields as $field => $sql) {
            preg_match('/ALTER\ TABLE\ ([a-z0-9\_]+)\ ADD/', $sql, $matches);

            if ($matches[1] && !columnExists($field, $matches[1])) {
                $this->db->pquery($sql);
            }
        }
    }

    public function deleteContactId()
    {
        $this->db->pquery('DELETE FROM vtiger_field WHERE tablename=? AND fieldname=?', ['vtiger_creditnotes4you', 'contactid']);
    }

    public function updateItemDetailFields()
    {
        $fieldNames = array('txtAdjustment', 'hdnSubTotal', 'hdnGrandTotal', 'hdnDiscountAmount', 'hdnS_H_Amount', 'pre_tax_total');
        $this->db->pquery('UPDATE vtiger_field SET uitype=? WHERE tabid=? AND fieldname IN(' . generateQuestionMarks($fieldNames) . ')',
            array(72, getTabid($this->moduleName), $fieldNames)
        );

        $module = Vtiger_Module_Model::getInstance($this->moduleName);
        $block = Vtiger_Block_Model::getInstance('LBL_ITEM_DETAILS', $module);

        if ($module && $block) {
            $blockId = $block->get('id');
            $this->db->pquery('UPDATE vtiger_field SET block=? WHERE tabid=? AND fieldname IN (?,?) AND block!=?',
                [$blockId, $module->getId(), 'currency_id', 'conversion_rate', $blockId]
            );
            $this->db->pquery('UPDATE vtiger_field SET masseditable=? WHERE block=?',
                [0, $blockId]
            );
        }
    }

    public function updateFields()
    {
        $this->deleteContactId();
        $this->updateItemDetailFields();
        $this->updateReferenceFields();

        $moduleInstance = Vtiger_Module::getInstance($this->moduleName);
        $blockInstance = Vtiger_Block::getInstance('LBL_ITEM_DETAILS', $moduleInstance);
        $this->db->pquery('UPDATE vtiger_field SET block=? WHERE tablename=? AND fieldname=?', [$blockInstance->id, 'vtiger_creditnotes4you', 'hdnS_H_Percent']);
    }

    public function updateReferenceFields()
    {
        $this->db->pquery('UPDATE vtiger_field SET uitype=? WHERE tablename=? AND fieldname=?', [10, 'vtiger_creditnotes4you', 'contact_id']);
        $this->db->pquery('UPDATE vtiger_field SET uitype=? WHERE tablename=? AND fieldname=?', [10, 'vtiger_creditnotes4you', 'account_id']);

        $this->setRelatedModule('contact_id', 'Contacts');
        $this->setRelatedModule('account_id', 'Accounts');
    }

    public function setRelatedModule($fieldName, $moduleName)
    {
        $module = Vtiger_Module_Model::getInstance($this->moduleName);
        $field = Vtiger_Field_Model::getInstance($fieldName, $module);

        if ($field) {
            $field->setRelatedModules([$moduleName]);
        }
    }

    /**
     * function to add Inactive Inventory Workflow for Credit Notes Module
     *
     * @param $moduleName
     * @param $event_type
     *
     * @return bool
     */
    public static function addWofkflowForCn($moduleName, $event_type)
    {
        $adb = PearDatabase::getInstance();

        vimport("~~modules/com_vtiger_workflow/include.inc");
        vimport("~~modules/com_vtiger_workflow/tasks/VTEntityMethodTask.inc");
        vimport("~~modules/com_vtiger_workflow/VTEntityMethodManager.inc");

        $moduleName = self::MODULE_NAME;

        $sql = 'SELECT workflow_id FROM com_vtiger_workflows WHERE module_name = ? AND workflowname LIKE "%Update Inventory Products On Refund%"';
        $result = $adb->pquery($sql, [$moduleName]);
        $noofrows = $adb->num_rows($result);
        if (1 > $noofrows) {

            $em = new VTEventsManager($adb);
            $em->registerHandler('vtiger.entity.beforesave',
                'modules/' . $moduleName . '/handlers/CreditNotes4YouHandler.php', 'CreditNotes4YouHandler');

            $emm = new VTEntityMethodManager($adb);
            $emm->addEntityMethod($moduleName, 'UpdateInventoryRefund',
                'modules/' . $moduleName . '/handlers/InventoryHandler.php', 'handleInventoryProductRelCreditNotes');

            $vtWorkFlow = new VTWorkflowManager($adb);
            $newWorkFlow = $vtWorkFlow->newWorkFlow($moduleName);
            $newWorkFlow->name = 'Update Inventory Products On Refund';
            $newWorkFlow->description = 'Update Inventory Products On Refund';
            $newWorkFlow->defaultworkflow = 0;
            $newWorkFlow->executionCondition = 3;
            $vtWorkFlow->save($newWorkFlow);

            $tm = new VTTaskManager($adb);
            $task = $tm->createTask('VTEntityMethodTask', $newWorkFlow->id);
            $task->active = 1;
            $task->summary = 'Update Inventory Products on Refund';
            $task->methodName = 'UpdateInventoryRefund';
            $tm->saveTask($task);

            self::activateWofkflowForCn($moduleName, $event_type);
        }

        return true;
    }

    /**
     * function to add activate Inventory Workflow for Credit Notes Module
     * and deactivate Inventory Workflow for Invoice Module
     *
     * @param $moduleName
     * @param $event_type
     *
     * @return bool
     */
    public static function activateWofkflowForCn($moduleName, $event_type)
    {
        self::changeWorkflowStatus(1);

        return true;
    }

    /**
     * function to change Invoice and CreditNotes Inventory update workflow status
     *
     * @param $creditNoteWfStatus
     *
     * @return bool
     */
    private static function changeWorkflowStatus($creditNoteWfStatus)
    {
        $adb = PearDatabase::getInstance();

        $result = $adb->pquery('SELECT workflow_id, module_name 
                          FROM com_vtiger_workflows 
                          WHERE summary LIKE "%Update Inventory Products On Refund%" 
                          AND module_name = ?', [
            self::MODULE_NAME,
        ]);
        if ($result) {
            while ($row = $adb->fetchByAssoc($result)) {
                $comWorkflowId = $row['workflow_id'];
                Settings_Workflows_Record_Model::updateWorkflowStatus($comWorkflowId, $creditNoteWfStatus);
            }
        }

        return true;
    }

    /**
     * @param bool $register
     */
    public function updateCustomLinks($register = true)
    {
        foreach ($this->registerCustomLinks as $customLink) {
            $module = Vtiger_Module::getInstance($customLink[0]);
            $type = $customLink[1];
            $label = $customLink[2];

            if($module) {
                $module->deleteLink($type, $label);

                if($register) {
                    $module->addLink($type, $label, $customLink[3], $customLink[4], $customLink[5], $customLink[6]);
                }
            }
        }
    }

    public function deleteCustomLinks()
    {
        $this->clearCustomLinks();
        $this->clearRelatedLists();
        $this->updateCustomLinks(false);
        $this->updateRelatedList(false);

        ModComments::removeWidgetFrom([$this->moduleName]);
        ModTracker::disableTrackingForModule(getTabid($this->moduleName));
    }

    public function clearCustomLinks()
    {
        Vtiger_Link::deleteLink(getTabid('Invoice'), 'DETAILVIEWBASIC', 'Generate Credit Note');
        Vtiger_Link::deleteLink(getTabid('Invoice'), 'DETAILVIEW', 'Generate Credit Note');
    }

    public function clearRelatedLists()
    {
        $moduleName = 'CreditNotes4You';
        $linked_module = Vtiger_Module::getInstance($moduleName);

        foreach ($this->relatedModulesList as $relatedModule) {
            $for_module = Vtiger_Module::getInstance($relatedModule);

            if ($for_module) {
                $for_module->unsetRelatedList($linked_module, $moduleName);
                $for_module->unsetRelatedList($linked_module, $moduleName, 'get_dependents_list');
            }
        }
    }

    /**
     * function to add activate Inventory Workflow for Invoice Module
     * and deactivate Inventory Workflow for Credit Notes Module
     *
     * @param $moduleName
     * @param $event_type
     *
     * @return bool
     */
    public static function deactivateWofkflowForCn($moduleName, $event_type)
    {
        self::changeWorkflowStatus(0);

        return true;
    }

    /**
     * Move the related records of the specified list of id's to the given record.
     *
     * @param $module String This module name
     * @param $transferEntityIds Array List of Entity Id's from which related records need to be transfered
     * @param $entityId Integer Id of the the Record to which the related records are to be moved
     *
     * @throws Exception
     */
    function transferRelatedRecords($module, $transferEntityIds, $entityId)
    {
        global $adb, $log;
        $log->debug("Entering function transferRelatedRecords ($module, $transferEntityIds, $entityId)");

        $rel_table_arr = array("Documents" => "vtiger_senotesrel", "Attachments" => "vtiger_seattachmentsrel");

        $tbl_field_arr = array("vtiger_senotesrel" => "notesid", "vtiger_seattachmentsrel" => "attachmentsid");

        $entity_tbl_field_arr = array("vtiger_senotesrel" => "crmid", "vtiger_seattachmentsrel" => "crmid");

        foreach ($transferEntityIds as $transferId) {
            foreach ($rel_table_arr as $rel_module => $rel_table) {
                $id_field = $tbl_field_arr[$rel_table];
                $entity_id_field = $entity_tbl_field_arr[$rel_table];
                // IN clause to avoid duplicate entries
                $sel_result = $adb->pquery("select $id_field from $rel_table where $entity_id_field=? " .
                    " and $id_field not in (select $id_field from $rel_table where $entity_id_field=?)",
                    array($transferId, $entityId));
                $res_cnt = $adb->num_rows($sel_result);
                if ($res_cnt > 0) {
                    for ($i = 0; $i < $res_cnt; $i++) {
                        $id_field_value = $adb->query_result($sel_result, $i, $id_field);
                        $adb->pquery("update $rel_table set $entity_id_field=? where $entity_id_field=? and $id_field=?",
                            array($entityId, $transferId, $id_field_value));
                    }
                }
            }
        }
        parent::transferRelatedRecords($module, $transferEntityIds, $entityId);
        $log->debug("Exiting transferRelatedRecords...");
    }

    /**
     * Customizing the restore procedure.
     */
    function restore($module, $recordId)
    {

        $cnStatus = self::getStatusById($module, $recordId);

        parent::restore($module, $recordId);

        if ('Refunded' === $cnStatus) {
            CreditNotes4You_ProductsUpdate_Helper::updateInventoryProductsByStatus($recordId, $cnStatus);
        }
    }

    /**
     * @param $moduleName
     * @param $recordId
     *
     * @return string
     */
    private static function getStatusById($moduleName, $recordId)
    {
        $status = '';
        if (self::MODULE_NAME === $moduleName && !empty($recordId)) {
            $db = PearDatabase::getInstance();
            $f = CRMEntity::getInstance($moduleName);
            $query = sprintf('SELECT %s FROM %s WHERE %s = ?', CreditNotes4You::$statusFieldName, $f->table_name,
                $f->table_index);
            $result = $db->pquery($query, [$recordId]);
            if (0 < $db->num_rows($result)) {
                $row = $db->fetchByAssoc($result, 0);
                $status = $row[CreditNotes4You::$statusFieldName];
            }
        }

        return $status;
    }

    /**
     * Customizing the Delete procedure.
     */
    function trash($module, $recordId)
    {

        $cnStatus = self::getStatusById($module, $recordId);
        if ('Refunded' === $cnStatus) {
            CreditNotes4You_ProductsUpdate_Helper::updateInventoryProductsByStatus($recordId, 'Created');
        }

        parent::trash($module, $recordId);
    }

    /**
     * Function to remove module tables when uninstalling module
     */
    private function dropModuleTables()
    {
        $this->db->query("DROP TABLE IF EXISTS vtiger_creditnotes4you");
        $this->db->query("DROP TABLE IF EXISTS vtiger_creditnotes4youcf");
        global $currentModule;
        require_once("modules/" . $currentModule . "/ITS4YouLicensing.php");
        $this->db->query("DROP TABLE IF EXISTS " . ITS4YouLicensing::license_table);
        $this->db->query("DROP TABLE IF EXISTS " . ITS4YouLicensing::version_table);
    }

}