<?php
/* * *******************************************************************************
 * The content of this file is subject to the ITS4YouImport license.
 * ("License"); You may not use this file except in compliance with the License
 * The Initial Developer of the Original Code is IT-Solutions4You s.r.o.
 * Portions created by IT-Solutions4You s.r.o. are Copyright(C) IT-Solutions4You s.r.o.
 * All Rights Reserved.
 * ****************************************************************************** */

class ITS4YouImport_Queue_Action extends Import_Queue_Action
{

    public static function add($request, $user)
    {
        $request->set("module", $request->get("targetModule"));
        $db = PearDatabase::getInstance();

        if (!Vtiger_Utils::CheckTable('its4you_import_queue')) {
            Vtiger_Utils::CreateTable(
                'its4you_import_queue',
                "(importid INT NOT NULL PRIMARY KEY,
				userid INT NOT NULL,
				tabid INT NOT NULL,
				field_mapping TEXT,
				default_values TEXT,
				merge_type INT(1),
				merge_fields TEXT,
				status INT(1) default 0,
                trigger_wf INT(1),
                create_picklist_vals INT(1),
                import_entity_id INT,
                source_record INT default 0,
                clear_pricebooks INT(1) )",
                true
            );
        }

        if ($request->get('is_scheduled')) {
            $status = self::$IMPORT_STATUS_SCHEDULED;
            $importEntityStatus = ITS4YouImport_Module_Model::$FIELD_IMPORT_STATUS_SCHEDULED;
        } else {
            $status = self::$IMPORT_STATUS_NONE;
            $importEntityStatus = ITS4YouImport_Module_Model::$FIELD_IMPORT_STATUS_NONE;
        }

        if ($request->get("trigger_workflow") === "on") {
            $trigger_wf = '1';
        } else {
            $trigger_wf = '0';
        }

        if ($request->get("create_picklist_values") === "on") {
            $create_picklist_vals = '1';
        } else {
            $create_picklist_vals = '0';
        }

        if ($request->get("clear_pricebooks") === "on") {
            $clear_pricebooks = '1';
        } else {
            $clear_pricebooks = '0';
        }

        $field_mapping = $request->get('field_mapping');
        $targetModule = $request->get('targetModule');
        $sourceRecord = $request->get('sourceRecord');

        if ('PriceBooks' === $targetModule && isRecordExists($sourceRecord)) {
            $priceBook = array(
                'services' => $request->get('services_identifier'),
                'products' => $request->get('products_identifier'),
            );
            $field_mapping = is_array($field_mapping) ? $field_mapping : json_encode($field_mapping);
            $field_mapping = array_merge($field_mapping, $priceBook);
        } else {
            $sourceRecord = 0;
        }

        $db->pquery('INSERT INTO its4you_import_queue VALUES(?,?,?,?,?,?,?,?,?,?,?,?,?)', array(
                $db->getUniqueID('its4you_import_queue'),
                $user->id,
                getTabid($request->get('module')),
                Zend_Json::encode($field_mapping),
                Zend_Json::encode($request->get('default_values')),
                $request->get('merge_type'),
                Zend_Json::encode($request->get('merge_fields')),
                $status,
                $trigger_wf,
                $create_picklist_vals,
                $request->get("import_entity_id"),
                $sourceRecord,
                $clear_pricebooks
            )
        );

        ITS4YouImport_Record_Model::updateStatus($request->get("import_entity_id"), $importEntityStatus);
    }

    public static function remove($importId)
    {
        self::updateImportEntityStatus($importId, ITS4YouImport_Module_Model::$FIELD_IMPORT_STATUS_COMPLETED);

        $db = PearDatabase::getInstance();
        if (Vtiger_Utils::CheckTable('its4you_import_queue')) {
            $db->pquery('DELETE FROM its4you_import_queue WHERE importid=?', array($importId));
        }
    }

    private static function updateImportEntityStatus($importId, $importEntityStatus)
    {
        $importEntityId = self::getImportEntityId($importId);
        if ($importEntityId) {
            ITS4YouImport_Record_Model::updateStatus($importEntityId, $importEntityStatus);
        }
    }

    private static function getImportEntityId($importId)
    {
        $db = PearDatabase::getInstance();
        $queueResult = $db->pquery("SELECT import_entity_id FROM its4you_import_queue WHERE importid = ?", array($importId));
        if ($db->num_rows($queueResult) > 0) {
            return $db->query_result($queueResult, 0, "import_entity_id");
        }
        return false;
    }

    public static function removeForUser($user)
    {
        $db = PearDatabase::getInstance();
        if (Vtiger_Utils::CheckTable('its4you_import_queue')) {
            $db->pquery('DELETE FROM its4you_import_queue WHERE userid=?', array($user->id));
        }
    }

    public static function getUserCurrentImportInfo($user)
    {
        $db = PearDatabase::getInstance();

        if (Vtiger_Utils::CheckTable('its4you_import_queue')) {
            $queueResult = $db->pquery('SELECT * FROM its4you_import_queue WHERE userid=? LIMIT 1', array($user->id));

            if ($queueResult && $db->num_rows($queueResult) > 0) {
                $rowData = $db->raw_query_result_rowdata($queueResult, 0);
                return self::getImportInfoFromResult($rowData);
            }
        }
        return null;
    }

    public static function getImportInfoFromResult($rowData)
    {
        return array(
            'id' => $rowData['importid'],
            'module' => getTabModuleName($rowData['tabid']),
            'field_mapping' => Zend_Json::decode($rowData['field_mapping']),
            'default_values' => Zend_Json::decode($rowData['default_values']),
            'merge_type' => $rowData['merge_type'],
            'merge_fields' => Zend_Json::decode($rowData['merge_fields']),
            'user_id' => $rowData['userid'],
            'status' => $rowData['status'],
            'trigger_workflow' => $rowData['trigger_wf'] == '1' ? true : false,
            'create_picklist_values' => $rowData['create_picklist_vals'] == '1' ? true : false,
            'import_entity_id' => $rowData['import_entity_id'],
            'source_record' => $rowData['source_record'],
            'clear_pricebooks' => $rowData['clear_pricebooks'],
        );
    }

    public static function getImportInfo($module, $user)
    {
        $db = PearDatabase::getInstance();

        if (Vtiger_Utils::CheckTable('its4you_import_queue')) {
            $queueResult = $db->pquery('SELECT * FROM its4you_import_queue WHERE tabid=? AND userid=?', array(getTabid($module), $user->id));

            if ($queueResult && $db->num_rows($queueResult) > 0) {
                $rowData = $db->raw_query_result_rowdata($queueResult, 0);
                return self::getImportInfoFromResult($rowData);
            }
        }
        return null;
    }

    public static function getImportInfoById($importId)
    {
        $db = PearDatabase::getInstance();

        if (Vtiger_Utils::CheckTable('its4you_import_queue')) {
            $queueResult = $db->pquery('SELECT * FROM its4you_import_queue WHERE importid=?', array($importId));

            if ($queueResult && $db->num_rows($queueResult) > 0) {
                $rowData = $db->raw_query_result_rowdata($queueResult, 0);
                return self::getImportInfoFromResult($rowData);
            }
        }
        return null;
    }

    public static function getAll($status = false)
    {
        $db = PearDatabase::getInstance();

        $query = 'SELECT * FROM its4you_import_queue';
        $params = array();
        if ($status !== false) {
            $query .= ' WHERE status = ?';
            array_push($params, $status);
        }
        $result = $db->pquery($query, $params);

        $noOfImports = $db->num_rows($result);
        $scheduledImports = array();
        for ($i = 0; $i < $noOfImports; ++$i) {
            $rowData = $db->raw_query_result_rowdata($result, $i);
            $scheduledImports[$rowData['importid']] = self::getImportInfoFromResult($rowData);
        }

        return $scheduledImports;
    }

    public static function updateStatus($importId, $status)
    {
        $db = PearDatabase::getInstance();
        $db->pquery('UPDATE its4you_import_queue SET status=? WHERE importid=?', array($status, $importId));

        switch ($status) {
            case self::$IMPORT_STATUS_RUNNING:
                $importEntityStatus = ITS4YouImport_Module_Model::$FIELD_IMPORT_STATUS_RUNNING;
                break;

            case self::$IMPORT_STATUS_COMPLETED:
                $importEntityStatus = ITS4YouImport_Module_Model::$FIELD_IMPORT_STATUS_COMPLETED;
                break;

            case self::$IMPORT_STATUS_HALTED:
                $importEntityStatus = ITS4YouImport_Module_Model::$FIELD_IMPORT_STATUS_HALTED;
                break;

            case self::$IMPORT_STATUS_NONE:
                $importEntityStatus = ITS4YouImport_Module_Model::$FIELD_IMPORT_STATUS_NONE;
                break;

            case self::$IMPORT_STATUS_SCHEDULED:
                $importEntityStatus = ITS4YouImport_Module_Model::$FIELD_IMPORT_STATUS_SCHEDULED;
                break;

            default:
                $importEntityStatus = "";
                break;
        }

        if ($importEntityStatus != "") {
            self::updateImportEntityStatus($importId, $importEntityStatus);
        }
    }

    /**
     * Updates counters of Import instance (ITS4YouImport_Record_Model)
     * It is public, because $importStatusCount must be provided by the caller
     * @param int $importId Import queue ID
     * @param array $importStatusCount Counters information
     */
    public static function updateImportEntityCounters($importId, $importStatusCount)
    {
        $importEntityId = self::getImportEntityId($importId);
        if ($importEntityId) {
            ITS4YouImport_Record_Model::updateCounters($importEntityId, $importStatusCount);
        }
    }

}
