<?php
/* * *******************************************************************************
 * The content of this file is subject to the ITS4YouImport license.
 * ("License"); You may not use this file except in compliance with the License
 * The Initial Developer of the Original Code is IT-Solutions4You s.r.o.
 * Portions created by IT-Solutions4You s.r.o. are Copyright(C) IT-Solutions4You s.r.o.
 * All Rights Reserved.
 * ****************************************************************************** */

require_once 'include/QueryGenerator/QueryGenerator.php';
require_once 'modules/ITS4YouImport/models/LogEntryBase.php';

class ITS4YouImport_Data_Action extends Import_Data_Action
{

    protected $trigger_workflow;
    protected $create_picklist_values;

    /** @var ITS4YouImport_Logger_Model Logger instance */
    protected $logger;

    /** @var ITS4YouImport_DataLogEntry_Model Log entry of currently processed
     * importing entry */
    protected $currentItemLogEntry;

    /** @var array Import info */
    protected $importInfo;

    /** @var int ITS4YouImport_Record_Model instance ID */
    protected $importEntityId;

    public function __construct($importInfo, $user)
    {
        parent::__construct($importInfo, $user);

        $this->importInfo = $importInfo;
        $this->trigger_workflow = $importInfo['trigger_workflow'];
        $this->create_picklist_values = $importInfo['create_picklist_values'];
        $this->importEntityId = $importInfo['import_entity_id'];
        $this->logger = null;
    }

    /**
     * Overriden method for background import purposes. Its body is the same as the parent's method body,
     * but explicit implementation is needed because of "Late Static Binding" in php.
     */
    public static function runScheduledImport()
    {
        require_once 'modules/Emails/class.phpmailer.php';
        require_once 'vtlib/Vtiger/Mailer.php';

        global $current_user;
        $scheduledImports = self::getScheduledImport();
        $vtigerMailer = new Vtiger_Mailer();
        $vtigerMailer->IsHTML(true);
        foreach ($scheduledImports as $scheduledId => $importDataController) {
            $current_user = $importDataController->user;
            $importDataController->batchImport = false;

            if (!$importDataController->initializeImport()) {
                continue;
            }
            $importDataController->importData();

            $importStatusCount = $importDataController->getImportStatusCount();

//            $emailSubject = 'vtiger CRM - Scheduled Import Report for ' . $importDataController->module;
            $emailSubject = 'vtiger CRM - ITS4YouImport - Scheduled XLS Import Report for ' . $importDataController->module;
            $viewer = new Vtiger_Viewer();
            $viewer->assign('FOR_MODULE', $importDataController->module);
            $viewer->assign('MODULE', 'ITS4YouImport');
            $viewer->assign('INVENTORY_MODULES', getInventoryModules());
            $viewer->assign('IMPORT_RESULT', $importStatusCount);
            $importResult = $viewer->view('Import_Result_Details.tpl', 'Import', true);
            $importResult = str_replace('align="center"', '', $importResult);
            $emailData = 'vtiger CRM has just completed your import process. <br/><br/>' .
                $importResult . '<br/><br/>' .
                'We recommend you to login to the CRM and check few records to confirm that the import has been successful.';

            $userName = getFullNameFromArray('Users', $importDataController->user->column_fields);
            $userEmail = $importDataController->user->email1;
            $vtigerMailer->ConfigSenderInfo($userEmail, $userName, $userEmail);
            $vtigerMailer->AddAddress($userEmail, $userName);

            $vtigerMailer->Subject = $emailSubject;
            $vtigerMailer->Body = $emailData;
            $vtigerMailer->Signature = '';
            $vtigerMailer->AltBody = strip_tags($emailData);
            $vtigerMailer->Send(true);

            $importDataController->finishImport();
        }
        Vtiger_Mailer::dispatchQueue(null);
    }

    /**
     * Overriden method which works with ITS4YouImport_Queue_Action instead of Import_Queue_Action
     * @return Import_Data_Action
     */
    public static function getScheduledImport()
    {
        $scheduledImports = array();
        $importQueue = ITS4YouImport_Queue_Action::getAll(ITS4YouImport_Queue_Action::$IMPORT_STATUS_SCHEDULED);
        foreach ($importQueue as $importId => $importInfo) {
            $userId = $importInfo['user_id'];
            $user = new Users();
            $user->id = $userId;
            $user->retrieve_entity_info($userId, 'Users');

            $scheduledImports[$importId] = new ITS4YouImport_Data_Action($importInfo, $user);
        }
        return $scheduledImports;
    }

    /**
     * Method to get Record details of import
     * @param int $importEntityId Import ID
     * @param int $userId User ID
     */
    public static function getImportDetailsEnhanced($importEntityId, $userId, $targetModule)
    {

        $adb = PearDatabase::getInstance();

        //update for calendar 600.700.1.1
        if ($targetModule === 'Events') {
            $targetModule = 'Calendar';
        }

        $importLogger = ITS4YouImport_Logger_Model::factory($importEntityId, $userId);
        /* @var $importLogger ITS4YouImport_Logger_Model */

        $user = Users_Record_Model::getInstanceById($userId, "Users");
        $tableName = ITS4YouImport_Utils_Helper::getDbTableName($user);
        $result = $adb->pquery("SELECT * FROM $tableName where status IN (?,?)", array(self::$IMPORT_RECORD_SKIPPED, self::$IMPORT_RECORD_FAILED));
        $importRecords = array();
        if ($result) {
            $headers = $adb->getColumnNames($tableName);
            $numOfHeaders = count($headers);
            $importRecords['headers'][] = "id"; // line id
            $importRecords['headers'][] = "log_message"; // log message
            for ($i = 3; $i < $numOfHeaders; $i++) {
                $importRecords['headers'][] = $headers[$i];
            }
            // handle translation of column header
            $transSql = "SELECT DISTINCT fieldname, fieldlabel FROM vtiger_field WHERE tabid = ? AND fieldname IN (" . generateQuestionMarks($headers) . ")";
            $transResult = $adb->pquery($transSql, array_merge(array(getTabid($targetModule)), $headers));
            while ($transRow = $adb->fetchByAssoc($transResult)) {
                $transHeaders[$transRow["fieldname"]] = vtranslate($transRow["fieldlabel"], $targetModule);
            }
            //sort translated headers by original headers
            $sortedTransHeaders = array();
            $sortedTransHeaders['id'] = vtranslate("LBL_LINE_ID", "ITS4YouImport");
            $sortedTransHeaders['log_message'] = vtranslate("LBL_LOG_MESSAGE", "ITS4YouImport");
            for ($i = 3; $i < $numOfHeaders; $i++) {
                if (isset($transHeaders[$headers[$i]])) {
                    $sortedTransHeaders[$headers[$i]] = $transHeaders[$headers[$i]];
                } else {
                    $sortedTransHeaders[$headers[$i]] = $headers[$i];
                }
            }
            $importRecords['transHeaders'] = $sortedTransHeaders;

            $noOfRows = $adb->num_rows($result);
            for ($i = 0; $i < $noOfRows; ++$i) {
                $row = $adb->fetchByAssoc($result, $i);
                $record = new Vtiger_Base_Model();
                foreach ($importRecords['headers'] as $header) {
                    $record->set($header, $row[$header]);
                }

                //enhanced information from logger
                $logEntry = $importLogger->getLogEntry($row["id"]);
                /* @var $logEntry ITS4YouImport_DataLogEntry_Model */
                if ($logEntry !== null) {
                    $logMessage = $logEntry->getLogMessage(ITS4YouImport_LogMessage_Model::$LOG_MSG_TYPE_ERROR);
                    /* @var $logMessage ITS4YouImport_LogMessage_Model */
                    if (empty($logMessage)) {
                        $logMessage = $logEntry->getLogMessage(ITS4YouImport_LogMessage_Model::$LOG_MSG_TYPE_WARNING);
                    }
                    if (!empty($logMessage)) {
                        $record->set("log_message", $logMessage->getMessageText());
                        $record->set("log_message_type", $logMessage->getType());
                    } else {
                        $record->set("log_message", "");
                    }
                }

                if ($row['status'] == self::$IMPORT_RECORD_SKIPPED) {
                    $importRecords['skipped'][] = $record;
                } else {
                    $importRecords['failed'][] = $record;
                }
            }

            return $importRecords;
        }
    }

    /**
     * Returns current instance of Logger
     * @return ITS4YouImport_Logger_Model
     */
    public function getLogger()
    {
        if ($this->logger == null) {
            $this->logger = ITS4YouImport_Logger_Model::factory($this->importEntityId, $this->user->id);
        }
        return $this->logger;
    }

    /**
     * Returns log entry of currently processed importing entry (i.e. it might
     * be used in method importRecord of specific CRMEntity module object)
     * @return ITS4YouImport_DataLogEntry_Model
     */
    public function getCurrentItemLogEntry()
    {
        return $this->currentItemLogEntry;
    }

    /**
     * Initialize import in terms of locking and checking if Import
     * functionality is already locked for selected module
     * @return bool true if import was initialized successfully,
     * else otherwise (it means that other user already initiated import
     * for selected module)
     */
    public function initializeImport()
    {
        $lockInfo = ITS4YouImport_Lock_Action::isLockedForModule($this->module);
        if ($lockInfo != null) {
            if ($lockInfo['userid'] != $this->user->id) {
                ITS4YouImport_Utils_Helper::showImportLockedError($lockInfo);
                return false;
            } else {
                return true;
            }
        } else {
            ITS4YouImport_Lock_Action::lock($this->id, $this->module, $this->user);
            return true;
        }
    }

    public function finishImport()
    {
        //TODO it is not handled when importStatusCount is provided externally from $focus instance
        //update 600.700.1.1 remove all queue params and then unlock user
        ITS4YouImport_Queue_Action::updateImportEntityCounters($this->id, $this->getImportStatusCount());
        ITS4YouImport_Queue_Action::remove($this->id);
        ITS4YouImport_Lock_Action::unLock($this->user, $this->module);
    }

    public function getImportStatusCount()
    {
        $adb = PearDatabase::getInstance();
        $tableName = Import_Utils_Helper::getDbTableName($this->user);
        $moduleModel = Vtiger_Module_Model::getInstance('ITS4YouImport');
        $priceBook = $moduleModel->isPriceBookImport($this->importInfo);
        $focus = CRMEntity::getInstance($this->module);
        if ($focus && method_exists($focus, 'getGroupQuery') && $priceBook === false) {
            $query = $focus->getGroupQuery($tableName);
        } else {
            $query = 'SELECT status FROM ' . $tableName;
        }
        $result = $adb->query($query, array());

        $statusCount = array('TOTAL' => 0, 'IMPORTED' => 0, 'FAILED' => 0, 'PENDING' => 0, 'CREATED' => 0, 'SKIPPED' => 0, 'UPDATED' => 0, 'MERGED' => 0);

        if ($result) {
            $noOfRows = $adb->num_rows($result);
            $statusCount['TOTAL'] = $noOfRows;
            for ($i = 0; $i < $noOfRows; ++$i) {
                $status = $adb->query_result($result, $i, 'status');
                if (self::$IMPORT_RECORD_NONE == $status) {
                    $statusCount['PENDING']++;
                } elseif (self::$IMPORT_RECORD_FAILED == $status) {
                    $statusCount['FAILED']++;
                } else {
                    $statusCount['IMPORTED']++;
                    switch ($status) {
                        case self::$IMPORT_RECORD_CREATED    :
                            $statusCount['CREATED']++;
                            break;
                        case self::$IMPORT_RECORD_SKIPPED    :
                            $statusCount['SKIPPED']++;
                            break;
                        case self::$IMPORT_RECORD_UPDATED    :
                            $statusCount['UPDATED']++;
                            break;
                        case self::$IMPORT_RECORD_MERGED    :
                            $statusCount['MERGED']++;
                            break;
                    }
                }
            }
        }
        return $statusCount;
    }

    /**
     * Importing data initiation
     */
    public function importData()
    {
        global $VTIGER_BULK_SAVE_MODE;

        $focus = CRMEntity::getInstance($this->module);
        $moduleModel = Vtiger_Module_Model::getInstance($this->module);
        // pre fetch the fields and premmisions of module
        Vtiger_Field_Model::getAllForModule($moduleModel);
        if ($this->user->is_admin == 'off') {
            Vtiger_Field_Model::preFetchModuleFieldPermission($moduleModel->getId());
        }

        //handle trigger_workflow flag
        $previous_VTIGER_BULK_SAVE_MODE = $VTIGER_BULK_SAVE_MODE;
        if ($this->trigger_workflow) {
            $VTIGER_BULK_SAVE_MODE = false;
        } else {
            $VTIGER_BULK_SAVE_MODE = true;
        }

        if ($this->module === 'PriceBooks' && isRecordExists($this->importInfo['source_record'])) {
            $this->logger = ITS4YouImport_Logger_Model::factory($this->importEntityId, $this->user->id);
            $this->createPriceBookProducts();

        } elseif (!in_array($this->module, ITS4YouImport_Utils_Helper::getInventoryModules())) {
            $this->logger = ITS4YouImport_Logger_Model::factory($this->importEntityId, $this->user->id);
            if (method_exists($focus, 'createRecords')) {
                $focus->createRecords($this);
            } else {
                $this->createRecords();
            }
        } else {
            $this->logger = ITS4YouImport_Logger_Model::factory($this->importEntityId, $this->user->id, true);
            $this->createInventoryRecords();
            $this->logger->purifyRepository();
        }
        //save current state of logger instance into db in its serialized form
        $this->logger->store();
        $this->updateMissingSeqNumber($this->module);

        $VTIGER_BULK_SAVE_MODE = $previous_VTIGER_BULK_SAVE_MODE;
    }

    /**
     * @param string $moduleName
     * @throws Exception
     */
    public function updateMissingSeqNumber($moduleName)
    {
        $focus = CRMEntity::getInstance($moduleName);

        if (vtlib_isModuleActive('ITS4YouMultiCompany')) {
            $adb = PearDatabase::getInstance();
            $tabId = getTabid($moduleName);
            $fieldResult = $adb->pquery('SELECT tablename, columnname FROM vtiger_field WHERE tabid = ? AND uitype = ?', array($tabId, 4));

            if ($adb->num_rows($fieldResult)) {
                $fieldInfo = $adb->query_result_rowdata($fieldResult);
                $fieldColumn = $fieldInfo['columnname'];

                if ($focus->table_name === $fieldInfo['tablename']) {
                    $recordsResult = $adb->query(sprintf('SELECT %s AS recordid FROM %s WHERE %s="" OR %s is NULL', $focus->table_index, $focus->table_name, $fieldColumn, $fieldColumn));

                    while ($row = $adb->fetchByAssoc($recordsResult)) {
                        if (!isRecordExists($row['recordid'])) {
                            continue;
                        }

                        $recordModel = Vtiger_Record_Model::getInstanceById($row['recordid']);
                        $companyId = ITS4YouMultiCompany_CustomRecordNumbering_Model::getCompanyForUser($recordModel->get('assigned_user_id'));

                        if ($companyId) {
                            $numbering = ITS4YouMultiCompany_CustomRecordNumbering_Model::getInstance($moduleName, $tabId);
                            $next = $numbering->setModuleSeqNumber('increment', $moduleName, '', '', $companyId);

                            if ($next) {
                                $adb->pquery(sprintf('UPDATE %s SET %s=? WHERE %s=?', $focus->table_name, $fieldColumn, $focus->table_index), [$next, $recordModel->getId()]);
                                $numbering->decrementStandardNumbering($moduleName);
                            }
                            if (columnExists('its4you_company', $focus->table_name)) {
                                $adb->pquery(sprintf('UPDATE %s SET its4you_company=? WHERE %s=?', $focus->table_name, $focus->table_index), [$companyId, $recordModel->getId()]);
                            }
                        }
                    }
                }
            }
        }

        $focus->updateMissingSeqNumber($moduleName);
    }

    private function createPriceBookProducts()
    {
        $adb = PearDatabase::getInstance();
        $tableName = ITS4YouImport_Utils_Helper::getDbTableName($this->user);
        $sql = 'SELECT * FROM ' . $tableName . ' WHERE status = ' . ITS4YouImport_Data_Action::$IMPORT_RECORD_NONE;

        if ($this->batchImport) {
            $configReader = new ITS4YouImport_Config_Model();
            $importBatchLimit = $configReader->get('importBatchLimit');
            $sql .= ' LIMIT ' . $importBatchLimit;
        }

        $result = $adb->query($sql);
        $numberOfRecords = $adb->num_rows($result);

        if ($numberOfRecords <= 0) {
            return;
        }

        $importInfo = $this->importInfo;
        $source_record = $importInfo['source_record'];
        $fieldMapping = $importInfo['field_mapping'];
        $productField = $fieldMapping['products'];
        $serviceField = $fieldMapping['services'];
        $priceBooksRecord = Vtiger_Record_Model::getInstanceById($source_record, 'PriceBooks');

        if ('1' === $importInfo['clear_pricebooks']) {
            $adb->pquery('DELETE FROM vtiger_pricebookproductrel WHERE pricebookid = ?', array($source_record));
        }

        for ($i = 0; $i < $numberOfRecords; $i++) {
            $row = $adb->raw_query_result_rowdata($result, $i);
            $rowId = $row['id'];
            $price = CurrencyField::convertToDBFormat($row['price'], $this->user, false);
            $identifier = $row['identifier'];
            $entityInfo = array('status' => self::$IMPORT_RECORD_FAILED);
            $fieldData = array(
                'identifier' => $identifier,
                'price' => $price,
            );

            $this->currentItemLogEntry = new ITS4YouImport_DataLogEntry_Model($rowId, $fieldData);
            $this->logger->addLogEntry($this->currentItemLogEntry);
            $currentModule = 'ITS4YouImport';
            if (!is_numeric($price)) {
                $this->addLogError("LBL_MANDATORY_FIELD_ERROR", array(vtranslate('Price', $currentModule), 'price'));
            } elseif (empty($identifier)) {
                $this->addLogError("LBL_MANDATORY_FIELD_ERROR", array(vtranslate('Identifier', $currentModule), 'identifier'));
            } else {
                preg_match('/(Products|Services):+(.+)/', $identifier, $matches);

                if (!empty($matches)) {
                    $field = $productField;
                    if ($matches[1] === 'Services') {
                        $field = $serviceField;
                    }
                    $product = $this->getPriceBooksItem($matches[1], $field, $matches[2]);
                } else {
                    $product = $this->getPriceBooksItem('Products', $productField, $identifier);

                    if (empty($product)) {
                        $product = $this->getPriceBooksItem('Services', $serviceField, $identifier);
                    }
                }

                if (empty($product)) {
                    $this->addLogError(vtranslate('LBL_PRODUCTS_NOT_EXIST', 'ITS4YouImport'));
                } else {
                    $productId = $product['crmid'];
                    $priceBooksRecord->updateListPrice($productId, $price);
                    $entityInfo['status'] = self::$IMPORT_RECORD_CREATED;
                }
            }


            $this->importedRecordInfo[$rowId] = $entityInfo;
            $this->updateImportStatus($rowId, $entityInfo);
            $this->currentItemLogEntry->addEntityInfo($entityInfo);
        }
    }

    public function addLogError($message, $msgParams = array())
    {
        $this->addLogMessage2LogEntry(new ITS4YouImport_LogMessage_Model(
            ITS4YouImport_LogMessage_Model::$LOG_MSG_TYPE_ERROR, $message, $msgParams
        ));
    }

    /**
     * Check if currentItemLogEntry exists and inserts a logMessage into it.
     * It must be used in all public methods of this class which are working with
     * $this->currentItemLogEntry, because the existence of that instance is not
     * guaranteed. The instance of currentItemLogEntry is created in $this->createRecords.
     * @param ITS4YouImport_LogMessage_Model $logMessageModel
     */
    private function addLogMessage2LogEntry(ITS4YouImport_LogMessage_Model $logMessageModel)
    {
        if ($this->currentItemLogEntry) {
            $this->currentItemLogEntry->addLogMessageModel($logMessageModel);
        }
    }

    public function getPriceBooksItem($module, $field, $value)
    {
        if ('crm_id' === $field) {
            return [
                'crmid' => $value,
            ];
        }

        $adb = PearDatabase::getInstance();
        $productSql = new QueryGenerator($module, $this->user);
        $productSql->addCondition($field, $value, 'c');

        $query = "SELECT crmid";
        $query .= $productSql->getFromClause();
        $query .= $productSql->getWhereClause();

        $result = $adb->pquery($query);
        $row = $adb->raw_query_result_rowdata($result, 0);

        return $row;
    }

    public function createRecords()
    {
        $adb = PearDatabase::getInstance();
        $moduleName = $this->module;

        $focus = CRMEntity::getInstance($moduleName);
        $moduleHandler = vtws_getModuleHandlerFromName($moduleName, $this->user);
        $moduleMeta = $moduleHandler->getMeta();
        $moduleObjectId = $moduleMeta->getEntityId();

        $moduleFields = $moduleMeta->getModuleFields();
        $entityData = array();
        $tableName = ITS4YouImport_Utils_Helper::getDbTableName($this->user);
        $sql = 'SELECT * FROM ' . $tableName . ' WHERE status = ' . ITS4YouImport_Data_Action::$IMPORT_RECORD_NONE;

        if ($this->batchImport) {
            $configReader = new ITS4YouImport_Config_Model();
            $importBatchLimit = $configReader->get('importBatchLimit');
            $sql .= ' LIMIT ' . $importBatchLimit;
        }
        $result = $adb->query($sql);
        $numberOfRecords = $adb->num_rows($result);

        if ($numberOfRecords <= 0) {
            return;
        }

        $fieldMapping = $this->fieldMapping;
        $fieldColumnMapping = $moduleMeta->getFieldColumnMapping();

        for ($i = 0; $i < $numberOfRecords; ++$i) {
            $row = $adb->raw_query_result_rowdata($result, $i);

            $rowId = $row['id'];
            $entityInfo = null;
            $fieldData = array();

            foreach ($fieldMapping as $fieldName => $index) {
                if(ITS4YouImport_Utils_Helper::isReferenceField($fieldName)) {
                    $searchFieldValue = !empty($row[$fieldName]) ? $row[$fieldName] : $row[strtolower($fieldName)];
                    $this->setSearchFields($fieldName, $searchFieldValue);
                } else {
                    $fieldData[$fieldName] = $row[$fieldName];
                }
            }

            $this->currentItemLogEntry = new ITS4YouImport_DataLogEntry_Model($rowId, $fieldData);
            $this->logger->addLogEntry($this->currentItemLogEntry);

            $mergeType = $this->mergeType;
            $createRecord = false;
            if (method_exists($focus, 'importRecord')) {
                $entityInfo = $focus->importRecord($this, $fieldData);
            } else {
                if (!empty($mergeType) && $mergeType != ITS4YouImport_Utils_Helper::$AUTO_MERGE_NONE) {

                    $queryGenerator = new QueryGenerator($moduleName, $this->user);
                    $customView = new CustomView($moduleName);
                    $viewId = $customView->getViewIdByName('All', $moduleName);
                    if (!empty($viewId)) {
                        $queryGenerator->initForCustomViewById($viewId);
                    } else {
                        $queryGenerator->initForDefaultCustomView();
                    }

                    $fieldsList = array('id');
                    $queryGenerator->setFields($fieldsList);

                    $mergeFields = $this->mergeFields;
                    if ($queryGenerator->getWhereFields() && $mergeFields) {
                        $queryGenerator->addConditionGlue(QueryGenerator::$AND);
                    }
                    foreach ($mergeFields as $index => $mergeField) {
                        if ($index != 0) {
                            $queryGenerator->addConditionGlue(QueryGenerator::$AND);
                        }
                        $comparisonValue = $fieldData[$mergeField];
                        $fieldInstance = $moduleFields[$mergeField];
                        $fieldDataType = $fieldInstance->getFieldDataType();
                        switch ($fieldDataType) {
                            case 'owner' :
                                $userId = getUserId_Ol($comparisonValue);
                                $comparisonValue = getUserFullName($userId);
                                break;
                            case 'reference' :
                                if (strpos($comparisonValue, '::::') > 0) {
                                    $referenceFileValueComponents = explode('::::', $comparisonValue);
                                } else {
                                    $referenceFileValueComponents = explode(':::', $comparisonValue);
                                }
                                if (count($referenceFileValueComponents) > 1) {
                                    $comparisonValue = trim($referenceFileValueComponents[1]);
                                }
                                break;
                            case 'currency'    :
                                if (!empty($comparisonValue)) {
                                    $comparisonValue = CurrencyField::convertToUserFormat($comparisonValue, $this->user, true, false);
                                }
                                break;
                        }
                        $queryGenerator->addCondition($mergeField, $comparisonValue, 'e', '', '', '', true);
                    }
                    $query = $queryGenerator->getQuery();
                    $duplicatesResult = $adb->query($query);
                    $noOfDuplicates = $adb->num_rows($duplicatesResult);

                    if ($noOfDuplicates > 0) {
                        if ($mergeType == ITS4YouImport_Utils_Helper::$AUTO_MERGE_IGNORE) {
                            $entityInfo['status'] = self::$IMPORT_RECORD_SKIPPED;
                            $this->currentItemLogEntry->addLogMessage(ITS4YouImport_LogMessage_Model::$LOG_MSG_TYPE_WARNING, "LBL_RECORD_SKIPPED");
                        } elseif ($mergeType == ITS4YouImport_Utils_Helper::$AUTO_MERGE_OVERWRITE || $mergeType == ITS4YouImport_Utils_Helper::$AUTO_MERGE_MERGEFIELDS) {

                            for ($index = 0; $index < $noOfDuplicates - 1; ++$index) {
                                $duplicateRecordId = $adb->query_result($duplicatesResult, $index, $fieldColumnMapping['id']);
                                $entityId = vtws_getId($moduleObjectId, $duplicateRecordId);
                                vtws_delete($entityId, $this->user);
                            }
                            $baseRecordId = $adb->query_result($duplicatesResult, $noOfDuplicates - 1, $fieldColumnMapping['id']);
                            $baseEntityId = vtws_getId($moduleObjectId, $baseRecordId);

                            if ($mergeType == ITS4YouImport_Utils_Helper::$AUTO_MERGE_OVERWRITE) {
                                $fieldData = $this->transformForImport($fieldData, $moduleMeta);

                                if(null !== $fieldData) {
                                    $fieldData['id'] = $baseEntityId;
                                    $entityInfo = vtws_update($fieldData, $this->user);
                                    $entityInfo['status'] = self::$IMPORT_RECORD_UPDATED;

                                    $this->currentItemLogEntry->addLogMessage(ITS4YouImport_LogMessage_Model::$LOG_MSG_TYPE_WARNING, "LBL_RECORD_UPDATED");
                                }
                            }

                            if ($mergeType == ITS4YouImport_Utils_Helper::$AUTO_MERGE_MERGEFIELDS) {
                                $filteredFieldData = array();
                                foreach ($fieldData as $fieldName => $fieldValue) {
                                    // empty will give false for value = 0
                                    if (!empty($fieldValue) || $fieldValue != "") {
                                        $filteredFieldData[$fieldName] = $fieldValue;
                                    }
                                }

                                // Custom handling for default values & mandatory fields
                                // need to be taken care than normal import as we merge
                                // existing record values with newer values.
                                $fillDefault = false;
                                $mandatoryValueChecks = false;

                                $existingFieldValues = vtws_retrieve($baseEntityId, $this->user);
                                $defaultFieldValues = $this->getDefaultFieldValues($moduleMeta);

                                foreach ($existingFieldValues as $fieldName => $fieldValue) {
                                    if (empty($fieldValue)
                                        && empty($filteredFieldData[$fieldName])
                                        && !empty(trim($defaultFieldValues[$fieldName]))
                                        && '????' != $defaultFieldValues[$fieldName]) {
                                        $filteredFieldData[$fieldName] = $defaultFieldValues[$fieldName];
                                    }
                                }

                                $filteredFieldData = $this->transformForImport($filteredFieldData, $moduleMeta, $fillDefault, $mandatoryValueChecks);

                                if (null !== $filteredFieldData) {
                                    $filteredFieldData['id'] = $baseEntityId;
                                    $entityInfo = vtws_revise($filteredFieldData, $this->user);
                                    $entityInfo['status'] = self::$IMPORT_RECORD_MERGED;
                                    $fieldData = $filteredFieldData;

                                    $this->currentItemLogEntry->addLogMessage(ITS4YouImport_LogMessage_Model::$LOG_MSG_TYPE_WARNING, "LBL_RECORD_MERGED");
                                }
                            }
                        } else {
                            $createRecord = true;
                        }
                    } else {
                        $createRecord = true;
                    }
                } else {
                    $createRecord = true;
                }
                if ($createRecord) {
                    $fieldData = $this->transformForImport($fieldData, $moduleMeta);
                    if ($fieldData == null) {
                        $entityInfo = null;
                    } else {
                        try {
                            $entityInfo = vtws_create($moduleName, $fieldData, $this->user);
                        } catch (Exception $e) {
                            $this->currentItemLogEntry->addLogMessage(ITS4YouImport_LogMessage_Model::$LOG_MSG_TYPE_ERROR, $e->getMessage());
                        }
                    }
                }
            }
            if ($entityInfo == null) {
                $entityInfo = array('id' => null, 'status' => self::$IMPORT_RECORD_FAILED);
            } else {
                if ($createRecord) {
                    $entityInfo['status'] = self::$IMPORT_RECORD_CREATED;
                }
            }
            if ($createRecord || $mergeType == ITS4YouImport_Utils_Helper::$AUTO_MERGE_MERGEFIELDS || $mergeType == ITS4YouImport_Utils_Helper::$AUTO_MERGE_OVERWRITE) {
                $recordId = explode('x', $entityInfo['id'])[1];
                $entityfields = getEntityFieldNames($this->module);
                switch ($this->module) {
                    case 'HelpDesk':
                        $entityfields['fieldname'] = array('ticket_title');
                        break;
                    case 'Documents':
                        $entityfields['fieldname'] = array('notes_title');
                        break;
                }
                $label = '';
                if (is_array($entityfields['fieldname'])) {
                    foreach ($entityfields['fieldname'] as $field) {
                        $label .= $fieldData[$field] . " ";
                    }
                } else {
                    $label = $fieldData[$entityfields['fieldname']];
                }

                $label = trim($label);
                $adb->pquery('UPDATE vtiger_crmentity SET label=? WHERE crmid=?', array($label, $recordId));
            }

            $this->importedRecordInfo[$rowId] = $entityInfo;
            $this->updateImportStatus($rowId, $entityInfo);
            $this->currentItemLogEntry->addEntityInfo($entityInfo);
        }

        if ($this->entityData) {
            $entity = new VTEventsManager($adb);
            $entity->triggerEvent('vtiger.batchevent.save', $this->entityData);
        }
        $this->entityData = null;
        $result = null;

        return true;
    }

    /**
     * @param string $moduleName
     * @param string $referenceModule
     * @param string $entityLabel
     * @return int
     */
    public function getReferenceEntityId($moduleName, $referenceModule, $entityLabel)
    {
        $helper = sprintf('%s_Import_Helper', $referenceModule);

        if ('Users' === $referenceModule) {
            $referenceEntityId = getUserId_Ol($entityLabel);

            if (empty($referenceEntityId) || !Import_Utils_Helper::hasAssignPrivilege($moduleName,
                    $referenceEntityId)) {
                $referenceEntityId = $this->user->id;
            }
        } elseif ('Currency' === $referenceModule) {
            $referenceEntityId = getCurrencyId($entityLabel);
        } elseif(class_exists($helper) && method_exists($helper, 'getEntityId')) {
            $referenceEntityId = $helper::getEntityId($referenceModule, $entityLabel);
        } elseif($this->hasSearchFields($referenceModule)) {
            $referenceEntityId = $this->getEntityIdBySearchFields($referenceModule, $entityLabel);
        } else {
            $referenceEntityId = getEntityId($referenceModule, $entityLabel);
        }

        return intval($referenceEntityId);
    }

    /**
     * @throws Exception
     */
    public function getEntityIdBySearchFields($referenceModule, $entityLabel)
    {
        $searchFields = $this->getSearchFields($referenceModule);
        $moduleModel = Vtiger_Module_Model::getInstance($referenceModule);
        $moduleFocus = CRMEntity::getInstance($referenceModule);
        $tableIds = $moduleFocus->tab_name_index;

        $tables = [];

        $where = 'WHERE vtiger_crmentity.deleted=0 ';
        $params = [];

        foreach ($searchFields as $fieldName => $fieldValue) {
            $fieldModel = Vtiger_Field_Model::getInstance($fieldName, $moduleModel);
            $tableId = $tableIds[$fieldModel->table];
            $table = $fieldModel->table;
            $tables[$table] = $tableId;

            $where .= sprintf('AND %s.%s=? ', $fieldModel->table, $fieldModel->column);
            $params[] = $fieldValue;
        }

        $sql = 'SELECT crmid FROM vtiger_crmentity ';

        foreach ($tables as $table => $tableId) {
            $sql .= sprintf('INNER JOIN %s ON %s.%s=vtiger_crmentity.crmid ', $table, $table, $tableId);
        }

        $sql .= $where;
        $adb = PearDatabase::getInstance();
        $result = $adb->pquery($sql, $params);
        $entityId = (int)$adb->query_result($result, 0, 'crmid');

        return !empty($entityId) ? $entityId : getEntityId($referenceModule, $entityLabel);
    }

    /**
     * @var array
     */
    public $searchFields = [];

    /**
     * @param $field
     * @param $value
     * @return void
     */
    public function setSearchFields(&$field, $value)
    {
        list($module, $field) = explode(ITS4YouImport_Utils_Helper::$referenceSeparator, $field);

        $this->searchFields[$module][$field] = $value;
    }

    public function hasSearchFields($module)
    {
        return !empty($this->searchFields[$module]);
    }

    public function getSearchFields($module)
    {
        return $this->searchFields[$module];
    }

    /**
     * Overidden method for enabling/disabling feature of "Create picklist values"
     * @param array $fieldData
     * @param VtigerCRMObjectMeta $moduleMeta
     * @param bool $fillDefault
     * @param bool $checkMandatoryFieldValues
     * @return type
     * @throws Exception
     * @global string $default_charset
     */
    public function transformForImport($fieldData, $moduleMeta, $fillDefault = true, $checkMandatoryFieldValues = true)
    {
        $current_user = vglobal('current_user');
        $moduleFields = $moduleMeta->getModuleFields();
        $defaultFieldValues = $this->getDefaultFieldValues($moduleMeta);
        $moduleName = $moduleMeta->getEntityName();

        foreach ($fieldData as $fieldName => $fieldValue) {
            $fieldInstance = $moduleFields[$fieldName];
            $fieldDataType = $fieldInstance->getFieldDataType();

            if ('Calendar' === $moduleName || 'Events' === $moduleName) {
                if ('date_start' === $fieldName || 'due_date' === $fieldName) {
                    $fieldDataType = 'date';
                }
            }

            /* @var $fieldInstance WebserviceField */
            switch ($fieldDataType) {
                case 'owner':
                    $ownerId = getUserId_Ol(trim($fieldValue));
                    if (empty($ownerId)) {
                        $ownerId = getGrpId($fieldValue);
                    }
                    if (empty($ownerId) && isset($defaultFieldValues[$fieldName])) {
                        $ownerId = $defaultFieldValues[$fieldName];
                    }
                    if (empty($ownerId) ||
                        !Import_Utils_Helper::hasAssignPrivilege($moduleName, $ownerId)) {
                        $ownerId = $this->user->id;
                    }
                    $fieldData[$fieldName] = $ownerId;
                    break;

                case 'multipicklist':
                    $trimmedValue = trim($fieldValue);

                    if (!$trimmedValue && isset($defaultFieldValues[$fieldName])) {
                        $explodedValue = explode(',', $defaultFieldValues[$fieldName]);
                    } else {
                        $explodedValue = explode(' |##| ', $trimmedValue);
                    }

                    foreach ($explodedValue as $key => $value) {
                        $explodedValue[$key] = trim($value);
                    }

                    $implodeValue = implode(' |##| ', $explodedValue);
                    $fieldData[$fieldName] = $implodeValue;
                    break;

                case 'reference':
                    $entityId = false;

                    if (!empty($fieldValue)) {
                        if (strpos($fieldValue, '::::') > 0) {
                            $fieldValueDetails = explode('::::', $fieldValue);
                        } elseif (strpos($fieldValue, ':::') > 0) {
                            $fieldValueDetails = explode(':::', $fieldValue);
                        } else {
                            $fieldValueDetails = $fieldValue;
                        }
                        
                        if (count((array)$fieldValueDetails) > 1) {
                            $referenceModuleName = trim($fieldValueDetails[0]);
                            $entityLabel = trim($fieldValueDetails[1]);
                            $entityId = getEntityId($referenceModuleName, $entityLabel);
                        } else {
                            $referencedModules = $fieldInstance->getReferenceList();
                            $entityLabel = $fieldValue;

                            foreach ($referencedModules as $referenceModule) {
                                $referenceModuleName = $referenceModule;
                                $referenceEntityId = $this->getReferenceEntityId($moduleName, $referenceModuleName, $entityLabel);

                                if (!empty($referenceEntityId)) {
                                    $entityId = $referenceEntityId;
                                    break;
                                }
                            }
                        }
                        if ((empty($entityId) || $entityId == 0) && !empty($referenceModuleName)) {
                            if (isPermitted($referenceModuleName, 'EditView') == 'yes') {
                                try {
                                    $wsEntityIdInfo = $this->createEntityRecord($referenceModuleName, $entityLabel);
                                    $entityId = explode('x', $wsEntityIdInfo['id'])[1];
                                } catch (Exception $e) {
                                    $entityId = false;
                                }
                            }
                        }
                        $fieldData[$fieldName] = $entityId;
                    } else {
                        $referencedModules = $fieldInstance->getReferenceList();
                        if ($referencedModules[0] == 'Users') {
                            if (isset($defaultFieldValues[$fieldName])) {
                                $fieldData[$fieldName] = $defaultFieldValues[$fieldName];
                            }

                            if (empty($fieldData[$fieldName]) || !Import_Utils_Helper::hasAssignPrivilege($moduleName, $fieldData[$fieldName])) {
                                $fieldData[$fieldName] = $this->user->id;
                            }
                        } else {
                            $fieldData[$fieldName] = '';
                        }
                    }

                    break;

                case 'picklist':
                    $fieldValue = trim($fieldValue);
                    global $default_charset;
                    if (empty($fieldValue) && isset($defaultFieldValues[$fieldName])) {
                        $fieldData[$fieldName] = $fieldValue = $defaultFieldValues[$fieldName];
                    }
                    $olderCacheEnable = Vtiger_Cache::$cacheEnable;
                    Vtiger_Cache::$cacheEnable = false;
                    if (!isset($this->allPicklistValues[$fieldName])) {
                        $this->allPicklistValues[$fieldName] = $fieldInstance->getPicklistDetails();
                    }
                    $allPicklistDetails = $this->allPicklistValues[$fieldName];

                    $allPicklistValues = array();
                    $allPicklistLabels = array();
                    foreach ($allPicklistDetails as $picklistDetails) {
                        $allPicklistValues[] = $picklistDetails['value'];
                        $allPicklistLabels[] = $picklistDetails['label'];
                    }

                    $picklistValueInLowerCase = strtolower(decode_html($fieldValue));
                    $allPicklistValuesInLowerCase = array_map('strtolower', $allPicklistValues);
                    $allPicklistLabelsInLowerCase = array_map('strtolower', $allPicklistLabels);
                    $picklistDetails = array_combine($allPicklistValuesInLowerCase, $allPicklistValues);
                    $picklistDetailsByLabel = array_combine($allPicklistLabelsInLowerCase, $allPicklistValues);

                    if (in_array($picklistValueInLowerCase, $allPicklistValuesInLowerCase)) {
                        $fieldData[$fieldName] = $picklistDetails[$picklistValueInLowerCase];
                    } elseif (in_array($picklistValueInLowerCase, $allPicklistLabelsInLowerCase)) {
                        $fieldData[$fieldName] = $picklistDetailsByLabel[$picklistValueInLowerCase];
                    } else {
                        if ($this->create_picklist_values) {
                            $moduleObject = Vtiger_Module::getInstance($moduleName);
                            $fieldObject = Vtiger_Field::getInstance($fieldName, $moduleObject);
                            $fieldObject->setPicklistValues(array(trim($fieldValue)));
                            unset($this->allPicklistValues[$fieldName]);
                        } else {
                            //special case handling for blank value as a picklist value
                            //update for calendar 600.700.1.1
                            if (('Calendar' === $moduleName || 'Events' === $moduleName) && 'activitytype' === $fieldName) {
                                $fieldData[$fieldName] = $fieldValue;
                            } elseif (empty($fieldValue) && !$fieldInstance->isMandatory()) {
                                $fieldData[$fieldName] = $fieldValue;
                            } else {
                                $this->addLogError("LBL_UNKNOWN_PICKLIST_VALUE", array(vtranslate($fieldInstance->getFieldLabelKey(), $moduleName), $fieldName));
                                return null;
                            }
                        }
                    }
                    Vtiger_Cache::$cacheEnable = $olderCacheEnable;
                    break;

                case 'currency':
                    $fieldValue = $this->getImportCurrency($fieldValue);

                    if(empty($fieldValue)) {
                        $fieldValue = $this->getImportCurrency($defaultFieldValues[$fieldName]);
                    }

                    if (!empty($fieldValue) && !is_numeric($fieldValue)) {
                        $this->addLogError("LBL_ERROR_DECIMAL_VALUE", array(vtranslate($fieldInstance->getFieldLabelKey(), $moduleName), $fieldName));
                        return null;
                    }

                    $fieldData[$fieldName] = $fieldValue;
                    break;

                case 'integer':
                    $fieldValue = trim($fieldValue);
                    if (!empty($fieldValue) && !is_int($fieldValue) && !ctype_digit($fieldValue)) {
                        $this->addLogError("LBL_ERROR_INTEGER_VALUE", array(vtranslate($fieldInstance->getFieldLabelKey(), $moduleName), $fieldName));
                        return null;
                    }

                    if(!$this->isInRange($fieldInstance, $fieldValue)) {
                        $this->addLogError('LBL_ERROR_INTEGER_OUT_RANGE', array(vtranslate($fieldInstance->getFieldLabelKey(), $moduleName), $fieldName));
                        return null;
                    }

                    $fieldData[$fieldName] = $fieldValue;
                    break;

                case 'double':
                    $fieldValue = $this->getImportDouble($fieldValue);

                    if (empty($fieldValue)) {
                        $fieldValue = $this->getImportDouble($defaultFieldValues[$fieldName]);
                    }

                    if (!empty($fieldValue) && !is_numeric($fieldValue)) {
                        $this->addLogError("LBL_ERROR_DECIMAL_VALUE", array(vtranslate($fieldInstance->getFieldLabelKey(), $moduleName), $fieldName));
                        return null;
                    }

                    $fieldData[$fieldName] = $fieldValue;
                    break;
                case 'email':
                    $fieldValue = trim($fieldValue);
                    if (!empty($fieldValue) && !filter_var($fieldValue, FILTER_VALIDATE_EMAIL)) {
                        $this->addLogError("LBL_ERROR_EMAIL_VALUE", array(vtranslate($fieldInstance->getFieldLabelKey(), $moduleName), $fieldName));
                        return null;
                    }
                    $fieldData[$fieldName] = $fieldValue;
                    break;

//                case 'password': break; //possible check, but probably not needed

                case 'datetime':
                    if (!empty($fieldValue)) {
                        if ($fieldValue == null || $fieldValue == '0000-00-00 00:00:00' || $fieldValue == '--') {
                            $fieldValue = '';
                        }
                        $valuesList = explode(' ', $fieldValue);
                        if (count($valuesList) == 1) {
                            $fieldValue = '';
                        }
                        $beforeValue = $fieldValue;
                        $fieldValue = getValidDBInsertDateTimeValue($fieldValue);

                        //if function above returns empty string, then value was incorrect
                        if ($beforeValue != '' && $fieldValue == '') {
                            $this->addLogError("LBL_ERROR_DATETIME_VALUE", array(vtranslate($fieldInstance->getFieldLabelKey(), $moduleName), $fieldName));
                            return null;
                        }

                        if (preg_match("/^[0-9]{2,4}[-][0-1]{1,2}?[0-9]{1,2}[-][0-3]{1,2}?[0-9]{1,2} ([0-1][0-9]|[2][0-3])([:][0-5][0-9]){1,2}$/", $fieldValue) == 0) {
                            $fieldValue = '';
                        }
                        $fieldData[$fieldName] = $fieldValue;
                    } elseif (isset($defaultFieldValues[$fieldName])) {
                        $fieldData[$fieldName] = $fieldValue = $defaultFieldValues[$fieldName];
                    }
                    break;

                case 'date':
                    if (!empty($fieldValue)) {
                        if ($fieldValue == null || $fieldValue == '0000-00-00' || $fieldValue == '--') {
                            $fieldValue = '';
                        }
                        $beforeValue = $fieldValue;
                        $fieldValue = getValidDBInsertDateValue($fieldValue);

                        //if function above returns empty string, then value was incorrect
                        if (!empty($beforeValue) && empty($fieldValue)) {
                            $this->addLogError("LBL_ERROR_DATE_VALUE", array(vtranslate($fieldInstance->getFieldLabelKey(), $moduleName), $fieldName));
                            return null;
                        }

                        if (preg_match("/^[0-9]{2,4}[-][0-1]{1,2}?[0-9]{1,2}[-][0-3]{1,2}?[0-9]{1,2}/", $fieldValue) == 0) {
                            $fieldValue = '';
                        }

                        $fieldData[$fieldName] = $fieldValue;
                    } elseif (isset($defaultFieldValues[$fieldName])) {
                        $fieldData[$fieldName] = $fieldValue = DateTimeField::convertToDBFormat($defaultFieldValues[$fieldName]);
                    }
                    break;

                default:
                    if (empty($fieldValue) && isset($defaultFieldValues[$fieldName])) {
                        $fieldData[$fieldName] = $fieldValue = $defaultFieldValues[$fieldName];
                    }
            }
        }

        if ($fillDefault) {
            foreach ($defaultFieldValues as $fieldName => $fieldValue) {
                if (empty($fieldData[$fieldName])) {
                    $fieldData[$fieldName] = $fieldValue;
                }
            }
        }

        // We should sanitizeData before doing final mandatory check below.
        $dataTransform = new DataTransform();
        $fieldData = $dataTransform->sanitizeData($fieldData, $moduleMeta);

        if ($fieldData !== null && $checkMandatoryFieldValues) {
            foreach ($moduleFields as $fieldName => $fieldInstance) {
                $fieldValue = $fieldData[$fieldName];
                if (empty($fieldValue) && $fieldInstance->isMandatory()) { //update 600.700.1.1 (empty($fieldValue) || $fieldValue === '????')
                    $fieldLabel = vtranslate($fieldInstance->getFieldLabelKey(), $moduleName);
                    $this->addLogError("LBL_MANDATORY_FIELD_ERROR", array($fieldLabel, $fieldName));
                    return null;
                }
            }
        }

        return $fieldData;
    }

    public function getImportCurrency($value)
    {
        $value = trim($value);
        //remove space if exists and currency symbol from end of string
        $value = preg_replace('/[^0-9]*$/', '', $value);

        if ($this->stringContain($value, ',') && $this->stringContain($value, '.')) {
            $value = str_replace(',', '', trim($value));
        } else {
            $value = str_replace(',', '.', trim($value));
        }

        return $value;
    }

    public function getImportDouble($value)
    {
        return trim(str_replace(
            array(',', '%'),
            array('.', ''),
            $value
        ));
    }

    public function stringContain($value, $find)
    {
        return false !== strpos($value, $find);
    }

    /**
     * @param object $field
     * @param int $value
     * @return bool
     * @throws Exception
     */
    public function isInRange($field, $value)
    {
        $value = intval($value);
        $columnType = $this->getColumnType($field);
        $columnRange = $this->getColumnRange($columnType);

        return !$columnRange || ($value >= $columnRange['min'] && $value <= $columnRange['max']);
    }

    /**
     * @param object $field
     * @return string
     * @throws Exception
     */
    public function getColumnType($field)
    {
        $adb = PearDatabase::getInstance();
        $result = $adb->pquery('SELECT data_type FROM information_schema.columns WHERE table_schema=? AND table_name=? AND column_name=?', array(vglobal('dbconfig')['db_name'], $field->getTableName(), $field->getColumnName()));

        return $adb->query_result($result, 0, 'data_type');
    }

    /**
     * @return array
     */
    public function getIntRanges()
    {
        return array(
            'bigint' => array(
                'min' => -pow(2, 63),
                'max' => pow(2, 63) - 1,
            ),
            'int' => array(
                'min' => -pow(2, 31),
                'max' => pow(2, 31) - 1,
            ),
            'smallint' => array(
                'min' => -pow(2, 15),
                'max' => pow(2, 15) - 1,
            ),
            'tinyint' => array(
                'min' => 0,
                'max' => 255
            ),
        );
    }

    /**
     * @param string $type
     * @return array
     */
    public function getColumnRange($type)
    {
        return $this->getIntRanges()[mb_strtolower($type)];
    }

    private function createInventoryRecords()
    {
        global $adb;
        $moduleName = $this->module;

        $moduleHandler = vtws_getModuleHandlerFromName($moduleName, $this->user);
        $moduleMeta = $moduleHandler->getMeta();
        $moduleObjectId = $moduleMeta->getEntityId();
        $moduleFields = $moduleMeta->getModuleFields();
        $focus = CRMEntity::getInstance($moduleName);

        $tableName = ITS4YouImport_Utils_Helper::getDbTableName($this->user);
        $sql = 'SELECT * FROM ' . $tableName . ' WHERE status = ' . ITS4YouImport_Data_Action::$IMPORT_RECORD_NONE . ' GROUP BY subject';

        if ($this->batchImport) {
            $configModel = new ITS4YouImport_Config_Model();
            $importBatchLimit = $configModel->get("importBatchLimit");
            $sql .= ' LIMIT ' . $importBatchLimit;
        }
        $result = $adb->query($sql);
        $numberOfRecords = $adb->num_rows($result);

        if ($numberOfRecords <= 0) {
            return;
        }

        $fieldMapping = $this->fieldMapping;
        $fieldColumnMapping = $moduleMeta->getFieldColumnMapping();

        for ($i = 0; $i < $numberOfRecords; ++$i) {
            $row = $adb->raw_query_result_rowdata($result, $i);
            $rowId = $row['id'];
            $entityInfo = null;
            $fieldData = array();
            $lineItems = array();
            $subject = $row['subject'];
            $sql = 'SELECT * FROM ' . $tableName . ' WHERE status = ' . ITS4YouImport_Data_Action::$IMPORT_RECORD_NONE . ' AND subject = "' . str_replace("\"", "\\\"", $subject) . '"';
            $subjectResult = $adb->query($sql);
            $count = $adb->num_rows($subjectResult);
            $subjectRowIDs = array();
            $headerDataRowID = 0;
            for ($j = 0; $j < $count; ++$j) {
                $subjectRow = $adb->raw_query_result_rowdata($subjectResult, $j);
                array_push($subjectRowIDs, $subjectRow['id']);

                $lineItemData = array();
                foreach ($fieldMapping as $fieldName => $index) {
                    if ($moduleFields[$fieldName]->getTableName() == 'vtiger_inventoryproductrel') {
                        $lineItemData[$fieldName] = $subjectRow[$fieldName];
                    }
                }

                $this->currentItemLogEntry = new ITS4YouImport_DataLogEntry_Model($subjectRow['id'], $lineItemData);
                $this->logger->addLogEntry($this->currentItemLogEntry);
                if ($j == 0) {
                    $headerDataRowID = $subjectRow['id'];
                }

                if ($subjectRow['productid'] == '' || $subjectRow['quantity'] == '' || $subjectRow['listprice'] == '') {
                    $this->addLogError("LBL_PRODUCTS_MISSING");
                } else {
                    $lineItems[$subjectRow['id']] = $lineItemData;
                }
            }
            foreach ($fieldMapping as $fieldName => $index) {
                $fieldData[$fieldName] = $row[strtolower($fieldName)];
            }
            if (!array_key_exists('assigned_user_id', $fieldData)) {
                $fieldData['assigned_user_id'] = $this->user->id;
            }

            //enhance log entry data with the header item data
            //first for header log entry
            $headerLogEntry = $this->logger->getLogEntry($headerDataRowID);
            /* @var $headerLogEntry ITS4YouImport_DataLogEntry_Model */
            if ($headerLogEntry) {
                $headerLogEntry->setItemData(array_merge($fieldData, $headerLogEntry->getItemData()));
            }

            if (!empty($lineItems)) {
                //then also for all lineItems log entries, this is needed for purpose
                //of downloading the failed entries and their re-import

                foreach ($lineItems as $rowId => $lineItemData) {
                    $logEntry = $this->logger->getLogEntry($rowId);
                    if ($logEntry) {
                        $logEntry->setItemData(array_merge($fieldData, $logEntry->getItemData()));
                    }
                }


                $entityInfo = $this->createInventoryRecord($fieldData, $lineItems);

                //handling of case when header data import fails
                if ($headerLogEntry && $headerLogEntry->isNotOK()) {
                    foreach ($lineItems as $rowId => $lineItemData) {
                        $logEntry = $this->logger->getLogEntry($rowId);
                        $logEntry->addLogMessage(ITS4YouImport_LogMessage_Model::$LOG_MSG_TYPE_ERROR, "LBL_ERROR_HEADER_FAILED");
                    }
                }
            }


            if ($entityInfo == null) {
                $entityInfo = array('id' => null, 'status' => $this->getImportRecordStatus('failed'));
            }
            foreach ($subjectRowIDs as $id) {
                $this->importedRecordInfo[$id] = $entityInfo;
                $this->updateImportStatus($id, $entityInfo);
            }
        }

        unset($result);
        return true;
    }

    private function createInventoryRecord($inventoryFieldData, $lineItemDetails)
    {
        global $adb, $log;
        $moduleName = $this->module;
        $fieldMapping = $this->fieldMapping;

        $inventoryHandler = vtws_getModuleHandlerFromName($moduleName, $this->user);
        $inventoryMeta = $inventoryHandler->getMeta();
        $moduleFields = $inventoryMeta->getModuleFields();
        $isRecordExist = isRecordExistInDB($inventoryFieldData, $inventoryMeta, $this->user);

        $lineItemHandler = vtws_getModuleHandlerFromName('LineItem', $this->user);
        $lineItemMeta = $lineItemHandler->getMeta();
        $lineItems = array();

        $headerRowId = 0;
        $productNotExist = 0;
        foreach ($lineItemDetails as $index => $lineItemFieldData) {
            $rowId = $index;
            $this->currentItemLogEntry = $this->logger->getLogEntry($rowId);
            if ($headerRowId == 0) {
                $headerRowId = $rowId;
            }

            $isLineItemExist = isRecordExistInDB($lineItemFieldData, $lineItemMeta, $this->user);
            if ($isLineItemExist) {

                $count = $index;
                $lineItemData = array();
                $lineItemFieldData = $this->transformForImport($lineItemFieldData, $lineItemMeta);
                foreach ($fieldMapping as $fieldName => $index) {
                    if ($moduleFields[$fieldName]->getTableName() == 'vtiger_inventoryproductrel') {
                        $lineItemData[$fieldName] = $lineItemFieldData[$fieldName];
                        if ($fieldName != 'productid') {
                            $inventoryFieldData[$fieldName] = '';
                        }
                    }
                }
                array_push($lineItems, $lineItemData);
            } else {
                $productNotExist++;
            }
        }
        if (empty($lineItems)) {
            if ($productNotExist > 0) {
                $this->addLogError('LBL_PRODUCTS_NOT_EXIST');
            } else {
                $this->addLogError('LBL_PRODUCTS_MISSING');
            }
            return null;
        } elseif ($isRecordExist == false) {
            foreach ($lineItemDetails[$count] as $key => $value) {
                $inventoryFieldData[$key] = $value;
            }
        }

        $this->currentItemLogEntry = $this->logger->getLogEntry($headerRowId);
        $fieldData = $this->transformForImport($inventoryFieldData, $inventoryMeta);

        if (empty($fieldData) || empty($lineItemDetails)) {
            return null;
        }
        if ($fieldData['currency_id'] == ' ') {
            $fieldData['currency_id'] = '1';
        }
        if (!isset($fieldData['hdnTaxType'])) { //update 600.700.1.1C
            $fieldData['hdnTaxType'] = '1';
        }
        $fieldData['LineItems'] = $lineItems;

        $webserviceObject = VtigerWebserviceObject::fromName($adb, $moduleName);
        $inventoryOperation = new VtigerInventoryOperation($webserviceObject, $this->user, $adb, $log);
        $entityInfo = $inventoryOperation->create($moduleName, $fieldData);
        $entityInfo['status'] = $this->getImportRecordStatus('created');

        //add entityInfo to all lineItems logEntries
        foreach ($lineItemDetails as $rowId => $lineItemFieldData) {
            $logEntry = $this->logger->getLogEntry($rowId);
            if ($logEntry) {
                $logEntry->addEntityInfo($entityInfo);
            }
        }

        return $entityInfo;
    }

    public function addLogWarning($message, $msgParams = array())
    {
        $this->addLogMessage2LogEntry(new ITS4YouImport_LogMessage_Model(
            ITS4YouImport_LogMessage_Model::$LOG_MSG_TYPE_WARNING, $message, $msgParams
        ));
    }

}
