<?php
/* * *******************************************************************************
 * The content of this file is subject to the MultiWarehouses4You license.
 * ("License"); You may not use this file except in compliance with the License
 * The Initial Developer of the Original Code is IT-Solutions4You s.r.o.
 * Portions created by IT-Solutions4You s.r.o. are Copyright(C) IT-Solutions4You s.r.o.
 * All Rights Reserved.
 * ****************************************************************************** */

class ITS4YouWarehouses_ProductMovementsListView_Model extends Vtiger_Base_Model
{

    protected $productId;
    protected $headerFields = array();
    protected $warehouseId;

    public function __construct($values = array())
    {
        parent::__construct($values);

        $this->productId = false;
        $this->warehouseId = false;
        $className = 'ITS4YouWarehouses_DummyListViewField_Model';

        $this->headerFields['doc_date'] = new $className('doc_date', 'doc_date', 'LBL_DATE', 5);

        $this->headerFields['qty'] = new $className('qty', 'qty', 'Qty', 7);
        $this->headerFields['qty']->set('signed', true);

        $this->headerFields['setype'] = new $className('setype', 'setype', 'Type', 2);

        $this->headerFields['record_no'] = new $className('record_no', 'record_no', 'Record No', 2);
        $this->headerFields['record_no']->set('navigatable', true);

        $this->headerFields['warehouse_name'] = new $className('warehouse_name', 'warehouse_name', 'Warehouse', 2);
        $this->headerFields['warehouse_name']->set('navigatable', false);

        $this->headerFields['assigned_user_id'] = new $className('assigned_user_id', 'smownerid', 'Assigned To', 53);
    }

    /**
     * @param $productId
     */
    public function setProductId($productId)
    {
        $this->productId = $productId;
    }

    /**
     * @param $warehouseId
     */
    public function setWarehouseId($warehouseId)
    {
        $this->warehouseId = $warehouseId;
    }

    /**
     * @return array
     */
    public function getHeaders()
    {
        return $this->headerFields;
    }

    /**
     * @param $pagingModel
     * @return array
     * @throws Exception
     */
    public function getEntries($pagingModel)
    {
        $db = PearDatabase::getInstance();
        $searchParams = $this->get('search_params');

        if (empty($searchParams)) {
            $searchParams = array();
        }
        list($searchQuery, $params) = $this->getSearchQuery($searchParams);
        $listQuery = $this->getQuery() . $searchQuery;
        $orderBy = $this->getForSql('orderby');
        $sortOrder = $this->getForSql('sortorder');
        $startIndex = $pagingModel->getStartIndex();
        $pageLimit = $pagingModel->getPageLimit();

        if (!empty($orderBy)) {
            $listQuery .= ' ORDER BY ' . $orderBy . ' ' . $sortOrder;
        } else {
            $listQuery .= ' ORDER BY modifiedtime DESC';
        }
        $listQuery .= " LIMIT $startIndex," . ($pageLimit + 1);
        $listResult = $db->pquery($listQuery, $params);
        $listViewRecordModels = array();
        $listViewEntries = $this->getListViewRecords($listResult);
        $pagingModel->calculatePageRange($listViewEntries);

        if ($db->num_rows($listResult) > $pageLimit) {
            array_pop($listViewEntries);
            $pagingModel->set('nextPageExists', true);
        } else {
            $pagingModel->set('nextPageExists', false);
        }
        $index = 0;

        foreach ($listViewEntries as $recordId => $record) {
            $rawData = $db->query_result_rowdata($listResult, $index++);
            $record['id'] = $recordId;
            $listViewRecordModels[$recordId] = $this->getRecordFromArray($record, $rawData);
        }
        return $listViewRecordModels;
    }

    /**
     * @param $searchParams
     * @return array
     */
    private function getSearchQuery($searchParams)
    {
        $searchQuery = '';
        $params = array(null);

        if ($this->productId !== false) {
            $params = array($this->productId, $this->productId, $this->productId);
        }

        if ($this->warehouseId !== false) {
            $params = array_merge($params, array($this->warehouseId, $this->warehouseId, $this->warehouseId));
        }

        if (isset($searchParams[0]['columns'])) {
            $subSearchParams = $searchParams[0]['columns'];

            foreach ($subSearchParams as $searchParamsArr) {

                switch ($searchParamsArr['columnname']) {
                    case 'record_no':
                        $searchQuery .= '(receiptcards_no LIKE ? OR issuecards_no LIKE ? OR warehousetransfer_no LIKE ?)';
                        $params = array_merge($params, array('%' . $searchParamsArr['value'] . '%', '%' . $searchParamsArr['value'] . '%', '%' . $searchParamsArr['value'] . '%'));
                        break;
                    case 'qty':
                        $searchQuery .= '(vtiger_inventoryproductrel.quantity=? OR vtiger_inventoryproductrel.quantity * vtiger_inventorysubproductrel.quantity=?)';
                        $params = array_merge($params, array($searchParamsArr['value'], $searchParamsArr['value']));
                        break;
                    case 'smownerid':
                        $searchQuery .= '(first_name LIKE ? OR last_name LIKE ? OR groupname LIKE ?)';
                        $params = array_merge($params, array('%' . $searchParamsArr['value'] . '%', '%' . $searchParamsArr['value'] . '%', '%' . $searchParamsArr['value'] . '%'));
                        break;
                    case 'warehouse_name':
                        $searchQuery .= '(rcWH.warehouse_name LIKE ? OR wtWH.warehouse_name LIKE ? OR icWH.warehouse_name LIKE ?)';
                        $params = array_merge($params, array('%' . $searchParamsArr['value'] . '%', '%' . $searchParamsArr['value'] . '%', '%' . $searchParamsArr['value'] . '%'));
                        break;
                    case 'doc_date':
                        $searchQuery .= ' (its4you_receiptcards.adoption_date BETWEEN ? AND ? OR its4you_warehousetransfers.duedate BETWEEN ? AND ? OR its4you_issuecards.issue_date BETWEEN ? AND ?)';
                        list($date_start, $date_end) = explode(',', $searchParamsArr['value']);
                        $date_start = DateTimeField::convertToDBFormat($date_start);
                        $date_end = DateTimeField::convertToDBFormat($date_end);
                        $params = array_merge($params, array($date_start, $date_end, $date_start, $date_end, $date_start, $date_end));
                        break;
                    case 'referenceid':
                        $searchQuery .= ' rcSO.subject LIKE ? ';
                        array_push($params, '%' . $searchParamsArr['value'] . '%');
                        break;
                    default:
                        $searchQuery .= $searchParamsArr['columnname'] . ' LIKE ? ';
                        array_push($params, '%' . $searchParamsArr['value'] . '%');
                        break;
                }

                if (isset($searchParamsArr['column_condition'])) {
                    $searchQuery .= $searchParamsArr['column_condition'] . ' ';
                }
            }
        }

        if (!empty($searchQuery)) {
            $searchQuery = ' AND ' . $searchQuery;
        }
        return array($searchQuery, $params);
    }

    /**
     * @return string
     */
    private function getQuery()
    {
        $query = 'SELECT vtiger_inventoryproductrel.sequence_no,
                        vtiger_crmentity.modifiedtime,
                        vtiger_crmentity.setype,
                        vtiger_crmentity.crmid,
                        vtiger_crmentity.smownerid,
                        CASE    WHEN vtiger_inventorysubproductrel.productid > 0 THEN vtiger_inventoryproductrel.quantity * vtiger_inventorysubproductrel.quantity
                                ELSE vtiger_inventoryproductrel.quantity 
                        END AS qty,
                        CASE    WHEN vtiger_crmentity.setype = "ITS4YouReceiptcards" THEN its4you_receiptcards.receiptcards_no 
                                WHEN vtiger_crmentity.setype = "ITS4YouWarehouseTransfers" THEN its4you_warehousetransfers.warehousetransfer_no 
                                ELSE its4you_issuecards.issuecards_no 
                        END AS record_no,
                        CASE    WHEN vtiger_crmentity.setype = "ITS4YouReceiptcards" THEN "+" 
                                ELSE "-" 
                        END AS "sign",
                        CASE    WHEN vtiger_users.user_name != "" THEN CONCAT_WS(" ", vtiger_users.first_name, vtiger_users.last_name) 
                                ELSE vtiger_groups.groupname 
                        END AS assigned_user_id,                        
                        CASE    WHEN vtiger_crmentity.setype = "ITS4YouReceiptcards" THEN rcWH.warehouse_name 
                                WHEN vtiger_crmentity.setype = "ITS4YouWarehouseTransfers" THEN wtWH.warehouse_name
                                ELSE icWH.warehouse_name 
                        END AS warehouse_name,
                        CASE    WHEN vtiger_crmentity.setype = "ITS4YouReceiptcards" THEN its4you_receiptcards.adoption_date 
                                WHEN vtiger_crmentity.setype = "ITS4YouWarehouseTransfers" THEN its4you_warehousetransfers.duedate
                                ELSE its4you_issuecards.issue_date 
                        END AS doc_date
                    FROM vtiger_inventoryproductrel
                    INNER JOIN vtiger_crmentity ON vtiger_crmentity.crmid = vtiger_inventoryproductrel.id
                    LEFT JOIN vtiger_inventorysubproductrel ON vtiger_inventorysubproductrel.id=vtiger_inventoryproductrel.id AND vtiger_inventorysubproductrel.sequence_no=vtiger_inventoryproductrel.sequence_no AND vtiger_inventorysubproductrel.productid=?  
                    LEFT JOIN its4you_receiptcards ON vtiger_inventoryproductrel.id = its4you_receiptcards.receiptcardid
                    LEFT JOIN its4you_issuecards ON vtiger_inventoryproductrel.id = its4you_issuecards.issuecardid
                    LEFT JOIN its4you_warehousetransfers ON vtiger_inventoryproductrel.id = its4you_warehousetransfers.warehousetransferid
                    LEFT JOIN vtiger_users ON vtiger_users.id = vtiger_crmentity.smownerid
                    LEFT JOIN vtiger_groups ON vtiger_groups.groupid = vtiger_crmentity.smownerid
                    LEFT JOIN its4you_warehouses AS rcWH ON rcWH.warehouseid = its4you_receiptcards.warehouseid
                    LEFT JOIN its4you_warehouses AS wtWH ON wtWH.warehouseid = its4you_warehousetransfers.fromwarehouseid
                    LEFT JOIN its4you_warehouses AS icWH ON icWH.warehouseid = its4you_issuecards.warehouseid
                    WHERE vtiger_crmentity.deleted = 0 
                        AND vtiger_crmentity.setype IN ("ITS4YouReceiptcards", "ITS4YouIssuecards", "ITS4YouWarehouseTransfers") 
                        AND (
                            its4you_receiptcards.receiptcardstatus = "Delivered" 
                            OR its4you_issuecards.issuecardstatus = "Transferred from Warehouse" 
                            OR its4you_issuecards.issuecardstatus = "Delivered" 
                            OR its4you_warehousetransfers.wtstatus = "Transferred from Warehouse" 
                            OR its4you_warehousetransfers.wtstatus = "Delivered"
                        )';

        if ($this->productId !== false) {
            $query .= ' AND (vtiger_inventoryproductrel.productid=? OR vtiger_inventorysubproductrel.productid=?) ';
        }

        if ($this->warehouseId !== false) {
            $query .= ' AND (rcWH.warehouseid = ? OR icWH.warehouseid = ? OR wtWH.warehouseid = ?) ';
        }
        return $query;
    }

    /**
     * @param $_dbResult
     * @return array
     */
    private function getListViewRecords($_dbResult)
    {
        $db = PearDatabase::getInstance();
        $listViewEntries = array();

        while ($row = $db->fetchByAssoc($_dbResult)) {

            foreach ($row as $key => $val) {

                switch ($key) {
                    case 'doc_date':
                        $tmpVal = DateTimeField::convertToUserFormat($val);
                        break;
                    case 'qty':
                        $tmpVal = ITS4YouWarehouses_Base_ActionBlock::formatNumberForDetailView($val);
                        break;
                    case 'referenceid':
                        $tmpVal = $this->getRecordLink($val);
                        break;
                    default:
                        $tmpVal = $val;
                }
                $listViewEntries[$row['crmid'] . '_' . $row['sequence_no']][$key] = $tmpVal;
            }
        }
        return $listViewEntries;
    }

    public function getRecordLink($record)
    {
        if (empty($record) || !isRecordExists($record)) {
            return '';
        }

        $module = getSalesEntityType($record);

        if(empty($module)) {
            return '';
        }

        $recordModel = Vtiger_Record_Model::getInstanceById($record, $module);

        return '<a href="' . $recordModel->getDetailViewUrl() . '">' . $recordModel->getName() . '</a>';
    }

    /**
     * @param $valueArray
     * @param $rawData
     * @return mixed
     */
    private function getRecordFromArray($valueArray, $rawData)
    {
        return (new ITS4YouWarehouses_ProductMovementsRecord_Model())->setData($valueArray)->setRawData($rawData);
    }

    /**
     * @return array
     * @throws Exception
     */
    public function getProductMovementsOverview()
    {
        $db = PearDatabase::getInstance();
        list($searchQuery, $params) = $this->getSearchQuery(array());
        $listQuery = $this->getQuery() . $searchQuery;
        $listResult = $db->pquery($listQuery, $params);
        $total = 0;

        while ($row = $db->fetchByAssoc($listResult)) {

            if ('+' === $row['sign']) {
                $total += $row['qty'];
            } else {
                $total -= $row['qty'];
            }
        }

        if ($this->warehouseId === false) {
            $recordModel = Vtiger_Record_Model::getInstanceById($this->productId);
            $qtyinstock = $recordModel->get('qtyinstock');
        } else {
            $whRecordModel = Vtiger_Record_Model::getInstanceById($this->warehouseId);
            /* @var $whRecordModel ITS4YouWarehouses_Record_Model */
            $qtyinstock = $whRecordModel->getProductQtyInStock($this->productId);
        }

        return array(
            'total' => ITS4YouWarehouses_Base_ActionBlock::formatNumberForDetailView($total),
            'qtyinstock' => ITS4YouWarehouses_Base_ActionBlock::formatNumberForDetailView($qtyinstock)
        );
    }

    /**
     * @return array
     */
    public function getLinks()
    {
        return array();
    }

    //since this function can handle only one module at time then both modules (tabid1 and tabid2) must be public in order to have all records visible

    /**
     * @return int
     * @throws Exception
     */
    public function getRelatedEntriesCount()
    {
        return $this->getListViewCount();
    }

    /**
     * Function to get the list view entries
     * @return int
     * @throws Exception
     */
    public function getListViewCount()
    {
        $db = PearDatabase::getInstance();

        $searchParams = $this->get('search_params');
        if (empty($searchParams)) {
            $searchParams = array();
        }

        list($searchQuery, $params) = $this->getSearchQuery($searchParams);
        $listQuery = $this->getQuery() . $searchQuery;

        $position = stripos($listQuery, 'FROM');
        if ($position) {
            $explode = explode('FROM', $listQuery);
            unset($explode[0]);

            $listQuery = 'SELECT count(*) AS count FROM ' . implode('FROM', $explode);
        }

        $listResult = $db->pquery($listQuery, $params);
        return $db->query_result($listResult, 0, 'count');
    }

    /**
     * @param $modulesArray
     * @return string
     */
    private function getListViewSecurityParameterComplex($modulesArray)
    {
        $sec_query = '';

        if (count($modulesArray) > 0) {
            $sec_query .= ' AND (';

            foreach ($modulesArray as $module) {
                $sec_query .= $this->getListViewSecurityParameter($module) . ' OR ';
            }
            $sec_query = rtrim($sec_query, ' OR ') . ') ';
        }
        return $sec_query;
    }

    /**
     * @param $module
     * @return string
     */
    private function getListViewSecurityParameter($module)
    {
        global $current_user;
        $is_admin = $profileGlobalPermission = $defaultOrgSharingPermission = $current_user_parent_role_seq = $current_user_groups = false;
        require 'user_privileges/user_privileges_' . $current_user->id . '.php';
        require 'user_privileges/sharing_privileges_' . $current_user->id . '.php';

        $tabid = getTabid($module);
        $sec_query = " (vtiger_crmentity.setype = '" . $module . "'";

        if ($is_admin == false && $profileGlobalPermission[1] == 1 && $profileGlobalPermission[2] == 1 && $defaultOrgSharingPermission[$tabid] == 3) {
            $sec_query .= " AND 
                        (vtiger_crmentity.smownerid IN (" . $current_user->id . " )
                            OR vtiger_crmentity.smownerid IN
                             (
                                 SELECT vtiger_user2role.userid FROM vtiger_user2role
                                 INNER JOIN vtiger_users ON vtiger_users.id=vtiger_user2role.userid
                                 INNER JOIN vtiger_role ON vtiger_role.roleid=vtiger_user2role.roleid
                                 WHERE vtiger_role.parentrole LIKE '" . $current_user_parent_role_seq . "::%'
                             )
                             OR vtiger_crmentity.smownerid IN
                             (
                                 SELECT shareduserid FROM vtiger_tmp_read_user_sharing_per
                                 WHERE userid=" . $current_user->id . " AND tabid=" . $tabid . "
                             )
                             OR
                             (";

            // Build the query based on the group association of current user.
            if (sizeof($current_user_groups) > 0) {
                $sec_query .= " vtiger_groups.groupid IN (" . implode(",", $current_user_groups) . ") OR ";
            }
            $sec_query .= " vtiger_groups.groupid IN
						(
                                                    SELECT vtiger_tmp_read_group_sharing_per.sharedgroupid
                                                    FROM vtiger_tmp_read_group_sharing_per
                                                    WHERE userid=" . $current_user->id . " and tabid=" . $tabid . "
						)";
            $sec_query .= ")
                    )";
        }
        $sec_query .= ")";

        return $sec_query;
    }

}

