<?php
/* * *******************************************************************************
 * The content of this file is subject to the MultiWarehouses4You license.
 * ("License"); You may not use this file except in compliance with the License
 * The Initial Developer of the Original Code is IT-Solutions4You s.r.o.
 * Portions created by IT-Solutions4You s.r.o. are Copyright(C) IT-Solutions4You s.r.o.
 * All Rights Reserved.
 * ****************************************************************************** */

require_once __dir__ . '/../handlers/OverallQtyInfo.php';

class ITS4YouWarehouses_QtyRecalculator_Helper
{
    protected $productId = false;
    protected $warehouseId = false;
    protected $warehouseIds = array();
    protected $products = array();
    protected $updateProducts = array();
    protected $db;

    /**
     * @throws Exception
     */
    public function recalculateAll()
    {
        $this->recalculateWarehouses();
        OverallQtyInfoHandler::recalculate();
    }

    protected function recalculateWarehouses($productId = false, $warehouseId = false)
    {
        $this->db = PearDatabase::getInstance();
        $this->productId = $productId;
        $this->warehouseId = $warehouseId;

        $this->retrieveReceiptCards();
        $this->retrieveIssueCards();
        $this->retrieveWarehouseTransfers();
        $this->retrieveReceiptCardsSubProducts();
        $this->retrieveIssueCardsSubProducts();
        $this->deleteWarehouseRelations();

        if (count($this->warehouseIds) > 0) {
            $this->retrieveUpdateProducts($warehouseId);
        }

        $this->updateProductsQuantityInStock($productId, $warehouseId);
    }

    public function retrieveReceiptCards()
    {
        $sql = "SELECT warehouseid, vtiger_inventoryproductrel.productid, sum(vtiger_inventoryproductrel.quantity) AS qty
            FROM its4you_receiptcards 
            INNER JOIN vtiger_crmentity ON vtiger_crmentity.crmid = its4you_receiptcards.receiptcardid
            INNER JOIN vtiger_inventoryproductrel ON vtiger_inventoryproductrel.id = its4you_receiptcards.receiptcardid
            WHERE vtiger_crmentity.deleted = 0 AND its4you_receiptcards.receiptcardstatus = 'Delivered' "
            . $this->getInventoryRelCondition() .
            ' GROUP BY productid, warehouseid';
        $result = $this->db->query($sql);

        if ($this->db->num_rows($result)) {
            while ($row = $this->db->fetchByAssoc($result)) {
                $warehouseId = $row['warehouseid'];
                $productId = $row['productid'];
                $quantity = $this->isProductsBundle($productId) ? 0 : $row['qty'];

                $this->setWarehouseIds($warehouseId);
                $this->setProduct($warehouseId, $productId);
                $this->increaseProductTotal($warehouseId, $productId, $quantity);

                $this->increaseProductFromReceiptCards($warehouseId, $productId, $quantity);
            }
        }
    }

    public function getInventoryRelCondition()
    {
        if (false !== $this->productId) {
            return ' AND vtiger_inventoryproductrel.productid = ' . $this->productId . ' ';
        } elseif (false !== $this->warehouseId) {
            return ' AND warehouseid = ' . $this->warehouseId . ' ';
        }

        return '';
    }

    public function setWarehouseIds($record)
    {
        if (!in_array($record, $this->warehouseIds)) {
            $this->warehouseIds[] = $record;
        }
    }

    public function setProduct($warehouseId, $productId)
    {
        if (!isset($this->products[$warehouseId][$productId])) {
            $this->products[$warehouseId][$productId] = array(
                'from_deliverynotes' => 0,
                'from_receiptcards' => 0,
                'in_transfer' => 0,
                'total' => 0
            );
        }
    }

    /**
     * @param int $warehouseId
     * @param int $productId
     * @param float $quantity
     */
    public function increaseProductFromReceiptCards($warehouseId, $productId, $quantity)
    {
        $this->products[$warehouseId][$productId]['from_receiptcards'] += $quantity;
    }

    /**
     * @param int $warehouseId
     * @param int $productId
     * @param float $quantity
     */
    public function increaseProductTotal($warehouseId, $productId, $quantity)
    {
        $this->products[$warehouseId][$productId]['total'] += $quantity;
    }

    public function retrieveIssueCards()
    {
        $sql = "SELECT warehouseid, vtiger_inventoryproductrel.productid, sum(vtiger_inventoryproductrel.quantity) AS qty
            FROM its4you_issuecards
            INNER JOIN vtiger_crmentity ON vtiger_crmentity.crmid = its4you_issuecards.issuecardid
            INNER JOIN vtiger_inventoryproductrel ON vtiger_inventoryproductrel.id = its4you_issuecards.issuecardid
            WHERE vtiger_crmentity.deleted = 0 AND (its4you_issuecards.issuecardstatus = 'Transferred from Warehouse' OR its4you_issuecards.issuecardstatus = 'Delivered') "
            . $this->getInventoryRelCondition() .
            ' GROUP BY productid, warehouseid';
        $result = $this->db->query($sql);

        if ($this->db->num_rows($result)) {
            while ($row = $this->db->fetchByAssoc($result)) {
                $warehouseId = $row['warehouseid'];
                $productId = $row['productid'];
                $quantity = $this->isProductsBundle($productId) ? 0 : $row['qty'];

                $this->setWarehouseIds($warehouseId);
                $this->setProduct($warehouseId, $productId);
                $this->increaseProductFromDeliveryNotes($warehouseId, $productId, $quantity);
                $this->decreaseProductTotal($warehouseId, $productId, $quantity);
            }
        }
    }

    /**
     * @param int $warehouseId
     * @param int $productId
     * @param float $quantity
     */
    public function increaseProductFromDeliveryNotes($warehouseId, $productId, $quantity)
    {
        $this->products[$warehouseId][$productId]['from_deliverynotes'] += $quantity;
    }

    /**
     * @param int $warehouseId
     * @param int $productId
     * @param float $quantity
     */
    public function decreaseProductTotal($warehouseId, $productId, $quantity)
    {
        $this->products[$warehouseId][$productId]['from_deliverynotes'] += $quantity;
        $this->products[$warehouseId][$productId]['total'] -= $quantity;
    }

    public function retrieveWarehouseTransfers()
    {
        $sql = "SELECT its4you_warehousetransfers.wtstatus, its4you_warehousetransfers.fromwarehouseid, vtiger_inventoryproductrel.productid, sum(vtiger_inventoryproductrel.quantity) AS qty
            FROM its4you_warehousetransfers 
            INNER JOIN vtiger_crmentity ON vtiger_crmentity.crmid = its4you_warehousetransfers.warehousetransferid
            INNER JOIN vtiger_inventoryproductrel ON vtiger_inventoryproductrel.id = its4you_warehousetransfers.warehousetransferid
            WHERE vtiger_crmentity.deleted = 0 AND (its4you_warehousetransfers.wtstatus = 'Transferred from Warehouse' OR its4you_warehousetransfers.wtstatus = 'Delivered') "
            . $this->getInventoryRelConditionWHT() .
            ' GROUP BY productid, fromwarehouseid';
        $result = $this->db->query($sql);

        if ($this->db->num_rows($result)) {
            while ($row = $this->db->fetchByAssoc($result)) {
                $warehouseId = $row['fromwarehouseid'];
                $productId = $row['productid'];
                $quantity = $row['qty'];

                $this->setWarehouseIds($warehouseId);
                $this->setProduct($warehouseId, $productId);
                $this->decreaseProductTotal($warehouseId, $productId, $quantity);

                if ($row['wtstatus'] == 'Transferred from Warehouse') {
                    $in_transfer = $quantity;
                    $from_receiptcards = 0;
                } else {
                    $in_transfer = 0;
                    $from_receiptcards = $quantity;
                }

                $this->increaseProductFromDeliveryNotes($warehouseId, $productId, $quantity);
                $this->increaseProductInTransfer($warehouseId, $productId, $in_transfer);
                $this->increaseProductFromReceiptCards($warehouseId, $productId, $from_receiptcards);
            }
        }
    }

    public function isProductsBundle($productId)
    {
        $adb = PearDatabase::getInstance();
        $result = $adb->pquery('SELECT vtiger_products.productid, quantity FROM vtiger_products
					INNER JOIN vtiger_crmentity ON vtiger_crmentity.crmid = vtiger_products.productid AND vtiger_crmentity.deleted = 0
					LEFT JOIN vtiger_seproductsrel ON vtiger_seproductsrel.crmid = vtiger_products.productid AND vtiger_seproductsrel.setype = ?
					WHERE vtiger_seproductsrel.productid = ?',
            ['Products', $productId]
        );

        return boolval($adb->num_rows($result));
    }

    public function getInventoryRelConditionWHT()
    {
        if (false !== $this->productId) {
            return ' AND vtiger_inventoryproductrel.productid = ' . $this->productId . ' ';
        } elseif (false !== $this->warehouseId) {
            return ' AND fromwarehouseid = ' . $this->warehouseId . ' ';
        }

        return '';
    }

    /**
     * @param int $warehouseId
     * @param int $productId
     * @param float $quantity
     */
    public function increaseProductInTransfer($warehouseId, $productId, $quantity)
    {
        $this->products[$warehouseId][$productId]['in_transfer'] += $quantity;
    }

    public function retrieveIssueCardsSubProducts()
    {
        $sql = "SELECT warehouseid, vtiger_inventoryproductrel.productid AS product, vtiger_inventoryproductrel.quantity AS qty, vtiger_inventorysubproductrel.productid AS sub_product, vtiger_inventorysubproductrel.quantity as sub_qty
            FROM its4you_receiptcards 
            INNER JOIN vtiger_crmentity ON vtiger_crmentity.crmid = its4you_receiptcards.receiptcardid AND vtiger_crmentity.deleted = 0
            INNER JOIN vtiger_inventoryproductrel ON vtiger_inventoryproductrel.id = its4you_receiptcards.receiptcardid
            INNER JOIN vtiger_inventorysubproductrel ON vtiger_inventorysubproductrel.id=vtiger_inventoryproductrel.id AND vtiger_inventorysubproductrel.sequence_no=vtiger_inventoryproductrel.sequence_no
            WHERE its4you_receiptcards.receiptcardstatus = 'Delivered' "
            . $this->getInventoryRelConditionSubProduct();
        $result = $this->db->query($sql);

        while ($row = $this->db->fetchByAssoc($result)) {
            $warehouseId = $row['warehouseid'];
            $productId = $row['sub_product'];
            $quantity = $row['qty'] * $row['sub_qty'];

            $this->setWarehouseIds($warehouseId);
            $this->setProduct($warehouseId, $productId);
            $this->increaseProductTotal($warehouseId, $productId, $quantity);
        }
    }


    public function retrieveReceiptCardsSubProducts()
    {
        $sql = "SELECT warehouseid, vtiger_inventoryproductrel.productid AS product, vtiger_inventoryproductrel.quantity AS qty, vtiger_inventorysubproductrel.productid AS sub_product, vtiger_inventorysubproductrel.quantity as sub_qty
            FROM its4you_issuecards
            INNER JOIN vtiger_crmentity ON vtiger_crmentity.crmid=its4you_issuecards.issuecardid AND vtiger_crmentity.deleted = 0
            INNER JOIN vtiger_inventoryproductrel ON vtiger_inventoryproductrel.id=its4you_issuecards.issuecardid
            INNER JOIN vtiger_inventorysubproductrel ON vtiger_inventorysubproductrel.id=vtiger_inventoryproductrel.id AND vtiger_inventorysubproductrel.sequence_no=vtiger_inventoryproductrel.sequence_no
            WHERE (its4you_issuecards.issuecardstatus='Transferred from Warehouse' OR its4you_issuecards.issuecardstatus='Delivered') "
            . $this->getInventoryRelConditionSubProduct();
        $result = $this->db->query($sql);

        while ($row = $this->db->fetchByAssoc($result)) {
            $warehouseId = $row['warehouseid'];
            $productId = $row['sub_product'];
            $quantity = $row['qty'] * $row['sub_qty'];

            $this->setWarehouseIds($warehouseId);
            $this->setProduct($warehouseId, $productId);
            $this->decreaseProductTotal($warehouseId, $productId, $quantity);
        }
    }

    public function getInventoryRelConditionSubProduct()
    {
        if (false !== $this->productId) {
            return ' AND vtiger_inventorysubproductrel.productid=' . $this->productId . ' ';
        } elseif (false !== $this->warehouseId) {
            return ' AND warehouseid=' . $this->warehouseId . ' ';
        }

        return '';
    }

    public function deleteWarehouseRelations()
    {
        $this->db->query('DELETE FROM its4you_warehouses_productrel' . $this->getWHProductRelCondition());
    }

    public function getWHProductRelCondition()
    {
        if (false !== $this->productId) {
            return ' WHERE productid = ' . $this->productId;
        } elseif (false !== $this->warehouseId) {
            return ' WHERE warehouseid = ' . $this->warehouseId;
        }

        return '';
    }

    public function retrieveUpdateProducts($warehouseId)
    {
        foreach ($this->warehouseIds as $warehouseid) {
            if (count($this->products[$warehouseid]) > 0) {
                foreach ($this->products[$warehouseid] as $prodId => $pd) {
                    $sql_i = 'INSERT INTO its4you_warehouses_productrel (warehouseid, productid, quantity) VALUES (?,?,?)';
                    $this->db->pquery($sql_i, array($warehouseid, $prodId, $pd['total']));

                    if (!isset($this->updateProducts[$prodId])) {
                        $this->updateProducts[$prodId] = $pd['total'];
                    } else {
                        $this->updateProducts[$prodId] += $pd['total'];
                    }
                }
            }
        }
        //in case of recalculating one warehouse we need to have overall information about Inventory,
        //so all affected products need to have information about their quantity across all warehouses
        if ($warehouseId !== false) {
            foreach ($this->updateProducts as $prodId => $qty) {
                $this->updateProducts[$prodId] = 0;
            }

            $affectedProductIds = array_keys($this->updateProducts);
            $sql = 'SELECT productid, quantity FROM its4you_warehouses_productrel WHERE productid IN (' . generateQuestionMarks($affectedProductIds) . ')';
            $result = $this->db->pquery($sql, $affectedProductIds);

            while ($row = $this->db->fetchByAssoc($result)) {
                $this->updateProducts[$row['productid']] += $row['quantity'];
            }
        }
    }

    public function updateProductsQuantityInStock($productId = false, $warehouseId = false)
    {
        foreach ($this->updateProducts as $productId => $qtyInStock) {
            $this->db->pquery('UPDATE vtiger_products SET qtyinstock=? WHERE productid=?', array($qtyInStock, $productId));
        }
    }

    /**
     * @throws Exception
     */
    public function recalculateSingleProduct($productId)
    {
        $this->recalculateWarehouses($productId);
        OverallQtyInfoHandler::recalculate($productId);
    }

    /**
     * @throws Exception
     */
    public function recalculateSingleWarehouse($warehouseId)
    {
        $this->recalculateWarehouses(false, $warehouseId);
        OverallQtyInfoHandler::recalculate();
    }

}
