<?php

/* * *******************************************************************************
 * The content of this file is subject to the MultiWarehouses4You license.
 * ("License"); You may not use this file except in compliance with the License
 * The Initial Developer of the Original Code is IT-Solutions4You s.r.o.
 * Portions created by IT-Solutions4You s.r.o. are Copyright(C) IT-Solutions4You s.r.o.
 * All Rights Reserved.
 * ****************************************************************************** */

require_once 'include/utils/utils.php';
require_once 'include/events/include.inc';

require_once __DIR__ . '/OverallQtyInfo.php';

/**
 * Class UpdateWarehouseBeforeSaveEventHandler
 */
class UpdateWarehouseBeforeSaveEventHandler extends VTEventHandler
{

    /**
     *
     * @param string $handlerType
     * @param VTEntityData $entityData
     * @throws Exception
     * @global object $adb
     * @global object $log
     */
    public function handleEvent($handlerType, $entityData)
    {
        $allowedModuleNames = array('ITS4YouIssuecards', 'ITS4YouReceiptcards', 'ITS4YouWarehouseTransfers', 'PurchaseOrder', 'SalesOrder', 'Invoice', 'ITS4YouWHDeliveryNotes');
        $moduleName = $entityData->getModuleName();
        $is_new = $entityData->isNew();

        if (in_array($moduleName, $allowedModuleNames)) {
            if (!$is_new && ($moduleName == 'ITS4YouIssuecards' || $moduleName == 'ITS4YouReceiptcards' || $moduleName == 'ITS4YouWarehouseTransfers')) {
                StartActualizeProductsFromCard($entityData, $moduleName, 'before');
            }
            OverallQtyInfoHandler::recalculateAtEvent($entityData, $moduleName, true);
        }
    }

}

/**
 * Class UpdateWarehouseAfterSaveEventHandler
 */
class UpdateWarehouseAfterSaveEventHandler extends VTEventHandler
{

    /**
     * @param $handlerType
     * @param $entityData
     * @throws Exception
     */
    public function handleEvent($handlerType, $entityData)
    {
        $allowedModuleNames = array('ITS4YouIssuecards', 'ITS4YouReceiptcards', 'ITS4YouWarehouseTransfers', 'PurchaseOrder', 'SalesOrder', 'Invoice', 'ITS4YouWHDeliveryNotes');
        $moduleName = $entityData->getModuleName();

        if (in_array($moduleName, $allowedModuleNames)) {
            if ($moduleName == 'ITS4YouIssuecards' || $moduleName == 'ITS4YouReceiptcards' || $moduleName == 'ITS4YouWarehouseTransfers') {
                StartActualizeProductsFromCard($entityData, $moduleName, 'after');
            }
            OverallQtyInfoHandler::recalculateAtEvent($entityData, $moduleName, false);

            if ($moduleName == 'ITS4YouReceiptcards') {
                ITS4YouWarehouses_RCUtil_ActionBlock::updateRelatedPurchaseOrderStatus($entityData->getId());
            }
        }

        if(in_array($moduleName, ['ITS4YouIssuecards', 'ITS4YouReceiptcards', 'ITS4YouWarehouseTransfers'])) {
            OverallQtyInfoHandler::recalculateWarehousesProductsCount($entityData, $moduleName);
        }
    }

}

/**
 * Class UpdateWarehouseBeforeDeleteEventHandler
 */
class UpdateWarehouseBeforeDeleteEventHandler extends VTEventHandler
{

    /**
     * @param $handlerType
     * @param $entityData
     * @throws Exception
     */
    public function handleEvent($handlerType, $entityData)
    {
        //module ITS4YouWHDeliveryNotes should be incl. as well because deleting record of DN automatically deletes related IC
        $allowedModuleNames = array('ITS4YouIssuecards', 'ITS4YouReceiptcards', 'ITS4YouWarehouseTransfers', 'PurchaseOrder', 'SalesOrder', 'ITS4YouWHDeliveryNotes', 'Invoice');
        $moduleName = $entityData->getModuleName();

        if (in_array($moduleName, $allowedModuleNames)) {
            if ($moduleName == 'ITS4YouIssuecards' || $moduleName == 'ITS4YouReceiptcards' || $moduleName == 'ITS4YouWarehouseTransfers') {
                StartActualizeProductsFromCard($entityData, $moduleName, 'before');
            }
            OverallQtyInfoHandler::recalculateAtEvent($entityData, $moduleName, true);
        }
    }

}

/**
 * Class UpdateWarehouseAfterDeleteEventHandler
 */
class UpdateWarehouseAfterDeleteEventHandler extends VTEventHandler
{

    /**
     * @param $handlerType
     * @param $entityData
     * @throws Exception
     */
    public function handleEvent($handlerType, $entityData)
    {
        //we need to trigger recalculation of Overall Qty Info only for these two modules
        //because they are related to SO(IC) resp. PO(RC)
        //module ITS4YouWHDeliveryNotes should be incl. as well because deleting record of DN automatically deletes related IC
        $allowedModuleNames = array('ITS4YouIssuecards', 'ITS4YouReceiptcards', 'ITS4YouWHDeliveryNotes');
        $moduleName = $entityData->getModuleName();
        if (in_array($moduleName, $allowedModuleNames)) {
            OverallQtyInfoHandler::recalculateAtEvent($entityData, $moduleName, false);
        }

        if (in_array($moduleName, ['ITS4YouIssuecards', 'ITS4YouReceiptcards', 'ITS4YouWarehouseTransfers'])) {
            OverallQtyInfoHandler::recalculateWarehousesProductsCount($entityData, $moduleName);
        }
    }

}

/**
 * @param $entityData
 * @param $moduleName
 * @param $mode
 * @throws Exception
 */
function StartActualizeProductsFromCard($entityData, $moduleName, $mode)
{
    global $adb;

    $card_id = $entityData->getId();
    $data = $entityData->getData();

    switch ($moduleName) {
        case "ITS4YouIssuecards":
            $update_type = "";
            $warehouseid = $data["warehouseid"];

            $sql = "SELECT issuecardstatus FROM its4you_issuecards WHERE issuecardid = ?";
            $result = $adb->pquery($sql, array($card_id));
            $old_status = $adb->query_result($result, 0, "issuecardstatus");

            if ($mode == 'before') {
                if ($old_status == "Transferred from Warehouse" || $old_status == "Delivered") {
                    $update_type = "+";
                }
            } else {
                $status = $data['issuecardstatus'];

                if ($status == "Transferred from Warehouse") {
                    $update_type = "-";
                } elseif ($status == "Delivered" && $old_status != "Transferred from Warehouse") {
                    $update_type = "-";
                }
            }

            if ($update_type != "") {
                ActualizeProductsFromCard($card_id, $warehouseid, $update_type);
            }

            //handle withdrawal of DN status in case of IC status change   
            //delete is not handle because it should not be possible to delete related record
//            if ($mode == "after")
//                UpdateICRelatedDN($card_id);

            break;

        case "ITS4YouReceiptcards":
            $warehouseid = $data["warehouseid"];

            if ($mode == 'before') {
                $sql = "SELECT receiptcardstatus FROM its4you_receiptcards WHERE receiptcardid = ?";
                $result = $adb->pquery($sql, array($card_id));
                $status = $adb->query_result($result, 0, "receiptcardstatus");
                $update_type = "-";
            } else {
                $status = $data['receiptcardstatus'];
                $update_type = "+";
            }

            if ($status != "Delivered") {
                $update_type = "";
            }

            if ($update_type != "") {
                ActualizeProductsFromCard($card_id, $warehouseid, $update_type);
            }
            break;

        case "ITS4YouWarehouseTransfers":
            $from_warehouseid = $data["fromwarehouseid"];
            $to_warehouseid = $data["towarehouseid"];

            $update_type_from = "";
            $update_type_to = "";

            $new_status = $data['wtstatus'];

            $sql = "SELECT wtstatus FROM its4you_warehousetransfers WHERE warehousetransferid = ?";
            $result = $adb->pquery($sql, array($card_id));
            $old_status = $adb->num_rows($result) > 0 ? $adb->query_result($result, 0, "wtstatus") : "";

            if ($mode == "before") {
                if ($old_status == "Transferred from Warehouse" || $old_status == "Delivered") {
                    $update_type_from = "+";
                    $update_type_to = "";
                }
            } else {
                if ($new_status == "Canceled") {
                    if ($old_status == "Transferred from Warehouse" || $old_status == "Delivered") {
                        $update_type_from = "+";
                    }
                    $update_type_to = "-";
                } else {
                    if ($new_status == "Transferred from Warehouse" || $new_status == "Delivered") {
                        $update_type_from = "-";
                    }
                    $update_type_to = "+";
                }
            }

            if ($update_type_from != "") {
                ActualizeProductsFromCard($card_id, $from_warehouseid, $update_type_from);
            }
            if ($update_type_to != "") {
                UpdateWTRelatedRC($entityData, $to_warehouseid, $update_type_to);
            }
            break;
    }
}

/**
 * @param $entityData
 * @param $warehouseId
 * @param $updateType
 * @throws Exception
 */
function UpdateWTRelatedRC($entityData, $warehouseId, $updateType)
{
    global $adb, $current_user;

    $transferId = $entityData->getId();
    $data = $entityData->getData();

    $transferModule = 'ITS4YouWarehouseTransfers';
    $receiptModule = 'ITS4YouReceiptcards';

    $sql = 'SELECT relcrmid FROM vtiger_crmentityrel WHERE crmid=? AND module=? AND relmodule=?';
    $result = $adb->pquery($sql, array($transferId, $transferModule, $receiptModule));
    $num_rows = $adb->num_rows($result);

    if ('+' === $updateType) {
        //if there is no related RC
        if ($num_rows == 0) {
            /**
             * @var ITS4YouReceiptcards_Record_Model $recordModel
             */
            $warehouseAssignedToId = ITS4YouWarehouses_Record_Model::getOwnerId($warehouseId);
            $recordModel = Vtiger_Record_Model::getCleanInstance($receiptModule);
            $recordModel->set('assigned_user_id', $warehouseAssignedToId);
            $recordModel->set('vendor_id', $transferId);
            $recordModel->set('warehouseid', $warehouseId);
            $recordModel->set('receiptcardstatus', 'Created');
            $recordModel->set('receiptcard_carrier', $data['wtcarrier']);
            $recordModel->set('tracking_no', $data['tracking_no']);
            $recordModel->set('receiptcardtype', 'Warehouse transfer');
            $recordModel->set('currency_id', $current_user->currency_id);
            $recordModel->set('conversion_rate', $current_user->conv_rate);
            $recordModel->save();

            //map WT to RC
            $relationModel = Vtiger_Relation_Model::getInstance(Vtiger_Module_Model::getInstance($transferModule), Vtiger_Module_Model::getInstance($receiptModule));
            $relationModel->addRelation($transferId, $recordModel->getId());
        } else {
            $recordModel = Vtiger_Record_Model::getInstanceById($adb->query_result($result, 0, 'relcrmid'), $receiptModule);
            $recordModel->set('mode', 'edit');
            $recordModel->set('referenceid', $transferId);
            $recordModel->set('receiptcardtype', 'Warehouse transfer');
            $recordModel->save();
            /* -------------- THERE IS NO NEED TO HANDLE PRODUCTS MANUALLY BECAUSE THEY ARE ALREADY HANDLED WITHIN THE RECEIPTCARD SAVE METHOD -------------- */
        }
    } else {
        if ($num_rows > 0) {
            $recordModel = Vtiger_Record_Model::getInstanceById($adb->query_result($result, 0, 'relcrmid'), $receiptModule);
            $recordModel->set('mode', 'edit');
            $recordModel->set('referenceid', $transferId);
            $recordModel->set('receiptcardtype', 'Warehouse transfer');
            $recordModel->set('receiptcardstatus', 'Canceled');
            $recordModel->set('description', vtranslate('LBL_WT_RC_CANCLED', $receiptModule) . $data['warehousetransfer_no'] . '.');
            $recordModel->save();
        }
    }
}

/**
 * @param $card_id
 * @throws Exception
 */
function UpdateICRelatedDN($card_id)
{
    global $adb;

    $sql = "SELECT its4you_issuecards.parent_id, its4you_deliverynotes.deliverynotestatus 
            FROM its4you_issuecards 
            INNER JOIN vtiger_crmentity
                    ON vtiger_crmentity.crmid = its4you_issuecards.parent_id
            INNER JOIN its4you_deliverynotes
                    ON its4you_deliverynotes.deliverynoteid = its4you_issuecards.parent_id
            WHERE vtiger_crmentity.deleted = 0              
              AND its4you_issuecards.issuecardid = ?";
    $result = $adb->pquery($sql, array($card_id));
    if ($adb->num_rows($result) > 0) {
        $dnId = $adb->query_result($result, 0, "parent_id");
        checkAndUpdateDNStatus($dnId);
    }
}

/**
 * @param $dnId
 * @throws Exception
 */
function checkAndUpdateDNStatus($dnId)
{
    global $adb;
    $sql = "SELECT its4you_deliverynotes.deliverynotestatus FROM its4you_deliverynotes WHERE deliverynoteid=?";
    $result1 = $adb->pquery($sql, array($dnId));
    $dnStatus = $adb->query_result($result1, 0, "deliverynotestatus");

    if ($dnStatus != "Canceled") {
        $sql = "SELECT its4you_issuecards.issuecardstatus, COUNT(its4you_issuecards.issuecardid) AS status_count
                FROM its4you_issuecards               
                INNER JOIN vtiger_crmentity
                   ON its4you_issuecards.issuecardid = vtiger_crmentity.crmid
                WHERE its4you_issuecards.parent_id = ?
                  AND vtiger_crmentity.deleted = 0
                GROUP BY its4you_issuecards.issuecardstatus";
        $result = $adb->pquery($sql, array($dnId));
        $allCount = 0;
        $statusCounts = array();
        while ($row = $adb->fetchByAssoc($result)) {
            $statusCounts[$row["issuecardstatus"]] = $row["status_count"];
            $allCount += $row["status_count"];
        }

        if ($dnStatus != "Ready to Deliver" && ($allCount == $statusCounts["Transferred from Warehouse"] || $allCount == $statusCounts["Transferred from Warehouse"] + $statusCounts["Delivered"])) {
            $newDnStatus = "Ready to Deliver";
        } elseif ($dnStatus != "Delivered" && $allCount == $statusCounts["Delivered"]) {
            $newDnStatus = "Delivered";
        } else {
            $newDnStatus = "Created";
        }

        if ($newDnStatus != "") {
            $sql = "UPDATE its4you_deliverynotes SET deliverynotestatus=? WHERE deliverynoteid=?";
            $adb->pquery($sql, array($newDnStatus, $dnId));
        }
    }
}

/**
 * @param $card_id
 * @param $warehouseid
 * @param $update_type
 * @return mixed
 * @throws Exception
 */
function ActualizeProductsFromCard($card_id, $warehouseid, $update_type)
{
    $adb = PearDatabase::getInstance();
    $result = $adb->pquery('SELECT * FROM vtiger_inventoryproductrel 
        INNER JOIN vtiger_crmentity ON vtiger_crmentity.crmid=vtiger_inventoryproductrel.id AND vtiger_crmentity.deleted=0
        WHERE id = ?',
        array($card_id)
    );

    if ($adb->num_rows($result)) {
        while ($row = $adb->fetchByAssoc($result)) {
            $quantity = UpdateSubProductInWarehouse($warehouseid, $update_type, $row) ? 0 : $row['quantity'];

            UpdateProductInWarehouse($warehouseid, $row['productid'], $update_type, $quantity);
        }
    }

    return $result;
}

/**
 * @throws Exception
 */
function UpdateSubProductInWarehouse($warehouseId, $updateType, $productRel)
{
    $adb = PearDatabase::getInstance();
    $result = $adb->pquery('SELECT * FROM vtiger_inventorysubproductrel
        INNER JOIN vtiger_crmentity ON vtiger_crmentity.crmid=vtiger_inventorysubproductrel.id AND vtiger_crmentity.deleted=0
        WHERE id=? AND sequence_no=?',
        array($productRel['id'], $productRel['sequence_no'])
    );

    if ($adb->num_rows($result)) {
        while ($row = $adb->fetchByAssoc($result)) {
            $subQty = $productRel['quantity'] * $row['quantity'];

            UpdateProductInWarehouse($warehouseId, $row['productid'], $updateType, $subQty);
        }

        return true;
    }

    return false;
}

/**
 * @param $warehouseid
 * @param $productid
 * @param $type
 * @param $new_quantity
 * @throws Exception
 */
function UpdateProductInWarehouse($warehouseid, $productid, $type, $new_quantity)
{
    global $adb;

    $sql = 'SELECT * FROM its4you_warehouses_productrel WHERE warehouseid=? AND productid=?';
    $result = $adb->pquery($sql, array($warehouseid, $productid));

    if ($adb->num_rows($result)) {
        $new = false;
        $quantity_in_warehouse = $adb->query_result($result, 0, 'quantity');
    } else {
        $new = true;
        $quantity_in_warehouse = 0;
    }

    if ('+' === $type) {
        $save_quantity = $quantity_in_warehouse + $new_quantity;
    } else {
        $save_quantity = $quantity_in_warehouse - $new_quantity;
    }

    if (!$new) {
        $sql_upr = 'UPDATE its4you_warehouses_productrel SET quantity=? WHERE warehouseid=? AND productid=?';
        $adb->pquery($sql_upr, array($save_quantity, $warehouseid, $productid));
    } else {
        $sql_ipr = 'INSERT INTO its4you_warehouses_productrel (warehouseid, productid, quantity) VALUES (?,?,?)';
        $adb->pquery($sql_ipr, array($warehouseid, $productid, $save_quantity));
    }

    UpdateProductInDetail($productid, $type, $new_quantity);
}

/**
 * @param $productid
 * @param $type
 * @param $new_quantity
 * @throws Exception
 */
function UpdateProductInDetail($productId, $type, $newQuantity)
{
    $adb = PearDatabase::getInstance();
    $result = $adb->pquery('SELECT qtyinstock FROM vtiger_products WHERE productid=?', array($productId));
    $qtyInStock = $adb->query_result($result, 0, 'qtyinstock');

    if ('+' === $type) {
        $saveQuantity = $qtyInStock + $newQuantity;
    } else {
        $saveQuantity = $qtyInStock - $newQuantity;
    }

    $adb->pquery('UPDATE vtiger_products SET qtyinstock = ? WHERE productid = ?',
        array($saveQuantity, $productId)
    );
}