<?php

/* * *******************************************************************************
 * The content of this file is subject to the MultiWarehouses4You license.
 * ("License"); You may not use this file except in compliance with the License
 * The Initial Developer of the Original Code is IT-Solutions4You s.r.o.
 * Portions created by IT-Solutions4You s.r.o. are Copyright(C) IT-Solutions4You s.r.o.
 * All Rights Reserved.
 * ****************************************************************************** */

class OverallQtyInfoHandler
{

    const TABLE_NAME = "its4you_wh_overall_qtyinfo";

    /**
     * Return single instance of OverallQtyInfoEntry
     * @param int $_productId
     * @return OverallQtyInfoEntry
     */
    public static function getQtyInfoEntry($_productId)
    {
        $tmpArr = self::getQtyInfoEntries(array($_productId));
        return reset($tmpArr);
    }

    /**
     * Returns single OverallQtyInfo entry
     * @param array $_productIds
     * @return array OverallQtyInfoEntry
     * @global PearDatabase $adb
     */
    public static function getQtyInfoEntries($_productIds)
    {
        global $adb;

        $qtyInfoEntries = array();

        if (!is_array($_productIds)) {
            $_productIds = array($_productIds);
        }
        if (count($_productIds) > 0) {
            $sql = "SELECT * FROM " . OverallQtyInfoHandler::TABLE_NAME . " WHERE productid IN (" . generateQuestionMarks($_productIds) . ")";
            $result = $adb->pquery($sql, $_productIds);
            while ($row = $adb->fetchByAssoc($result)) {
                $qtyInfoEntries[$row["productid"]] = new OverallQtyInfoEntry($row["productid"], $row["tobe_receipted"], $row["tobe_delivered"]);
            }

            foreach ($_productIds as $prodId) {
                if (!isset($qtyInfoEntries[$prodId])) {
                    $qtyInfoEntries[$prodId] = OverallQtyInfoEntry::createDummyEntry($prodId);
                }
            }
        }

        return $qtyInfoEntries;
    }

    public static function recalculateWarehousesProductsCount($entityData, $moduleName)
    {
        $sql = 'UPDATE its4you_warehouses SET products_in_warehouse = ? WHERE warehouseid = ?';
        $warehouseIds = array_filter([
            $entityData->get('warehouseid'),
            $entityData->get('fromwarehouseid'),
            $entityData->get('towarehouseid'),
        ]);

        foreach ($warehouseIds as $warehouseId) {
            if (empty($warehouseId)) {
                continue;
            }

            $listViewModel = new ITS4YouWarehouses_ProductsInWarehouseListView_Model(
                array('warehouseid' => $warehouseId)
            );
            $params = [
                $listViewModel->getRelatedEntriesCount(),
                $warehouseId
            ];

            PearDatabase::getInstance()->pquery($sql, $params);
        }
    }

    /**
     *
     * @param VTEntityData $_entityData
     * @param string $_moduleName
     * @param string $_beforeEvent
     * @return void
     * @throws Exception
     */
    public static function recalculateAtEvent($_entityData, $_moduleName, $_beforeEvent)
    {
        $qtyInfoEntries = array();
        switch ($_moduleName) {
            case "ITS4YouReceiptcards":
                if ($_beforeEvent && $_entityData->isNew()) {
                    if (isset($_REQUEST["qtyInfoHandlerPurchaseOrders"]) && count($_REQUEST["qtyInfoHandlerPurchaseOrders"]) > 0) {
                        $purchaseOrderIds = $_REQUEST["qtyInfoHandlerPurchaseOrders"];
                    } else {
                        $parentModuleName = getSalesEntityType($_entityData->get("referenceid"));
                        $purchaseOrderIds = ($parentModuleName == "PurchaseOrder" ? array($_entityData->get("referenceid") => $_entityData->get("referenceid")) : array());
                    }
                } else {
                    $purchaseOrderIds = ITS4YouWarehouses_RCUtil_ActionBlock::getRelatedPurchaseOrders($_entityData->getId());
                }

                if (count($purchaseOrderIds) > 0) {
                    $awaitingQty = ITS4YouWarehouses_Base_ActionBlock::getOverallAwaitingQty(array(), array_keys($purchaseOrderIds));
                    $productIds = array_keys($awaitingQty);
                    $qtyInfoEntries = self::getQtyInfoEntries($productIds);
                    foreach ($qtyInfoEntries as $prodId => $qtyInfoEntry) {
                        /* @var $qtyInfoEntry OverallQtyInfoEntry */
                        if ($_beforeEvent) {
                            $newQty = $qtyInfoEntry->getToBeReceiptedQty() - $awaitingQty[$prodId];
                        } else {
                            $newQty = $qtyInfoEntry->getToBeReceiptedQty() + $awaitingQty[$prodId];
                        }
                        $qtyInfoEntry->setToBeReceiptedQty($newQty);
                    }
                }
                break;

            case "PurchaseOrder":
                if ($_beforeEvent && $_entityData->isNew()) {
                    return;
                }

                $awaitingQty = ITS4YouWarehouses_Base_ActionBlock::getOverallAwaitingQty(array(), array($_entityData->getId()));

                $productIds = array_keys($awaitingQty);
                $qtyInfoEntries = self::getQtyInfoEntries($productIds);
                foreach ($qtyInfoEntries as $prodId => $qtyInfoEntry) {
                    /* @var $qtyInfoEntry OverallQtyInfoEntry */
                    if ($_beforeEvent) {
                        $newQty = $qtyInfoEntry->getToBeReceiptedQty() - $awaitingQty[$prodId];
                    } else {
                        $newQty = $qtyInfoEntry->getToBeReceiptedQty() + $awaitingQty[$prodId];
                    }
                    $qtyInfoEntry->setToBeReceiptedQty($newQty);
                }
                break;

            case "ITS4YouIssuecards":
                $deliveryNoteId = $_entityData->get("parent_id");
                $salesOrderIds = ($deliveryNoteId != "" ? ITS4YouWarehouses_DNUtil_ActionBlock::getRelatedSalesOrder($deliveryNoteId) : array());
                $invoiceIds = ($deliveryNoteId != "" ? ITS4YouWarehouses_DNUtil_ActionBlock::getRelatedInvoice($deliveryNoteId) : array());

                $toBeDeliveredQty = array();
                //first handle related SOs
                if (count($salesOrderIds) > 0) {
                    $toBeDeliveredQty = ITS4YouWarehouses_Base_ActionBlock::getSOToBeDeliveredQty(array_keys($salesOrderIds));
                } elseif (count($invoiceIds) > 0) {
                    //if there is no related SOs then handle related INVs (process begins in INV) => see the commented code below, the logic is the same
                    $toBeDeliveredQty = ITS4YouWarehouses_Base_ActionBlock::getINVToBeDeliveredQty(array_keys($invoiceIds));
                }

                if (count($toBeDeliveredQty) > 0) {
                    $productIds = array_keys($toBeDeliveredQty);
                    $qtyInfoEntries = self::getQtyInfoEntries($productIds);
                    foreach ($qtyInfoEntries as $prodId => $qtyInfoEntry) {
                        /* @var $qtyInfoEntry OverallQtyInfoEntry */
                        if ($_beforeEvent) {
                            $newQty = $qtyInfoEntry->getToBeDeliveredQty() - $toBeDeliveredQty[$prodId];
                        } else {
                            $newQty = $qtyInfoEntry->getToBeDeliveredQty() + $toBeDeliveredQty[$prodId];
                        }
                        $qtyInfoEntry->setToBeDeliveredQty($newQty);
                    }
                }
                break;


            case "ITS4YouWHDeliveryNotes":
                //module ITS4YouWHDeliveryNotes should be incl. as well because deleting record of DN automatically deletes related IC
                //creation of DN is directly followed by creation of IC, therefore we omit the adjustment of products for new DN, only
                //Services are handled
                if (!$_entityData->isNew()) {
                    $deliveryNoteId = $_entityData->getId();
                    $salesOrderIds = ($deliveryNoteId != "" ? ITS4YouWarehouses_DNUtil_ActionBlock::getRelatedSalesOrder($deliveryNoteId) : array());
                    $invoiceIds = ($deliveryNoteId != "" ? ITS4YouWarehouses_DNUtil_ActionBlock::getRelatedInvoice($deliveryNoteId) : array());

                    $toBeDeliveredQty = array();
                    //first handle related SOs
                    if (count($salesOrderIds) > 0) {
                        $toBeDeliveredQty = ITS4YouWarehouses_Base_ActionBlock::getSOToBeDeliveredQty(array_keys($salesOrderIds));
                    } elseif (count($invoiceIds) > 0) {
                        //if there is no related SOs then handle related INVs (process begins in INV) => see the commented code below, the logic is the same
                        $toBeDeliveredQty = ITS4YouWarehouses_Base_ActionBlock::getINVToBeDeliveredQty(array_keys($invoiceIds));
                    }

                    if (count($toBeDeliveredQty) > 0) {
                        $productIds = array_keys($toBeDeliveredQty);
                        $qtyInfoEntries = self::getQtyInfoEntries($productIds);
                        foreach ($qtyInfoEntries as $prodId => $qtyInfoEntry) {
                            /* @var $qtyInfoEntry OverallQtyInfoEntry */
                            if ($_beforeEvent) {
                                $newQty = $qtyInfoEntry->getToBeDeliveredQty() - $toBeDeliveredQty[$prodId];
                            } else {
                                $newQty = $qtyInfoEntry->getToBeDeliveredQty() + $toBeDeliveredQty[$prodId];
                            }
                            $qtyInfoEntry->setToBeDeliveredQty($newQty);
                        }
                    }
                } else {
                    $toBeDeliveredQty = array();
                    //first handle related SOs
                    if ($_entityData->get("referenceid") != "") {
                        $toBeDeliveredQty = ITS4YouWarehouses_Base_ActionBlock::getSOToBeDeliveredQty(array($_entityData->get("referenceid")));
                    } elseif ($_entityData->get("invoice_id") != "") {
                        //if there is no related SOs then handle related INVs (process begins in INV) => see the commented code below, the logic is the same
                        $toBeDeliveredQty = ITS4YouWarehouses_Base_ActionBlock::getINVToBeDeliveredQty(array($_entityData->get("invoice_id")));
                    }

                    if (count($toBeDeliveredQty) > 0) {
                        $productIds = array_keys($toBeDeliveredQty);
                        //we use temp array and only in case of Services we handle it and add to $qtyInfoEntries
                        $tempQtyInfoEntries = self::getQtyInfoEntries($productIds);
                        foreach ($tempQtyInfoEntries as $prodId => $tempQtyInfoEntry) {
                            /* @var $qtyInfoEntry OverallQtyInfoEntry */
                            if ($tempQtyInfoEntry->getSEType() == "Services") {
                                if ($_beforeEvent) {
                                    $newQty = $tempQtyInfoEntry->getToBeDeliveredQty() - $toBeDeliveredQty[$prodId];
                                } else {
                                    $newQty = $tempQtyInfoEntry->getToBeDeliveredQty() + $toBeDeliveredQty[$prodId];
                                }
                                $tempQtyInfoEntry->setToBeDeliveredQty($newQty);
                                $qtyInfoEntries[$tempQtyInfoEntry->getProductId()] = $tempQtyInfoEntry;
                            }
                        }
                    }
                }
                break;

            case "SalesOrder":
                if ($_beforeEvent && $_entityData->isNew()) {
                    return;
                }

                $toBeDeliveredQty = ITS4YouWarehouses_Base_ActionBlock::getSOToBeDeliveredQty(array($_entityData->getId()));
                $productIds = array_keys($toBeDeliveredQty);
                $qtyInfoEntries = self::getQtyInfoEntries($productIds);
                foreach ($qtyInfoEntries as $prodId => $qtyInfoEntry) {
                    /* @var $qtyInfoEntry OverallQtyInfoEntry */
                    if ($_beforeEvent) {
                        $newQty = $qtyInfoEntry->getToBeDeliveredQty() - $toBeDeliveredQty[$prodId];
                    } else {
                        $newQty = $qtyInfoEntry->getToBeDeliveredQty() + $toBeDeliveredQty[$prodId];
                    }
                    $qtyInfoEntry->setToBeDeliveredQty($newQty);
                }
                break;

            //branch for handling business process from Invoice as a starting point
            case "Invoice":
                if (($_beforeEvent && $_entityData->isNew()) || $_entityData->get("salesorderid") != "") {
                    return;
                }

                $toBeDeliveredQty = ITS4YouWarehouses_Base_ActionBlock::getINVToBeDeliveredQty(array($_entityData->getId()));
                $productIds = array_keys($toBeDeliveredQty);
                $qtyInfoEntries = self::getQtyInfoEntries($productIds);
                foreach ($qtyInfoEntries as $prodId => $qtyInfoEntry) {
                    /* @var $qtyInfoEntry OverallQtyInfoEntry */
                    if ($_beforeEvent) {
                        $newQty = $qtyInfoEntry->getToBeDeliveredQty() - $toBeDeliveredQty[$prodId];
                    } else {
                        $newQty = $qtyInfoEntry->getToBeDeliveredQty() + $toBeDeliveredQty[$prodId];
                    }
                    $qtyInfoEntry->setToBeDeliveredQty($newQty);
                }
                break;

            default:
                return;
        }

        self::saveQtyInfoEntries($qtyInfoEntries);
    }

    private static function saveQtyInfoEntries($_qtyInfoEntries)
    {
        global $adb;

        if (count($_qtyInfoEntries) <= 0) {
            return;
        }

        $productIds = array_keys($_qtyInfoEntries);
        $sql = "DELETE FROM " . OverallQtyInfoHandler::TABLE_NAME . " WHERE productid IN (" . generateQuestionMarks($productIds) . ")";
        $adb->pquery($sql, $productIds);

        $sql = "INSERT INTO " . OverallQtyInfoHandler::TABLE_NAME . " (productid, tobe_receipted, tobe_delivered) VALUES ";
        $params = array();
        foreach ($_qtyInfoEntries as $prodId => $qtyInfoEntry) {
            /* @var $qtyInfoEntry OverallQtyInfoEntry */
            $sql .= "(?,?,?),";
            $params = array_merge($params, array($prodId, $qtyInfoEntry->getToBeReceiptedQty(), $qtyInfoEntry->getToBeDeliveredQty()));
        }
        $sql = rtrim($sql, ',');
        $adb->pquery($sql, $params);
    }

    /**
     *
     * @param int|bool $_productId
     * @throws Exception
     * @global PearDatabase $adb
     */
    public static function recalculate($_productId = false)
    {
        $adb = PearDatabase::getInstance();

        if ($_productId == false) {
            $awaitingQty = ITS4YouWarehouses_Base_ActionBlock::getOverallAwaitingQty();
            $toBeDeliveredQty = ITS4YouWarehouses_Base_ActionBlock::getOverallToBeDeliveredQty();
        } else {
            $awaitingQty = ITS4YouWarehouses_Base_ActionBlock::getOverallAwaitingQty(array($_productId));
            $toBeDeliveredQty = ITS4YouWarehouses_Base_ActionBlock::getOverallToBeDeliveredQty(array($_productId));
        }

        $qtyInfoEntries = array();
        foreach ($awaitingQty as $prodId => $awaitQty) {
            $toBeDelQty = (isset($toBeDeliveredQty[$prodId]) ? $toBeDeliveredQty[$prodId] : 0);
            $qtyInfoEntry = new OverallQtyInfoEntry($prodId, $awaitQty, $toBeDelQty);
            $qtyInfoEntries[$prodId] = $qtyInfoEntry;
        }

        foreach ($toBeDeliveredQty as $prodId => $toBeDelQty) {
            if (!isset($qtyInfoEntries[$prodId])) {
                $qtyInfoEntry = new OverallQtyInfoEntry($prodId, 0, $toBeDelQty);
                $qtyInfoEntries[$prodId] = $qtyInfoEntry;
            }
        }
        //in case only specific productId was defined then we need to remove other entries          
        if ($_productId != false) {
            $tmpQtyInfoEntries = $qtyInfoEntries;
            $qtyInfoEntries = array();
            $qtyInfoEntries[$_productId] = $tmpQtyInfoEntries[$_productId];

            //save recalculated entries into DB  
            self::saveQtyInfoEntries($qtyInfoEntries);
        } elseif (count($qtyInfoEntries) > 0) {
            //if all products should be recalculated then we need to handle those products which
            //are not contained in SO, INV nor PO but are present in OverallQtyInfoHandler::TABLE_NAME table
            $sql = "SELECT productid FROM " . OverallQtyInfoHandler::TABLE_NAME . " WHERE productid NOT IN (" . generateQuestionMarks(array_keys($qtyInfoEntries)) . ")";
            $result = $adb->pquery($sql, array_keys($qtyInfoEntries));
            while ($row = $adb->fetchByAssoc($result)) {
                $qtyInfoEntry = new OverallQtyInfoEntry($row["productid"], 0, 0);
                $qtyInfoEntries[$row["productid"]] = $qtyInfoEntry;
            }

            //save recalculated entries into DB  
            self::saveQtyInfoEntries($qtyInfoEntries);
        }
    }

}

class OverallQtyInfoEntry
{

    private $productId;
    private $toBeReceiptedQty;
    private $toBeDeliveredQty;

    public function __construct($_productId, $_toBeReceiptedQty, $_toBeDeliveredQty)
    {
        $this->productId = $_productId;
        $this->toBeReceiptedQty = $_toBeReceiptedQty;
        $this->toBeDeliveredQty = $_toBeDeliveredQty;
    }

    public static function createDummyEntry($_productId)
    {
        return new OverallQtyInfoEntry($_productId, 0, 0);
    }

    public function getProductId()
    {
        return $this->productId;
    }

    public function getToBeDeliveredQty()
    {
        return $this->toBeDeliveredQty;
    }

    public function setToBeDeliveredQty($_value)
    {
        $this->toBeDeliveredQty = $_value;
    }

    public function getToBeReceiptedQty()
    {
        return $this->toBeReceiptedQty;
    }

    public function setToBeReceiptedQty($_value)
    {
        $this->toBeReceiptedQty = $_value;
    }

    /**
     * @return string
     * @throws Exception
     */
    public function getSEType()
    {
        $db = PearDatabase::getInstance();
        $sql = "SELECT setype FROM vtiger_crmentity WHERE crmid = ?";
        $result = $db->pquery($sql, array($this->productId));
        return $db->query_result($result, 0, "setype");
    }

}
