<?php
/* * *******************************************************************************
 * The content of this file is subject to the MultiWarehouses4You license.
 * ("License"); You may not use this file except in compliance with the License
 * The Initial Developer of the Original Code is IT-Solutions4You s.r.o.
 * Portions created by IT-Solutions4You s.r.o. are Copyright(C) IT-Solutions4You s.r.o.
 * All Rights Reserved.
 * ****************************************************************************** */

class CrossInventoryRelEventHandler extends VTEventHandler
{

    const TABLE_NAME = "its4you_wh_crossinventoryrel";

    private $allowedModules = array("Invoice", "ITS4YouReceiptcards", "ITS4YouIssuecards", "ITS4YouWarehouseTransfers", "ITS4YouWHDeliveryNotes");

    /**
     * Check whether relationship between line items of source module record and target module record exists.
     * If so, then it adjusts line items and returns it
     * @param $_sourceLineItems
     * @param array $_targetIDs
     * @param bool $cloneSourceLineItems
     * @return array Array of adjusted line items
     * @global PearDatabase $adb
     */
    public static function adjustLineItems($_sourceLineItems, $_targetIDs, $cloneSourceLineItems = false)
    {
        global $adb;

        $sourceLineItems = array();
        //clone source line items
        if (!$cloneSourceLineItems) {
            $sourceLineItems = $_sourceLineItems;
        } else {
            foreach ($_sourceLineItems as $lineItemId => $lineItem) {
                $sourceLineItems[$lineItemId] = clone $lineItem;
            }
        }

        //init $targetLineItems
        $targetLineItems = array();
        foreach ($_targetIDs as $targetId) {
            $tmpLineItems = ITS4YouWarehouses_InventoryLineItem_Helper::getParentLineItems($targetId);
            /* @var $lineItem ITS4YouWarehouses_InventoryLineItem_Helper */
            foreach ($tmpLineItems as $lineItemId => $lineItem) {
                $targetLineItems[$targetId][$lineItem->getProductId()][$lineItem->getSequenceNo()] = $lineItem->getQuantity();
            }
        }
        //init negative quantity remainder
        $overReceiptRemainder = array();
        foreach ($sourceLineItems as $lineItem) {
            $overReceiptRemainder[$lineItem->getProductId()] = 0;
        }

        $adjustedLineItems = array();
        /* @var $lineItem ITS4YouWarehouses_InventoryLineItem_Helper */
        foreach ($sourceLineItems as $lineItemId => $lineItem) {
            $sql = "SELECT productid, target_crmid, target_sequence_no
                    FROM " . CrossInventoryRelEventHandler::TABLE_NAME . " 
                    WHERE productid = ? AND source_crmid = ? AND source_sequence_no = ? AND target_crmid IN (" . generateQuestionMarks($_targetIDs) . ")";
            $params = array_merge(array($lineItem->getProductId(), $lineItem->getParentId(), $lineItem->getSequenceNo()), $_targetIDs);
            $result = $adb->pquery($sql, $params);

            $lineItemQty = $lineItem->getQuantity();
            if ($adb->num_rows($result) > 0) {
                //one source line item can be related to more targets
                while ($row = $adb->fetchByAssoc($result)) {
                    if (isset($targetLineItems[$row["target_crmid"]][$row["productid"]][$row["target_sequence_no"]])) {
                        $lineItemQty -= $targetLineItems[$row["target_crmid"]][$row["productid"]][$row["target_sequence_no"]];
                    }
                }
            }
            //subtract remainder negative quantity of previous line item
            $lineItemQty -= $overReceiptRemainder[$lineItem->getProductId()];
            if ($lineItemQty > 0) {
                $lineItem->setQuantity($lineItemQty);
                $adjustedLineItems[$lineItemId] = $lineItem;
                $overReceiptRemainder[$lineItem->getProductId()] = 0;
            } else {
                $overReceiptRemainder[$lineItem->getProductId()] += abs($lineItemQty);
            }
        }

        //we need to make sure that all negative remainder quantities were used for decreasing line items' quantity
        //even when same product with higher sequence_no (lower at inventory table) were over receipted
        foreach ($adjustedLineItems as $lineItemId => $lineItem) {
            if ($overReceiptRemainder[$lineItem->getProductId()] <= 0) {
                continue;
            }

            $lineItemQty = $lineItem->getQuantity() - $overReceiptRemainder[$lineItem->getProductId()];
            if ($lineItemQty > 0) {
                $lineItem->setQuantity($lineItemQty);
                $overReceiptRemainder[$lineItem->getProductId()] = 0;
            } else {
                unset($adjustedLineItems[$lineItemId]);
                $overReceiptRemainder[$lineItem->getProductId()] += abs($lineItemQty);
            }
        }

        return $adjustedLineItems;
    }

    /**
     * @param array $sourceLineItems
     * @param array $_targetIDs
     * @throws Exception
     */
    public static function addCrossInventoryRels($sourceLineItems, $_targetIDs)
    {
        $adb = PearDatabase::getInstance();

        if (empty($sourceLineItems) || empty($_targetIDs)) {
            return;
        }

        //init $targetLineItems
        $targetLineItems = array();
        foreach ($_targetIDs as $targetId) {
            $targetActionBlock = ITS4YouWarehouses_Base_ActionBlock::factory(getSalesEntityType($targetId), $targetId);
            $tmpLineItems = $targetActionBlock->getLineItems();
            /* @var $lineItem ITS4YouWarehouses_InventoryLineItem_Helper */
            foreach ($tmpLineItems as $lineItemId => $lineItem) {
                $targetLineItems[$targetId][$lineItem->getProductId()][$lineItem->getSequenceNo()] = $lineItem;
            }
        }

        foreach ($sourceLineItems as $lineItemId => $lineItem) {
            $sql = "SELECT productid, target_crmid, target_sequence_no
                    FROM " . CrossInventoryRelEventHandler::TABLE_NAME . " 
                    WHERE productid = ? AND source_crmid = ? AND source_sequence_no = ? AND target_crmid IN (" . generateQuestionMarks($_targetIDs) . ")";
            $params = array_merge(array($lineItem->getProductId(), $lineItem->getParentId(), $lineItem->getSequenceNo()), $_targetIDs);
            $result = $adb->pquery($sql, $params);

            if ($adb->num_rows($result) > 0) {
                //one source line item can be related to more targets
                while ($row = $adb->fetchByAssoc($result)) {
                    if (isset($targetLineItems[$row["target_crmid"]][$row["productid"]][$row["target_sequence_no"]])) {
                        $lineItem->addCrossInventoryLineItem($targetLineItems[$row["target_crmid"]][$row["productid"]][$row["target_sequence_no"]]);
                    }
                }
            }
        }
    }

    /**
     *
     * @param string $name
     * @param VTEntityData $entityData
     */
    public function handleEvent($name, $entityData)
    {
        $moduleName = $entityData->getModuleName();
        if (!in_array($moduleName, $this->allowedModules)) {
            return;
        }

        if (isset($_REQUEST["totalProductCount"]) && $entityData->isNew()) {
            $lineCount = vtlib_purify($_REQUEST["totalProductCount"]);
            for ($sequence = 1; $sequence <= $lineCount; $sequence++) {
                if (isset($_REQUEST["lineItemId" . $sequence]) && $_REQUEST["lineItemId" . $sequence] != "" && $_REQUEST["lineItemId" . $sequence] != "0") {
                    $lineItemId = vtlib_purify($_REQUEST["lineItemId" . $sequence]);
                    $sourceLineItem = ITS4YouWarehouses_InventoryLineItem_Helper::getLineItemFromDB($lineItemId);
                    if ($sourceLineItem !== null) {
                        CrossInventoryRelEventHandler::saveNewRel($sourceLineItem, $entityData->getId(), $sequence);
                    }
                }
            }
        }
    }

    /**
     * Store new relationship between line item of source module record and target module record
     * @param ITS4YouWarehouses_InventoryLineItem_Helper $_sourceLineItem Line item of source module record
     * @param int $_targetId ID of target module record
     * @param int $_targetSequence
     */
    private static function saveNewRel($_sourceLineItem, $_targetId, $_targetSequence)
    {
        global $adb;

        $params = array($_sourceLineItem->getProductId(), $_sourceLineItem->getParentId(), $_sourceLineItem->getSequenceNo(), $_targetId, $_targetSequence);
        $sql = "INSERT INTO " . CrossInventoryRelEventHandler::TABLE_NAME . " (productid, source_crmid, source_sequence_no, target_crmid, target_sequence_no) VALUES(" . generateQuestionMarks($params) . ")";
        $adb->pquery($sql, $params);
    }

}
