<?php
/* * *******************************************************************************
 * The content of this file is subject to the MultiWarehouses4You license.
 * ("License"); You may not use this file except in compliance with the License
 * The Initial Developer of the Original Code is IT-Solutions4You s.r.o.
 * Portions created by IT-Solutions4You s.r.o. are Copyright(C) IT-Solutions4You s.r.o.
 * All Rights Reserved.
 * ****************************************************************************** */

class ITS4YouWarehouses_Item_ActionBlock
{

    const SERVICE_ITEM_AVAIL_QTY = "";

    /** @var int */
    protected $id;

    /** @var string */
    protected $name;

    /** @var string */
    protected $link;

    /** @var int */
    protected $parentId;

    /** @var double */
    protected $parentQty;

    /** @var double */
    protected $availableQty;

    /** @var double */
    protected $relatedInboundItemQty = 0;

    /** @var ITS4YouWarehouses_ItemInfo_ActionBlock */
    protected $itemInfo;

    /** @var double */
    protected $restQty;

    /** @var array */
    protected $statuses;
    private $setype;

    public function __construct($_id, $_name, $_parentId, $_parentQty, $_availableQty, $_itemInfo, $_setype = "Products")
    {
        $this->id = $_id;
        $this->name = $_name;
        $this->link = "index.php?module=" . $_setype . "&view=Detail&record=" . $_id;
        $this->parentId = $_parentId;
        $this->parentQty = $_parentQty;
        $this->availableQty = $_availableQty;
        $this->itemInfo = $_itemInfo;

        $this->restQty = $_parentQty;
        $this->statuses = array();
        $this->setype = $_setype;
        $this->modifyImplicitStatuses();
    }

    protected function modifyImplicitStatuses()
    {
        if ($this->restQty > 0) {    // if there is still some to-be-delivered quantity left
            $missingQty = OverallQtyInfoHandler::getQtyInfoEntry($this->id)->getToBeDeliveredQty() - $this->availableQty;
            if ($missingQty > 0 && $this->setype == "Products") { // if there is NOT enough qty at stock for overall to be delivered SO; only for Products

                if ($this->availableQty > 0) {
                    $this->statuses[ITS4YouWarehouses_ItemStatus_ActionBlock::STATUS_AVAILABLE] = new ITS4YouWarehouses_ItemStatusAvailable_ActionBlock($this->availableQty);
                }

                $overallToBeReceiptedQty = OverallQtyInfoHandler::getQtyInfoEntry($this->id)->getToBeReceiptedQty();
                if ($overallToBeReceiptedQty - $missingQty < $this->itemInfo->getReorderLevel() &&
                    $overallToBeReceiptedQty - $missingQty < $this->itemInfo->getOptimalLevel()) {
                    $optimalDiff = $this->itemInfo->getOptimalLevel() - $overallToBeReceiptedQty + $missingQty;
                    $this->statuses[ITS4YouWarehouses_ItemStatus_ActionBlock::STATUS_NOTAVAILABLE] = new ITS4YouWarehouses_ItemStatusNA_ActionBlock($optimalDiff);
                } else {
                    try {
                        unset($this->statuses[ITS4YouWarehouses_ItemStatus_ActionBlock::STATUS_NOTAVAILABLE]);
                        $this->getDisplayStatus();
                    } catch (Exception $ex) {
                        //in exceptional cases we want to keep STATUS_NOT_AVAILABLE set
                        //i.e. we have two SOs having the same product SO1 = 10 Qty and SO2 = 15 Qty and Qty in stock = 0
                        //  => when we are creating PO from SO1 we will order 25 Qty but until this 25 Qty is not receipted
                        //     SO2 stays in this exceptional state => therefore we need to handle such state in this way

                        $this->statuses[ITS4YouWarehouses_ItemStatus_ActionBlock::STATUS_NOTAVAILABLE] = new ITS4YouWarehouses_ItemStatusNA_ActionBlock($this->getRestQty());
                    }
                }
            } else {    // if there is enough qty at stock
                $this->statuses[ITS4YouWarehouses_ItemStatus_ActionBlock::STATUS_AVAILABLE] = new ITS4YouWarehouses_ItemStatusAvailable_ActionBlock($this->restQty);
            }
        } else {    // if all to-be-delivered quantity was handled then remove possible statuses
            unset($this->statuses[ITS4YouWarehouses_ItemStatus_ActionBlock::STATUS_AVAILABLE]);
            unset($this->statuses[ITS4YouWarehouses_ItemStatus_ActionBlock::STATUS_NOTAVAILABLE]);
            unset($this->statuses[ITS4YouWarehouses_ItemStatus_ActionBlock::STATUS_AVAILABLE_PART]);
        }
    }

    /**
     *
     * @return ITS4YouWarehouses_ItemStatus_ActionBlock
     * @throws Exception
     */
    public function getDisplayStatus()
    {
        $prioStatus = null;
        $minPrio = PHP_INT_MAX;
        foreach ($this->statuses as $tmpStatus) {
            if ($tmpStatus->getDisplayPrio() < 0) {
                continue;
            }

            if ($tmpStatus->getDisplayPrio() < $minPrio) {
                $minPrio = $tmpStatus->getDisplayPrio();
                $prioStatus = $tmpStatus;
            }
        }

        if ($prioStatus == null) {
            throw new Exception("ITS4YouWHDeliveryNotes :: Unexpected situation. Item has no relevant display status.");
        }

        return $prioStatus;
    }

    public function getRestQty()
    {
        return $this->restQty;
    }

    public function getId()
    {
        return $this->id;
    }

    public function getAvailableQty()
    {
        if ($this->setype == "Products") {
            return $this->availableQty;
        } else {
            //if item is Service, then return corresponding status qty, which is set in method modifyImplicitStatuses
            return $this->getStatusQty(ITS4YouWarehouses_ItemStatus_ActionBlock::STATUS_AVAILABLE);
        }
    }

    /**
     * Returns quantity of item in specific status
     * @param string $_statusId ID of status
     * @return int If item is in specific status then number of units in the status is returned, else 0
     */
    public function getStatusQty($_statusId)
    {
        if (isset($this->statuses[$_statusId])) {
            return $this->statuses[$_statusId]->getItemQty();
        } else {
            return 0;
        }
    }

    public function getParentId()
    {
        return $this->parentId;
    }

    public function getParentQty()
    {
        return $this->parentQty;
    }

    public function getLink()
    {
        return $this->link;
    }

    public function getName()
    {
        return $this->name;
    }

    public function getStatuses()
    {
        return $this->statuses;
    }

    public function getRelatedInboundItemQty()
    {
        return $this->relatedInboundItemQty;
    }

    public function getShortName()
    {
        return strlen($this->name) > 20 ? substr($this->name, 0, 17) . "..." : $this->name;
    }

    public function getSEType()
    {
        return $this->setype;
    }

    /**
     *
     * @return ITS4YouWarehouses_ItemInfo_ActionBlock
     * @throws Exception
     * @global PearDatabase $adb
     */
    public function getItemInfo()
    {
        if (!$this->itemInfo) {
            $productInfo = $this->getProductInfo();

            if (!empty($productInfo)) {
                $this->itemInfo = ITS4YouWarehouses_ItemInfo_ActionBlock::initialize($productInfo);
            } else {
                throw new Exception("MultiWarehouses4You :: Product with ID '" . $this->id . "' not found.");
            }
        }

        return $this->itemInfo;
    }

    /**
     * @return array
     */
    public function getProductInfo()
    {
        $adb = PearDatabase::getInstance();
        $result = $adb->pquery('SELECT * FROM vtiger_products WHERE productid = ?', array($this->id));

        return (array)$adb->fetchByAssoc($result);
    }

    /**
     * Set instance of ITS4YouWarehouses_ItemInfo_ActionBlock class which contains all additional details about action block item
     * @param ITS4YouWarehouses_ItemInfo_ActionBlock $_itemInfo
     */
    public function setItemInfo($_itemInfo)
    {
        $this->itemInfo = $_itemInfo;
    }

    /**
     * Merge - add quantities of explicit statuses with current instance quantities
     * @param ITS4YouWarehouses_Item_ActionBlock $_sourceActionBlockItem Source action block item whose explicit status quantities are merged
     * @throws Exception
     */
    public function mergeItems($_sourceActionBlockItem)
    {
        /* @var $statusInstance ITS4YouWarehouses_ItemStatus_ActionBlock */

        if (get_class($_sourceActionBlockItem) != 'ITS4YouWarehouses_ItemInbound_ActionBlock') {
            $statusInstances = $_sourceActionBlockItem->statuses;
            foreach ($statusInstances as $statusInstance) {
                if (!$statusInstance->getIsExplicit()) {
                    continue;
                }

                $qty = $statusInstance->getItemQty() + $this->getStatusQty($statusInstance->getId());
                $this->modifyExplicitStatus($statusInstance->getId(), $qty);
            }
        } else {
            $this->relatedInboundItemQty += $_sourceActionBlockItem->parentQty;
            //handle other explicit statuses of $_sourceActionBlockItem (i.e. RECEIPTED) - type ITS4YouWarehouses_ItemInbound_ActionBlock
            //but without adjusting restQty of $this - ITS4YouWarehouses_Item_ActionBlock
            $statusInstances = $_sourceActionBlockItem->statuses;
            foreach ($statusInstances as $statusInstance) {
                if (!$statusInstance->getIsExplicit()) {
                    continue;
                }

                $qty = $statusInstance->getItemQty() + $this->getStatusQty($statusInstance->getId());
                $this->setExplicitStatusQty($statusInstance->getId(), $qty);
            }
            $this->modifyImplicitStatuses();
        }
    }

    /**
     * Modify (create or update) explicit status and its quantity
     * @param string $_statusId ID of status
     * @param int $_qty Quantity of item in specific status
     * @throws Exception
     */
    public function modifyExplicitStatus($_statusId, $_qty)
    {
        $prevQty = $this->getStatusQty($_statusId);
        /* @var $statusInstance ITS4YouWarehouses_ItemStatus_ActionBlock */
        $statusInstance = $this->setExplicitStatusQty($_statusId, $_qty);
        if ($statusInstance->getIsAdjustingQty()) {
            $this->restQty += $prevQty;
            $this->restQty -= $_qty;
            $this->modifyImplicitStatuses();
        }
    }

    /**
     * @param $_statusId
     * @param $_qty
     * @return ITS4YouWarehouses_ItemStatus_ActionBlock|ITS4YouWarehouses_ItemStatusAvailableExisting_ActionBlock|ITS4YouWarehouses_ItemStatusAwaiting_ActionBlock|ITS4YouWarehouses_ItemStatusCanceled_ActionBlock|ITS4YouWarehouses_ItemStatusDelivered_ActionBlock|ITS4YouWarehouses_ItemStatusInvoiced_ActionBlock|ITS4YouWarehouses_ItemStatusIssued_ActionBlock|ITS4YouWarehouses_ItemStatusReceipted_ActionBlock|ITS4YouWarehouses_ItemStatusReturned_ActionBlock|ITS4YouWarehouses_ItemStatusToBeInvoiced_ActionBlock
     * @throws Exception
     */
    protected function setExplicitStatusQty($_statusId, $_qty)
    {
        /* @var $statusInstance ITS4YouWarehouses_ItemStatus_ActionBlock */
        $statusInstance = null;
        if (isset($this->statuses[$_statusId])) {
            $statusInstance = $this->statuses[$_statusId];
        } else {
            switch ($_statusId) {
                case ITS4YouWarehouses_ItemStatus_ActionBlock::STATUS_DELIVERED:
                    $statusInstance = new ITS4YouWarehouses_ItemStatusDelivered_ActionBlock(0);
                    break;

                case ITS4YouWarehouses_ItemStatus_ActionBlock::STATUS_INVOICED:
                    $statusInstance = new ITS4YouWarehouses_ItemStatusInvoiced_ActionBlock(0);
                    break;

                case ITS4YouWarehouses_ItemStatus_ActionBlock::STATUS_ISSUED:
                    $statusInstance = new ITS4YouWarehouses_ItemStatusIssued_ActionBlock(0);
                    break;

                case ITS4YouWarehouses_ItemStatus_ActionBlock::STATUS_TOBEINVOICED:
                    $statusInstance = new ITS4YouWarehouses_ItemStatusToBeInvoiced_ActionBlock(0);
                    break;

                case ITS4YouWarehouses_ItemStatus_ActionBlock::STATUS_AVAILABLE_EXIST:
                    $statusInstance = new ITS4YouWarehouses_ItemStatusAvailableExisting_ActionBlock(0);
                    break;

                case ITS4YouWarehouses_ItemStatus_ActionBlock::STATUS_RECEIPTED:
                    $statusInstance = new ITS4YouWarehouses_ItemStatusReceipted_ActionBlock(0);
                    break;

                case ITS4YouWarehouses_ItemStatus_ActionBlock::STATUS_AWAITING:
                    $statusInstance = new ITS4YouWarehouses_ItemStatusAwaiting_ActionBlock(0);
                    break;

                case ITS4YouWarehouses_ItemStatus_ActionBlock::STATUS_CANCELED:
                    $statusInstance = new ITS4YouWarehouses_ItemStatusCanceled_ActionBlock(0);
                    break;

                case ITS4YouWarehouses_ItemStatus_ActionBlock::STATUS_RETURNED:
                    $statusInstance = new ITS4YouWarehouses_ItemStatusReturned_ActionBlock(0);
                    break;
            }
        }

        if ($statusInstance != null && $statusInstance->getIsExplicit() === true) {
            $statusInstance->setItemQty($_qty);
            if ($_qty <= 0) {
                unset($this->statuses[$_statusId]);
            } else {
                $this->statuses[$_statusId] = $statusInstance;
            }
        } else {
            throw new Exception("ITS4YouWHDeliveryNotes :: Provided ActionBlockItemStatus is not explicit.");
        }

        return $statusInstance;
    }

}