<?php

/* * *******************************************************************************
 * The content of this file is subject to the MultiWarehouses4You license.
 * ("License"); You may not use this file except in compliance with the License
 * The Initial Developer of the Original Code is IT-Solutions4You s.r.o.
 * Portions created by IT-Solutions4You s.r.o. are Copyright(C) IT-Solutions4You s.r.o.
 * All Rights Reserved.
 * ****************************************************************************** */

require_once __DIR__ . '/../handlers/OverallQtyInfo.php';
require_once __DIR__ . '/../handlers/CrossInventoryRel.php';

/**
 * Abstract class for all action blocks
 */
abstract class ITS4YouWarehouses_Base_ActionBlock
{

    /** @var int Record ID of action block parent */
    protected $parentId;
    /** @var ITS4YouWarehouses_StatusActionsPoolBase_ActionBlock Pool of all statuses actions */
    protected $statusActionsPool;
    /** @var array Array of ITS4YouWarehouses_InventoryLineItem_Helper */
    protected $lineItems = array();
    /** @var array Array of all items in action block */
    protected $items;

    /**
     * ITS4YouWarehouses_Base_ActionBlock constructor.
     * @param $_parentId
     */
    protected function __construct($_parentId)
    {
        $this->parentId = $_parentId;
        $this->initStatusActionsPool();
        $this->createParentItems();
    }

    /**
     *
     */
    protected function initStatusActionsPool()
    {
        $this->statusActionsPool = ITS4YouWarehouses_StatusActionsPoolBase_ActionBlock::getInstance();
    }

    /**
     *
     */
    protected function createParentItems()
    {
        global $adb;

        $sql = "SELECT  CASE WHEN vtiger_products.productid != '' THEN vtiger_products.productname ELSE vtiger_service.servicename END AS 'productname',
                        CASE WHEN vtiger_products.productid != '' THEN vtiger_products.qtyinstock ELSE '0' END AS 'available_qty',            
                        " . ITS4YouWarehouses_ItemInfo_ActionBlock::getItemInfoQuery() . ",
                        parent_rel.productid,
                        SUM(parent_rel.quantity) AS parent_qty,
                        vtiger_crmentity.setype
                FROM vtiger_inventoryproductrel AS parent_rel
                INNER JOIN vtiger_crmentity
                    ON vtiger_crmentity.crmid=parent_rel.productid
                LEFT JOIN vtiger_products
                    ON vtiger_products.productid=parent_rel.productid                                
                LEFT JOIN vtiger_service
                    ON vtiger_service.serviceid=parent_rel.productid                                
                WHERE vtiger_crmentity.deleted = 0 
                    AND parent_rel.id=?                 
                GROUP BY parent_rel.productid 
                ORDER BY parent_rel.sequence_no";
        $result = $adb->pquery($sql, array($this->parentId));

        if ($adb->num_rows($result) > 0) {
            while ($row = $adb->fetchByAssoc($result)) {
                $parentItem = new ITS4YouWarehouses_Item_ActionBlock($row["productid"], $row["productname"], $this->parentId, $row["parent_qty"], $row["available_qty"], ITS4YouWarehouses_ItemInfo_ActionBlock::initialize($row), $row["setype"]);
                $this->addItem($parentItem);
            }
        }
    }

    /**
     * @param ITS4YouWarehouses_Item_ActionBlock $item
     * @return bool
     */
    final protected function addItem(ITS4YouWarehouses_Item_ActionBlock $item)
    {
        $servicesEnabledPreference = ITS4YouWHDeliveryNotes_Preferences_Model::getInstance(ITS4YouWHDeliveryNotes_Preferences_Model::PREFNAME_SERVICES_ENABLED);
        if ($servicesEnabledPreference->getValue() == "0") {
            if ($item->getSEType() === "Products") {
                $this->items[$item->getId()] = $item;
                return true;
            }
        } else {
            $this->items[$item->getId()] = $item;
            return true;
        }
        return false;
    }

    /**
     * @param array $_itemIds
     * @param array $_purchaseOrderIds
     * @return array
     * @throws Exception
     */
    public static function getOverallAwaitingQty($_itemIds = array(), $_purchaseOrderIds = array())
    {
        global $adb;

        if (empty($_purchaseOrderIds)) {
            if (empty($_itemIds)) {
                $sql = "SELECT purchaseorderid 
                        FROM vtiger_purchaseorder
                        INNER JOIN vtiger_crmentity
                                ON vtiger_crmentity.crmid = vtiger_purchaseorder.purchaseorderid
                        WHERE vtiger_crmentity.deleted = 0
                          AND vtiger_purchaseorder.postatus != 'Cancelled'";
                $params = array();
            } else {
                $sql = "SELECT DISTINCT purchaseorderid 
                        FROM vtiger_purchaseorder
                        INNER JOIN vtiger_crmentity
                                ON vtiger_crmentity.crmid = vtiger_purchaseorder.purchaseorderid
                        INNER JOIN vtiger_inventoryproductrel
                                ON vtiger_inventoryproductrel.id = vtiger_purchaseorder.purchaseorderid
                        WHERE vtiger_crmentity.deleted = 0
                          AND vtiger_purchaseorder.postatus != 'Cancelled'         
                          AND vtiger_inventoryproductrel.productid IN (" . generateQuestionMarks($_itemIds) . ")";
                $params = $_itemIds;
            }

            $result = $adb->pquery($sql, $params);
            $_purchaseOrderIds = array();
            while ($row = $adb->fetchByAssoc($result)) {
                $_purchaseOrderIds[] = $row["purchaseorderid"];
            }
        }

        $awaitingQty = array();
        foreach ($_purchaseOrderIds as $purchaseOrderId) {
            $actionBlock = self::factory("PurchaseOrder", $purchaseOrderId);
            if ($actionBlock->isCanceled()) {
                continue;
            }
            /* @var $actionBlock ITS4YouWarehouses_PO_ActionBlock */
            foreach ($actionBlock->items as $itemId => $item) {
                /* @var $item ITS4YouWarehouses_Item_ActionBlock */
                $productId = $item->getId();
                if (isset($awaitingQty[$productId])) {
                    $awaitingQty[$productId] += $item->getStatusQty(ITS4YouWarehouses_ItemStatus_ActionBlock::STATUS_AWAITING);
                } else {
                    $awaitingQty[$productId] = $item->getStatusQty(ITS4YouWarehouses_ItemStatus_ActionBlock::STATUS_AWAITING);
                }
            }
        }

        foreach ($_itemIds as $itemId) {
            if (!isset($awaitingQty[$itemId])) {
                $awaitingQty[$itemId] = 0;
            }
        }

        return $awaitingQty;
    }

    /**
     * Simple factory method for single-place of action block instance creation
     * @param string $_parentModule
     * @param number $_parentId
     * @return ITS4YouWarehouses_WT_ActionBlock|ITS4YouWarehouses_IC_ActionBlock|ITS4YouWarehouses_RC_ActionBlock|ITS4YouWarehouses_SO_ActionBlock|ITS4YouWarehouses_DN_ActionBlock|ITS4YouWarehouses_PO_ActionBlock|ITS4YouWarehouses_INV_ActionBlock|ITS4YouWarehouses_IP_ActionBlock
     * @throws Exception
     */
    public static function factory($_parentModule, $_parentId)
    {
        switch ($_parentModule) {
            case 'SalesOrder':
                return new ITS4YouWarehouses_SO_ActionBlock($_parentId);

            case 'ITS4YouWHDeliveryNotes':
                return new ITS4YouWarehouses_DN_ActionBlock($_parentId);

            case 'Invoice':
                return new ITS4YouWarehouses_INV_ActionBlock($_parentId);

            case 'PurchaseOrder':
                return new ITS4YouWarehouses_PO_ActionBlock($_parentId);

            case 'ITS4YouIncomingProducts':
                return new ITS4YouWarehouses_IP_ActionBlock();

            case 'ITS4YouReceiptcards':
                return new ITS4YouWarehouses_RC_ActionBlock($_parentId);

            case 'ITS4YouIssuecards':
                return new ITS4YouWarehouses_IC_ActionBlock($_parentId);

            case 'ITS4YouWarehouseTransfers':
                return new ITS4YouWarehouses_WT_ActionBlock($_parentId);

            default:
                throw new Exception('Warehouses :: Unsupported action block module ActionBlockBase.');
        }
    }

    /**
     * @param array $_itemIds
     * @return array
     * @throws Exception
     */
    public static function getOverallToBeDeliveredQty($_itemIds = array())
    {
        global $adb;

        //SalesOrders
        if (empty($_itemIds)) {
            $sql = "SELECT salesorderid 
                    FROM vtiger_salesorder
                    INNER JOIN vtiger_crmentity
                            ON vtiger_crmentity.crmid = vtiger_salesorder.salesorderid
                    WHERE vtiger_crmentity.deleted = 0
                        AND vtiger_salesorder.sostatus != 'Cancelled'";
            $params = array();
        } else {
            $sql = "SELECT salesorderid 
                    FROM vtiger_salesorder
                    INNER JOIN vtiger_crmentity
                            ON vtiger_crmentity.crmid = vtiger_salesorder.salesorderid
                    INNER JOIN vtiger_inventoryproductrel
                            ON vtiger_inventoryproductrel.id = vtiger_salesorder.salesorderid
                    WHERE vtiger_crmentity.deleted = 0
                        AND vtiger_salesorder.sostatus != 'Cancelled'
                        AND vtiger_inventoryproductrel.productid IN (" . generateQuestionMarks($_itemIds) . ")";
            $params = $_itemIds;
        }
        $result = $adb->pquery($sql, $params);

        $salesOrderIds = array();
        while ($row = $adb->fetchByAssoc($result)) {
            $salesOrderIds[] = $row["salesorderid"];
        }
        $SOToBeDeliveredQty = self::getSOToBeDeliveredQty($salesOrderIds);

        //Invoices
        if (empty($_itemIds)) {
            $sql = "SELECT invoiceid 
                    FROM vtiger_invoice
                    INNER JOIN vtiger_crmentity
                            ON vtiger_crmentity.crmid = vtiger_invoice.invoiceid
                    WHERE vtiger_crmentity.deleted = 0
                      AND vtiger_invoice.invoicestatus != 'Cancel'
                      AND vtiger_invoice.salesorderid IS NULL";
            $params = array();
        } else {
            $sql = "SELECT invoiceid 
                    FROM vtiger_invoice
                    INNER JOIN vtiger_crmentity
                            ON vtiger_crmentity.crmid = vtiger_invoice.invoiceid
                    INNER JOIN vtiger_inventoryproductrel
                            ON vtiger_inventoryproductrel.id = vtiger_invoice.invoiceid
                    WHERE vtiger_crmentity.deleted = 0
                      AND vtiger_invoice.salesorderid IS NULL
                      AND vtiger_invoice.invoicestatus != 'Cancel'
                      AND vtiger_inventoryproductrel.productid IN (" . generateQuestionMarks($_itemIds) . ")";
            $params = $_itemIds;
        }
        $result = $adb->pquery($sql, $params);

        $invoiceIds = array();
        while ($row = $adb->fetchByAssoc($result)) {
            $invoiceIds[] = $row["invoiceid"];
        }
        $INVToBeDeliveredQty = self::getINVToBeDeliveredQty($invoiceIds);

        $overallToBeDeliveredQty = array();
        foreach ($SOToBeDeliveredQty as $productId => $restQty) {
            if (isset($overallToBeDeliveredQty[$productId])) {
                $overallToBeDeliveredQty[$productId] += $restQty;
            } else {
                $overallToBeDeliveredQty[$productId] = $restQty;
            }
        }
        foreach ($INVToBeDeliveredQty as $productId => $restQty) {
            if (isset($overallToBeDeliveredQty[$productId])) {
                $overallToBeDeliveredQty[$productId] += $restQty;
            } else {
                $overallToBeDeliveredQty[$productId] = $restQty;
            }
        }

        return $overallToBeDeliveredQty;
    }

    /**
     * Returns to be delivered qty for an array of Sales Order IDs
     * @param array $_salesOrderIds
     * @return array
     * @throws Exception
     */
    public static function getSOToBeDeliveredQty($_salesOrderIds)
    {
        return self::getGenericToBeDeliveredQty($_salesOrderIds, "SalesOrder");
    }

    /**
     * @param $_recordIds
     * @param $_moduleName
     * @return array
     * @throws Exception
     */
    private static function getGenericToBeDeliveredQty($_recordIds, $_moduleName)
    {
        $toBeDeliveredQty = array();
        foreach ($_recordIds as $recordId) {
            $actionBlock = ITS4YouWarehouses_DNRelated_ActionBlock::factory($_moduleName, $recordId);
            if ($actionBlock->isCanceled()) {
                continue;
            }
            /* @var $actionBlock ITS4YouWarehouses_Base_ActionBlock */
            foreach ($actionBlock->items as $itemId => $item) {
                /* @var $item ITS4YouWarehouses_Item_ActionBlock */
                $productId = $item->getId();
                if (isset($toBeDeliveredQty[$productId])) {
                    $toBeDeliveredQty[$productId] += $item->getRestQty();
                } else {
                    $toBeDeliveredQty[$productId] = $item->getRestQty();
                }
            }
        }

        return $toBeDeliveredQty;
    }

    /**
     * Returns to be delivered qty for an array of Invoice IDs
     * @param array $_invoiceIds
     * @return array
     * @throws Exception
     */
    public static function getINVToBeDeliveredQty($_invoiceIds)
    {
        return self::getGenericToBeDeliveredQty($_invoiceIds, "Invoice");
    }

    /**
     * @param $number
     * @return string
     */
    public static function formatNumberForEditView($number)
    {
        $recordModel = Users_Record_Model::getCurrentUserModel();
        $no_of_currency_decimals = $recordModel->get("no_of_currency_decimals");

        return number_format($number, $no_of_currency_decimals, ".", "");
    }

    /**
     * @return array
     * @throws Exception
     */
    public function getSideWidgetViewModel()
    {
        $viewModel = $this->getMainViewModel();

        $newItems = array();
        foreach ($this->items as $item) {
            /* @var $item ITS4YouWarehouses_Item_ActionBlock */
            $itemId = $item->getId();
            $statuses = $item->getStatuses();
            foreach ($statuses as $status) {
                /* @var $status ITS4YouWarehouses_ItemStatus_ActionBlock */
                if ($status->getDisplayPrio() < 0) {
                    continue;
                }
                $statusId = $status->getId();
                $actionInstance = $this->statusActionsPool->getStatusAction($statusId);
                if ($actionInstance == null) {
                    continue;
                }

                $newItems[$statusId]["items"][$itemId] = $itemId;
                if (isset($newItems[$statusId]["count"])) {
                    $newItems[$statusId]["count"]++;
                } else {
                    $newItems[$statusId]["count"] = 1;
                    $newItems[$statusId]["status_name"] = $status->getName();
                    $newItems[$statusId]["status_action"] = $actionInstance->getSingleAction();
                }
            }
        }

        if (isset($newItems[ITS4YouWarehouses_ItemStatus_ActionBlock::STATUS_AVAILABLE]) && isset($newItems[ITS4YouWarehouses_ItemStatus_ActionBlock::STATUS_AVAILABLE_EXIST])) {
            unset($newItems[ITS4YouWarehouses_ItemStatus_ActionBlock::STATUS_AVAILABLE_EXIST]);
        }

        $viewModel["items"] = $this->sortSideWidgetStatuses($newItems);

        //add lineItems to viewModel
        $viewModel["lineItems"] = $this->getLineItems();

        return $viewModel;
    }

    /**
     * @return array
     * @throws Exception
     */
    public function getMainViewModel()
    {
        /* @var $item ITS4YouWarehouses_Item_ActionBlock */
        /* @var $displayStatus ITS4YouWarehouses_ItemStatus_ActionBlock */
        /* @var $statusInstance ITS4YouWarehouses_ItemStatus_ActionBlock */
        /* @var $actionInstance ITS4YouWarehouses_ItemStatusActionViewModel_Helper */
        /* @var $buttonViewModel ITS4YouWarehouses_ButtonActionViewModel_Helper */
        $items = array();
        foreach ($this->items as $item) {
            $itemId = $item->getId();
            $displayStatus = $item->getDisplayStatus();
            $actionInstance = $this->statusActionsPool->getStatusAction($displayStatus->getId());

            $items[$itemId]["status_id"] = $displayStatus->getId();
            $items[$itemId]["link"] = $item->getLink();
            $items[$itemId]["name"] = $item->getName();
            $items[$itemId]["name_short"] = $item->getShortName();
            $items[$itemId]["ordered_qty"] = self::formatNumberForDetailView($item->getParentQty());
            //$items[$itemId]["available_qty"] = self::formatNumberForDetailView($item->getAvailableQty());
            $items[$itemId]["available_qty"] = ($item->getSEType() == "Products" ? self::formatNumberForDetailView($item->getAvailableQty()) : ITS4YouWarehouses_Item_ActionBlock::SERVICE_ITEM_AVAIL_QTY);
            $items[$itemId]["issued_qty"] = self::formatNumberForDetailView($item->getStatusQty(ITS4YouWarehouses_ItemStatus_ActionBlock::STATUS_ISSUED));
            $items[$itemId]["delivered_qty"] = self::formatNumberForDetailView($item->getStatusQty(ITS4YouWarehouses_ItemStatus_ActionBlock::STATUS_DELIVERED));
            $items[$itemId]["invoiced_qty"] = self::formatNumberForDetailView($item->getStatusQty(ITS4YouWarehouses_ItemStatus_ActionBlock::STATUS_INVOICED));
            $items[$itemId]["status_action"] = ($actionInstance != null ? $actionInstance->getSingleAction() : "");
            $items[$itemId]["status_name"] = $displayStatus->getName();

            $toBeInvoicedQty = $item->getStatusQty(ITS4YouWarehouses_ItemStatus_ActionBlock::STATUS_DELIVERED) - $item->getStatusQty(ITS4YouWarehouses_ItemStatus_ActionBlock::STATUS_INVOICED) - $item->getStatusQty(ITS4YouWarehouses_ItemStatus_ActionBlock::STATUS_RETURNED);
            $toBeInvoicedQty = $toBeInvoicedQty < 0 ? 0 : $toBeInvoicedQty;
            $items[$itemId]["toBeInvoiced_qty"] = self::formatNumberForDetailView($toBeInvoicedQty);

            $items[$itemId]["canceled_qty"] = self::formatNumberForDetailView($item->getStatusQty(ITS4YouWarehouses_ItemStatus_ActionBlock::STATUS_CANCELED));

            $items[$itemId]["purchased_qty"] = self::formatNumberForDetailView($item->getRelatedInboundItemQty());
            $items[$itemId]["receipted_qty"] = self::formatNumberForDetailView($item->getStatusQty(ITS4YouWarehouses_ItemStatus_ActionBlock::STATUS_RECEIPTED));

            $items[$itemId]["returned_qty"] = self::formatNumberForDetailView($item->getStatusQty(ITS4YouWarehouses_ItemStatus_ActionBlock::STATUS_RETURNED));
        }
        //add action buttons
        $actions = array();
        $statuses = $this->getAllItemsAllStatusesDistinct();
        foreach ($statuses as $statusId => $statusInstance) {
            $actionInstance = $this->statusActionsPool->getStatusAction($statusId);
            if ($actionInstance != null && $actionInstance->isAnyButtonDefined()) {
                $idx = 0;
                $buttonViewModel = $actionInstance->getButton($idx);
                while ($buttonViewModel != null) {
                    $actions[$statusId][$idx]["style"] = $buttonViewModel->getButtonStyle();
                    $actions[$statusId][$idx]["action"] = $buttonViewModel->getButtonAction();
                    $actions[$statusId][$idx]["label"] = $buttonViewModel->getButtonLabel();
                    $actions[$statusId][$idx]["title"] = $buttonViewModel->getButtonTitle();
                    $buttonViewModel = $actionInstance->getButton(++$idx);
                }
            }
        }

        //Project MultiWarehouses4You specialities
        //STATUS_AVAILABLE and STATUS_AVAILABLE_PART share same viewmodel, in case both are set then remove one of them
        //in order to avoid duplicitly displayed buttons
        if (isset($actions[ITS4YouWarehouses_ItemStatus_ActionBlock::STATUS_AVAILABLE]) && isset($actions[ITS4YouWarehouses_ItemStatus_ActionBlock::STATUS_AVAILABLE_PART])) {
            unset($actions[ITS4YouWarehouses_ItemStatus_ActionBlock::STATUS_AVAILABLE_PART]);
        }
        return array(
            "items" => $items,
            "actions" => $actions,
            "overview" => array(
                "title" => vtranslate("LBL_DISPLAY_OVERVIEW_TITLE", "ITS4YouWHDeliveryNotes"),
                "label" => vtranslate("LBL_DISPLAY_OVERVIEW", "ITS4YouWHDeliveryNotes")
            ),
            "template" => "ActionBlockBase.tpl"
        );
    }

    /**
     * @param number $number
     * @return string
     */
    public static function formatNumberForDetailView($number)
    {
        return Vtiger_Currency_UIType::transformDisplayValue($number, null, true);
    }

    /**
     * Return sorted array of all distinct statuses of all items
     * @return array Sorted array of statuses
     */
    protected function getAllItemsAllStatusesDistinct()
    {
        /* @var $itemStatus ITS4YouWarehouses_ItemStatus_ActionBlock */

        $statuses = array();
        foreach ($this->items as $item) {
            $itemStatuses = $item->getStatuses();
            foreach ($itemStatuses as $itemStatus) {
                if ($itemStatus->getDisplayPrio() < 0) {
                    continue;
                }

                if (!isset($statuses[$itemStatus->getId()])) {
                    $statuses[$itemStatus->getId()] = $itemStatus;
                }
            }
        }
        //sort statuses
        uasort($statuses, array("ITS4YouWarehouses_ItemStatus_ActionBlock", "compare"));

        return $statuses;
    }

    /**
     * @param $_newItems
     * @return array
     */
    protected function sortSideWidgetStatuses($_newItems)
    {
        $sortKey = array(
            ITS4YouWarehouses_ItemStatus_ActionBlock::STATUS_NOTAVAILABLE,
            ITS4YouWarehouses_ItemStatus_ActionBlock::STATUS_AVAILABLE_PART,
            ITS4YouWarehouses_ItemStatus_ActionBlock::STATUS_AWAITING,
            ITS4YouWarehouses_ItemStatus_ActionBlock::STATUS_RECEIPTED,
            ITS4YouWarehouses_ItemStatus_ActionBlock::STATUS_AVAILABLE,
            ITS4YouWarehouses_ItemStatus_ActionBlock::STATUS_ISSUED,
            ITS4YouWarehouses_ItemStatus_ActionBlock::STATUS_DELIVERED,
            ITS4YouWarehouses_ItemStatus_ActionBlock::STATUS_TOBEINVOICED,
            ITS4YouWarehouses_ItemStatus_ActionBlock::STATUS_INVOICED
        );

        $sortedNewItems = array();
        $unsortedNewItems = array();
        foreach ($sortKey as $statusId) {
            if (isset($_newItems[$statusId])) {
                $sortedNewItems[$statusId] = $_newItems[$statusId];
            }
        }

        foreach ($_newItems as $statusId => $statusData) {
            if (!isset($sortedNewItems[$statusId])) {
                $unsortedNewItems[$statusId] = $statusData;
            }
        }

        return array_merge($sortedNewItems, $unsortedNewItems);
    }

    /**
     * @return array
     */
    public function getLineItems()
    {
        if (empty($this->lineItems)) {
            $this->lineItems = ITS4YouWarehouses_InventoryLineItem_Helper::getParentLineItems($this->parentId);
        }

        return $this->lineItems;
    }

    /**
     * @throws Exception
     * @var array $_selectedProducts
     */
    public function getOverviewModel($_selectedProducts)
    {
        throw new Exception("MultiWarehouses4You :: Not implemented method in descendant class.");
    }

    /**
     * @param int $itemId
     * @return ITS4YouWarehouses_Item_ActionBlock
     */
    protected function getItem($itemId)
    {
        if (isset($this->items[$itemId])) {
            return $this->items[$itemId];
        } else {
            return null;
        }
    }

    /**
     * @return array
     */
    protected function getItems()
    {
        return $this->items;
    }

    /**
     * @param $itemId
     */
    protected function removeItem($itemId)
    {
        unset($this->items[$itemId]);
    }

    /**
     * Return sorted array of all distinct displayed statuses of all items
     * @return array Sorted array of statuses
     * @throws Exception
     */
    protected function getAllItemsDisplayStatusesDistinct()
    {
        /* @var $item ITS4YouWarehouses_Item_ActionBlock */

        $statuses = array();
        foreach ($this->items as $item) {
            $statuses[$item->getId()] = $item->getDisplayStatus();
        }
        //sort statuses
        uasort($statuses, array("ITS4YouWarehouses_ItemStatus_ActionBlock", "compare"));

        return $statuses;
    }

    /**
     * @var string
     */
    protected $isReturnable = 'false';

    /**
     * @param array $lineItems
     * @return array
     */
    public function getReturnProductsLineItems($lineItems = [])
    {
        $items = [];

        if (empty($lineItems)) {
            $lineItems = $this->lineItems;
        }

        foreach ($lineItems as $lineItemId => $lineItem) {
            /* @var $lineItem ITS4YouWarehouses_InventoryLineItem_Helper */

            if ($this->getItem($lineItem->getProductId()) && 'Services' === $this->getItem($lineItem->getProductId())->getSEType()) {
                continue;
            }

            $quantity = $lineItem->getQuantity();
            $quantityReturned = $lineItem->getQuantity() - $lineItem->getAdjustedQuantity();

            $items[$lineItemId]['productId'] = $lineItem->getProductId();
            $items[$lineItemId]['name'] = $lineItem->getName();
            $items[$lineItemId]['qty_to_return'] = $lineItem->getAdjustedQuantity();
            $items[$lineItemId]['qty_default'] = self::formatNumberForEditView(0);
            $items[$lineItemId]['qty_raw'] = $quantity;
            $items[$lineItemId]['qty'] = self::formatNumberForDetailView($quantity);
            $items[$lineItemId]['qty_returned_raw'] = $quantityReturned;
            $items[$lineItemId]['qty_returned'] = self::formatNumberForDetailView($quantityReturned);

            if ($lineItem->getAdjustedQuantity()) {
                $this->isReturnable = 'true';
            }
        }

        return $items;
    }
}